/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6GFX_MOD_GL_UTILS_H
#define LIQUIDWAR6GFX_MOD_GL_UTILS_H

#include "../../gfx.h"

#include <SDL/SDL.h>
#include <SDL/SDL_opengl.h>
#include <SDL/SDL_image.h>
#include <SDL/SDL_ttf.h>

#ifndef GL_CLAMP_TO_EDGE
#define GL_CLAMP_TO_EDGE GL_CLAMP
#endif

#define MOD_GL_UTILS_MENU_TEXTURE_W 512
#define MOD_GL_UTILS_MENU_TEXTURE_H 128

/*
 * Internal event buffer. One might wonder why we recode
 * some stuff which could be handled by SDL directly.
 * Answer: we need to do some basic processing before
 * sending the info to guile/scheme. For instance we
 * want key strokes to arrive with all the keycode/scancode/label
 * to be sent. We might also wish to send virtual key events
 * such as "mouse up" or "joy up".
 */
#define MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE 256

/*
 * Size of the buffer we use to keep track of which key is pressed
 * or not. The index in this array is a keysym, not an ascii code.
 */
#define MOD_GL_UTILS_KEYPRESS_STATE_SIZE 512

/*
 * We keep 64 menu bitmaps in memory, to avoid redrawing them
 * at each display frame.
 */
#define MOD_GL_UTILS_MENUCACHE_ARRAY_SIZE 64

/*
 * We keep 32 textures in memory, this is to avoid transfering
 * bitmaps from SDL structures to the OpenGL world at each frame
 * display.
 */
#define MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE 32


/*
 * OpenGL should handle at least this texture size. Theorically
 * we should support any MAX_TEXTURE_SIZE value, but in practise,
 * most recent drivers support 512x512, and the "liquidwarish" 
 * answer to this problem is -> use another backend.
 */
#define MOD_GL_UTILS_REQUIRED_TEXTURE_SIZE 512

/*
 * How many different colors for armies.
 */
#define MOD_GL_SHADES_FOR_FIGHTERS_SCALE 64

#define MOD_GL_NB_JOYSTICKS 4
#define MOD_GL_NB_JOYSTICK_BUTTONS 6

typedef struct mod_gl_utils_path_s
{
  char *data_dir;
}
mod_gl_utils_path_t;

typedef struct mod_gl_utils_mem_counter_s
{
  int new_counter;
  int delete_counter;
}
mod_gl_utils_mem_counter_t;

typedef struct mod_gl_utils_caps_s
{
  GLint max_texture_size;
}
mod_gl_utils_caps_t;

typedef struct mod_gl_utils_screen_s
{
  int width;
  int height;
  int bpp;
  int fullscreen;
}
mod_gl_utils_screen_t;

typedef enum mod_gl_utils_render_mode_e
{
  MOD_GL_UTILS_RENDER_NONE = 0,
  MOD_GL_UTILS_RENDER_2D = 1,
  MOD_GL_UTILS_RENDER_2D_BLEND = 2,
  MOD_GL_UTILS_RENDER_3D_MAP = 3,
  MOD_GL_UTILS_RENDER_3D_MENU = 4,
  MOD_GL_UTILS_RENDER_3D_PICK = 5,
  MOD_GL_UTILS_RENDER_3D_WARP = 6
}
mod_gl_utils_render_mode_t;

typedef struct mod_gl_utils_rect_array_s
{
  int n_w;
  int n_h;
  int *x0;
  int *y0;
  int *w;
  int *h;
}
mod_gl_utils_rect_array_t;

typedef struct mod_gl_utils_surface_array_s
{
  mod_gl_utils_rect_array_t layout;
  SDL_Surface **surfaces;
}
mod_gl_utils_surface_array_t;

typedef struct mod_gl_utils_texture_array_s
{
  mod_gl_utils_rect_array_t layout;
  GLuint *textures;
}
mod_gl_utils_texture_array_t;

typedef struct mod_gl_utils_keyboard_state_s
{
  lw6gui_keypress_t keypress_buffer[MOD_GL_UTILS_KEYPRESS_BUFFER_SIZE];
  int keypress_current;
  int keypress_queue;
  int keypress_state[MOD_GL_UTILS_KEYPRESS_STATE_SIZE];
}
mod_gl_utils_keyboard_state_t;

typedef struct mod_gl_utils_button_state_s
{
  int pressed;
  int queue;
  int last_press;
  int last_repeat;
}
mod_gl_utils_button_state_t;

typedef struct mod_gl_utils_mouse_state_s
{
  int x;
  int y;
  int moved;
  mod_gl_utils_button_state_t button_left;
  mod_gl_utils_button_state_t button_right;
  mod_gl_utils_button_state_t wheel_up;
  mod_gl_utils_button_state_t wheel_down;
}
mod_gl_utils_mouse_state_t;

typedef struct mod_gl_utils_joystick_state_s
{
  mod_gl_utils_button_state_t up;
  mod_gl_utils_button_state_t down;
  mod_gl_utils_button_state_t left;
  mod_gl_utils_button_state_t right;
  mod_gl_utils_button_state_t buttons[MOD_GL_NB_JOYSTICK_BUTTONS];
  SDL_Joystick *sdl_joystick;
}
mod_gl_utils_joystick_state_t;

typedef struct mod_gl_utils_joysticks_state_s
{
  int supported;
  int driver_nb_joysticks;
  mod_gl_utils_joystick_state_t joystick_state[MOD_GL_NB_JOYSTICKS];
}
mod_gl_utils_joysticks_state_t;

typedef struct mod_gl_utils_quit_state_s
{
  int quit;
}
mod_gl_utils_quit_state_t;

typedef struct mod_gl_utils_texture_data_s
{
  lw6sys_list_t *to_delete;
}
mod_gl_utils_texture_data_t;

typedef struct mod_gl_utils_font_data_s
{
  TTF_Font *menu;
  TTF_Font *hud;
  TTF_Font *cursor;
  TTF_Font *system;
}
mod_gl_utils_font_data_t;

typedef struct mod_gl_utils_menucache_item_s
{
  char *key;
  SDL_Surface *surface;
}
mod_gl_utils_menucache_item_t;

typedef struct mod_gl_utils_menucache_array_s
{
  int last_stored_item;
    mod_gl_utils_menucache_item_t
    item_array[MOD_GL_UTILS_MENUCACHE_ARRAY_SIZE];
}
mod_gl_utils_menucache_array_t;

typedef struct mod_gl_utils_texturecache_item_s
{
  int used;
  int surface_w;
  int surface_h;
  char *str_key;
  int int_key;
  GLuint texture;
}
mod_gl_utils_texturecache_item_t;

typedef struct mod_gl_utils_texturecache_array_s
{
  mod_gl_utils_texturecache_item_t
    item_array[MOD_GL_UTILS_TEXTURECACHE_ARRAY_SIZE];
}
mod_gl_utils_texturecache_array_t;

typedef struct mod_gl_utils_const_data_s
{
  /*
   * Set by the hud module
   */
  float viewport_x;
  float viewport_y;
  float viewport_w;
  float viewport_h;
  /*
   * Set by the utils module itself
   */
  float mode_sleep;
  float quit_sleep;
  float persp_fovy;
  float persp_znear;
  float persp_zfar;
  int menu_font_size;
  int hud_font_size;
  int cursor_font_size;
  int system_font_size;
  int system_font_dh;
  int system_font_dw;
  float system_font_hcoef;
  float gradient_opacity;
  float zones_opacity;
  float mainlight_r;
  float mainlight_g;
  float mainlight_b;
  float mainlight_a;
  float mainlight_x;
  float mainlight_y;
  float mainlight_z;
  int joystick_limit_up;
  int joystick_limit_down;
  int joystick_limit_left;
  int joystick_limit_right;
}
mod_gl_utils_const_data_t;

typedef struct mod_gl_utils_texture_1x1_s
{
  /*
   * Set by the background module
   */
  GLuint menu_fg;
  GLuint menu_bg;
  GLuint menu_fg_disabled;
  GLuint menu_bg_disabled;
  GLuint menu_fg_selected;
  GLuint menu_bg_selected;
}
mod_gl_utils_texture_1x1_t;

typedef struct mod_gl_utils_team_color_map_s
{
  lw6sys_color_f_t team_colors_f[LW6MAP_MAX_NB_TEAMS];
  Uint32 team_colors[LW6MAP_MAX_NB_TEAMS][MOD_GL_SHADES_FOR_FIGHTERS_SCALE +
					  1];
}
mod_gl_utils_team_color_map_t;

typedef struct mod_gl_utils_timer_data_s
{
  int ticks_offset;
}
mod_gl_utils_timer_data_t;

typedef struct mod_gl_utils_last_action_s
{
  int game_texture_array_update_id;
  int game_texture_array_update_rounds;
}
mod_gl_utils_last_action_t;

typedef struct mod_gl_utils_repeat_settings_s
{
  int delay;
  int interval;
}
mod_gl_utils_repeat_settings_t;

typedef struct mod_gl_utils_context_s
{
  void (*resize_callback) (int width, int height, int fullscreen);
  mod_gl_utils_path_t path;
  mod_gl_utils_mem_counter_t surface_counter;
  mod_gl_utils_mem_counter_t texture_counter;
  mod_gl_utils_caps_t caps;
  mod_gl_utils_screen_t screen;
  mod_gl_utils_render_mode_t render_mode;
  mod_gl_utils_texture_data_t texture_data;
  mod_gl_utils_font_data_t font_data;
  mod_gl_utils_keyboard_state_t keyboard_state;
  mod_gl_utils_mouse_state_t mouse_state;
  mod_gl_utils_joysticks_state_t joysticks_state;
  mod_gl_utils_quit_state_t quit_state;
  mod_gl_utils_const_data_t const_data;
  mod_gl_utils_texture_1x1_t textures_1x1;
  mod_gl_utils_menucache_array_t menucache_array;
  mod_gl_utils_texturecache_array_t texturecache_array;
  mod_gl_utils_team_color_map_t team_color_map;
  mod_gl_utils_timer_data_t timer_data;
  mod_gl_utils_last_action_t last_action;
  mod_gl_utils_repeat_settings_t repeat_settings;
}
mod_gl_utils_context_t;

/*
 * In color.c
 */
extern SDL_Color mod_gl_utils_color_8_to_sdl (lw6sys_color_8_t color_8);
extern SDL_Color mod_gl_utils_color_f_to_sdl (lw6sys_color_f_t * color_f);
extern void mod_gl_utils_update_team_color_map (mod_gl_utils_team_color_map_t
						* team_color_map,
						lw6map_style_t * map_style);
inline static Uint32
mod_gl_utils_get_shaded_color_for_fighter (mod_gl_utils_context_t *
					   utils_context, int team_id,
					   int health)
{
  return utils_context->team_color_map.team_colors[team_id][(health *
							     MOD_GL_SHADES_FOR_FIGHTERS_SCALE)
							    /
							    LW6MAP_MAX_FIGHTER_HEALTH];
}

// gl-utils-colorize.c
extern SDL_Surface *mod_gl_utils_colorize_image (mod_gl_utils_context_t *
						 utils_context,
						 SDL_Surface * image,
						 lw6map_color_couple_t *
						 color);

/*
 * In data.c
 */
extern SDL_Surface *mod_gl_utils_load_image (mod_gl_utils_context_t *
					     utils_context, char *filename);
extern void mod_gl_utils_unload_image (mod_gl_utils_context_t * utils_context,
				       SDL_Surface * image);
extern GLuint mod_gl_utils_load_texture (mod_gl_utils_context_t *
					 utils_context,
					 SDL_Surface * surface);
extern void mod_gl_utils_unload_texture (mod_gl_utils_context_t *
					 utils_context, int texture);
extern int mod_gl_utils_load_fonts (mod_gl_utils_context_t * utils_context);
extern void mod_gl_utils_unload_fonts (mod_gl_utils_context_t *
				       utils_context);
extern int mod_gl_utils_load_data (mod_gl_utils_context_t * utils_context);
extern void mod_gl_utils_unload_data (mod_gl_utils_context_t * utils_context);

/*
 * In draw.c
 */
extern void mod_gl_utils_draw_rectfill (SDL_Surface * surface, int x1, int y1,
					int x2, int y2, Uint32 color);

/*
 * In event.c
 */
extern int mod_gl_utils_poll_key (mod_gl_utils_context_t * utils_context,
				  lw6gui_keypress_t * keypress);
extern int mod_gl_utils_is_key_esc (mod_gl_utils_context_t * utils_context,
				    int keysym);
extern int mod_gl_utils_is_key_enter (mod_gl_utils_context_t * utils_context,
				      int keysym);
extern int mod_gl_utils_is_key_up (mod_gl_utils_context_t * utils_context,
				   int keysym);
extern int mod_gl_utils_is_key_down (mod_gl_utils_context_t * utils_context,
				     int keysym);
extern int mod_gl_utils_is_key_left (mod_gl_utils_context_t * utils_context,
				     int keysym);
extern int mod_gl_utils_is_key_right (mod_gl_utils_context_t * utils_context,
				      int keysym);
extern int mod_gl_utils_poll_quit (mod_gl_utils_context_t * utils_context);
extern void mod_gl_utils_send_quit (mod_gl_utils_context_t * utils_context);
extern int mod_gl_utils_get_key_state (mod_gl_utils_context_t * utils_context,
				       int keysym);
extern void mod_gl_utils_get_default_keys_state (mod_gl_utils_context_t *
						 utils_context, int *up,
						 int *down, int *left,
						 int *right, int *esc,
						 int *enter);
extern int mod_gl_utils_poll_mouse_move (mod_gl_utils_context_t *
					 utils_context, int *x, int *y);
extern int mod_gl_utils_poll_mouse_button_left (mod_gl_utils_context_t *
						utils_context);
extern int mod_gl_utils_poll_mouse_button_right (mod_gl_utils_context_t *
						 utils_context);
extern int mod_gl_utils_poll_mouse_wheel_up (mod_gl_utils_context_t *
					     utils_context);
extern int mod_gl_utils_poll_mouse_wheel_down (mod_gl_utils_context_t *
					       utils_context);
extern void mod_gl_utils_get_mouse_state (mod_gl_utils_context_t *
					  utils_context, int *x, int *y,
					  int *button_left,
					  int *button_right);
extern int mod_gl_utils_warp_mouse (mod_gl_utils_context_t *
				    utils_context, int x, int y);
extern void mod_gl_utils_show_mouse (mod_gl_utils_context_t * utils_context,
				     int state);
extern int mod_gl_utils_poll_joystick_up (mod_gl_utils_context_t *
					  utils_context, int i);
extern int mod_gl_utils_poll_joystick_down (mod_gl_utils_context_t *
					    utils_context, int i);
extern int mod_gl_utils_poll_joystick_left (mod_gl_utils_context_t *
					    utils_context, int i);
extern int mod_gl_utils_poll_joystick_right (mod_gl_utils_context_t *
					     utils_context, int i);
extern int mod_gl_utils_poll_joystick_button (mod_gl_utils_context_t *
					      utils_context, int i, int b);
extern void mod_gl_utils_get_joystick_state (mod_gl_utils_context_t *
					     utils_context, int *x, int *y,
					     int *button_a, int *button_b,
					     int i, int b1, int b2);
extern int mod_gl_utils_set_key_repeat (mod_gl_utils_context_t *
					utils_context, int key_repeat_delay,
					int key_repeat_interval);

/*
 * In font.c
 */
extern int mod_gl_utils_get_system_text_width (mod_gl_utils_context_t *
					       utils_context, char *text);
extern int mod_gl_utils_get_system_text_height (mod_gl_utils_context_t *
						utils_context, char *text);
extern void mod_gl_utils_draw_system_text (mod_gl_utils_context_t *
					   utils_context,
					   lw6gui_look_t * look, char *text,
					   int x, int y);
extern void mod_gl_utils_draw_system_text_top_left (mod_gl_utils_context_t *
						    utils_context,
						    lw6gui_look_t * look,
						    char **text_list);
extern void mod_gl_utils_draw_system_text_top_right (mod_gl_utils_context_t *
						     utils_context,
						     lw6gui_look_t * look,
						     char **text_list);
extern void mod_gl_utils_draw_system_text_bottom_left (mod_gl_utils_context_t
						       * utils_context,
						       lw6gui_look_t * look,
						       char **text_list);
extern void mod_gl_utils_draw_system_text_bottom_right (mod_gl_utils_context_t
							* utils_context,
							lw6gui_look_t * look,
							char **text_list);


/*
 * In game.c
 */
extern void mod_gl_utils_update_game_texture (mod_gl_utils_context_t *
					      utils_context,
					      SDL_Surface * surface,
					      GLuint texture,
					      lw6ker_game_state_t *
					      game_state,
					      lw6gui_look_t * look,
					      int layer_id);
extern void mod_gl_utils_update_game_texture_array (mod_gl_utils_context_t *
						    utils_context,
						    mod_gl_utils_surface_array_t
						    * surface_array,
						    mod_gl_utils_texture_array_t
						    * texture_array,
						    lw6ker_game_state_t *
						    game_state,
						    lw6gui_look_t *
						    look, int layer_id);

/*
 * In gradient.c
 */
extern SDL_Surface
  * mod_gl_utils_create_gradient_surface (mod_gl_utils_context_t *
					  utils_context,
					  lw6ker_map_state_t * map_state,
					  int team_id);

/*
 * In info.c
 */
extern void mod_gl_utils_display_log (mod_gl_utils_context_t * utils_context,
				      lw6gui_look_t * look, char **log_list);
extern void mod_gl_utils_display_fps (mod_gl_utils_context_t * utils_context,
				      lw6gui_look_t * look, int fps);

/*
 * In joystick.c
 */
extern int mod_gl_utils_joystick_init (mod_gl_utils_joysticks_state_t *
				       joysticks_state);
extern void mod_gl_utils_joystick_quit (mod_gl_utils_joysticks_state_t *
					joysticks_state);

/*
 * In light.c
 */
extern void mod_gl_utils_main_light (mod_gl_utils_context_t * utils_context);

/*
 * In mode.c
 */
extern int mod_gl_utils_set_video_mode (mod_gl_utils_context_t *
					utils_context, int width, int height,
					int fullscreen,
					void (*resize_callback) (int width,
								 int height,
								 int
								 fullscreen));
extern int mod_gl_utils_resize_video_mode (mod_gl_utils_context_t *
					   utils_context, int width,
					   int height);
extern int mod_gl_utils_get_video_mode (mod_gl_utils_context_t *
					utils_context, int *width,
					int *height, int *fullscreen);
extern int mod_gl_utils_sync_viewport (mod_gl_utils_context_t *
				       utils_context);
extern int mod_gl_utils_sync_mode (mod_gl_utils_context_t * utils_context,
				   int force);

/*
 * In menucache.c
 */
extern int mod_gl_utils_store_button_in_menucache (mod_gl_utils_context_t *
						   utils_context,
						   lw6gui_look_t * look,
						   lw6gui_menuitem_t *
						   menuitem,
						   SDL_Surface * surface);
extern SDL_Surface
  * mod_gl_utils_get_button_from_menucache (mod_gl_utils_context_t * context,
					    lw6gui_look_t * look,
					    lw6gui_menuitem_t * menuitem);
extern void mod_gl_utils_clear_menucache (mod_gl_utils_context_t *
					  utils_context);

/*
 * In path.c
 */
extern int mod_gl_utils_path_init (mod_gl_utils_context_t *
				   utils_context, int argc, char *argv[]);
extern void mod_gl_utils_path_quit (mod_gl_utils_context_t * utils_context);

/*
 * In pixel.c
 */
extern Uint32 mod_gl_utils_getpixel (SDL_Surface * surface, int x, int y);
extern void mod_gl_utils_putpixel (SDL_Surface * surface, int x, int y,
				   Uint32 pixel);

/*
 * In rectarray.c
 */
extern int mod_gl_utils_rect_array_init (mod_gl_utils_context_t *
					 utils_context,
					 mod_gl_utils_rect_array_t *
					 rect_array, int w, int h,
					 int tile_size);
extern void mod_gl_utils_rect_array_clear (mod_gl_utils_context_t *
					   utils_context,
					   mod_gl_utils_rect_array_t *
					   rect_array);

/*
 * In render.c
 */
extern void mod_gl_utils_prepare_buffer (mod_gl_utils_context_t *
					 utils_context, lw6gui_look_t * look);
extern void mod_gl_utils_swap_buffers (mod_gl_utils_context_t *
				       utils_context);
extern void mod_gl_utils_set_render_mode_2d (mod_gl_utils_context_t *
					     utils_context);
extern void mod_gl_utils_set_render_mode_2d_blend (mod_gl_utils_context_t *
						   utils_context);
extern void mod_gl_utils_set_render_mode_3d_map (mod_gl_utils_context_t *
						 utils_context);
extern void mod_gl_utils_set_render_mode_3d_menu (mod_gl_utils_context_t *
						  utils_context);
extern void mod_gl_utils_set_render_mode_3d_pick (mod_gl_utils_context_t *
						  utils_context, int x,
						  int y);
extern void mod_gl_utils_set_render_mode_3d_warp (mod_gl_utils_context_t *
						  utils_context);

/*
 * In resolution.c
 */
extern void mod_gl_utils_get_low_resolution (mod_gl_utils_context_t *
					     utils_context, int *width,
					     int *height);
extern void mod_gl_utils_get_standard_resolution (mod_gl_utils_context_t *
						  utils_context, int *width,
						  int *height);
extern void mod_gl_utils_get_high_resolution (mod_gl_utils_context_t *
					      utils_context, int *width,
					      int *height);
extern void mod_gl_utils_find_closest_resolution (mod_gl_utils_context_t *
						  utils_context,
						  int *closest_width,
						  int *closest_height,
						  int wished_width,
						  int wished_height);

/*
 * In setup.c
 */
extern mod_gl_utils_context_t *mod_gl_utils_init ();
extern int mod_gl_utils_set_video_mode (mod_gl_utils_context_t *
					utils_context, int width, int height,
					int fullscreen,
					void (*resize_callback) (int width,
								 int height,
								 int
								 fullscreen));
extern int mod_gl_utils_resize_video_mode (mod_gl_utils_context_t *
					   utils_context, int width,
					   int height);
extern void mod_gl_utils_quit (mod_gl_utils_context_t * utils_context);

/*
 * In surface.c
 */
extern SDL_Surface *mod_gl_utils_create_surface (mod_gl_utils_context_t *
						 utils_context, int w, int h);
extern void mod_gl_utils_delete_surface (mod_gl_utils_context_t *
					 utils_context,
					 SDL_Surface * surface);
extern void mod_gl_utils_clear_surface_with_color (SDL_Surface * surface,
						   SDL_Color color);
extern void mod_gl_utils_clear_surface (SDL_Surface * surface);
extern SDL_Surface *mod_gl_utils_map2surface_xywh (mod_gl_utils_context_t *
						   utils_context,
						   lw6map_level_t * level,
						   int x, int y, int w,
						   int h);
extern SDL_Surface *mod_gl_utils_map2surface_wh (mod_gl_utils_context_t *
						 utils_context,
						 lw6map_level_t * level,
						 int w, int h);
extern SDL_Surface *mod_gl_utils_map2surface (mod_gl_utils_context_t *
					      utils_context,
					      lw6map_level_t * level);

/*
 * In surfacearray.c
 */
extern int mod_gl_utils_surface_array_init (mod_gl_utils_context_t *
					    utils_context,
					    mod_gl_utils_surface_array_t *
					    surface_array, int w, int h,
					    int tile_size);
extern int
mod_gl_utils_surface_array_init_from_surface (mod_gl_utils_context_t *
					      utils_context,
					      mod_gl_utils_surface_array_t *
					      surface_array,
					      SDL_Surface * surface,
					      int tile_size);
extern int mod_gl_utils_surface_array_init_from_map (mod_gl_utils_context_t *
						     utils_context,
						     mod_gl_utils_surface_array_t
						     * surface_array,
						     lw6map_level_t * level,
						     int tile_size);
extern void mod_gl_utils_surface_array_clear (mod_gl_utils_context_t *
					      utils_context,
					      mod_gl_utils_surface_array_t *
					      surface_array);
static inline void
mod_gl_utils_set_surface_array_surface (mod_gl_utils_surface_array_t *
					surface_array, int n_x, int n_y,
					SDL_Surface * surface)
{
  surface_array->surfaces[n_y * surface_array->layout.n_w + n_x] = surface;
}

static inline SDL_Surface *
mod_gl_utils_get_surface_array_surface (mod_gl_utils_surface_array_t *
					surface_array, int n_x, int n_y)
{
  return (surface_array->surfaces[n_y * surface_array->layout.n_w + n_x]);
}

/*
 * In texture.c
 */
extern int mod_gl_utils_power_of_two_le (int size);
extern int mod_gl_utils_power_of_two_ge (int size);
extern float mod_gl_utils_texture_scale (int size);
extern GLuint mod_gl_utils_surface2texture_xywh (mod_gl_utils_context_t *
						 utils_context,
						 SDL_Surface * surface, int x,
						 int y, int w, int h);
extern GLuint mod_gl_utils_surface2texture_wh (mod_gl_utils_context_t *
					       utils_context,
					       SDL_Surface * surface, int w,
					       int h);
extern GLuint mod_gl_utils_surface2texture (mod_gl_utils_context_t *
					    utils_context,
					    SDL_Surface * surface);
extern GLuint mod_gl_utils_map2texture_xywh (mod_gl_utils_context_t *
					     utils_context,
					     lw6map_level_t * level, int x,
					     int y, int w, int h);
extern GLuint mod_gl_utils_map2texture_wh (mod_gl_utils_context_t *
					   utils_context,
					   lw6map_level_t * level, int w,
					   int h);
extern GLuint mod_gl_utils_map2texture (mod_gl_utils_context_t *
					utils_context,
					lw6map_level_t * level);
extern void mod_gl_utils_delete_texture (mod_gl_utils_context_t *
					 utils_context, GLuint texture);
extern void mod_gl_utils_schedule_delete_texture (mod_gl_utils_context_t *
						  utils_context,
						  GLuint texture);
extern void mod_gl_utils_delete_scheduled_textures (mod_gl_utils_context_t *
						    utils_context);
extern void mod_gl_utils_texture_update (mod_gl_utils_context_t *
					 utils_context, GLuint texture,
					 SDL_Surface * surface);

/*
 * In texture1x1.c
 */
extern GLuint mod_gl_utils_texture_1x1_color2texture (mod_gl_utils_context_t *
						      utils_context,
						      lw6sys_color_8_t
						      color_8);
extern int mod_gl_utils_texture_1x1_update (mod_gl_utils_context_t *
					    utils_context,
					    lw6gui_look_t * look);
extern void mod_gl_utils_texture_1x1_clear (mod_gl_utils_context_t *
					    utils_context);

/*
 * In texturearray.c
 */
extern int mod_gl_utils_texture_array_init (mod_gl_utils_context_t *
					    utils_context,
					    mod_gl_utils_texture_array_t *
					    texture_array, int w, int h,
					    int tile_size);
extern void mod_gl_utils_texture_array_update (mod_gl_utils_context_t *
					       utils_context,
					       mod_gl_utils_texture_array_t *
					       texture_array,
					       mod_gl_utils_surface_array_t *
					       surface_array);
extern void mod_gl_utils_texture_array_clear (mod_gl_utils_context_t *
					      utils_context,
					      mod_gl_utils_texture_array_t *
					      texture_array);
extern void mod_gl_utils_schedule_delete_texture_array (mod_gl_utils_context_t
							* utils_context,
							mod_gl_utils_texture_array_t
							* texture_array);
static inline void
mod_gl_utils_set_texture_array_texture (mod_gl_utils_texture_array_t *
					texture_array, int n_x, int n_y,
					GLuint texture)
{
  texture_array->textures[n_y * texture_array->layout.n_w + n_x] = texture;
}

static inline GLuint
mod_gl_utils_get_texture_array_texture (mod_gl_utils_texture_array_t *
					texture_array, int n_x, int n_y)
{
  return (texture_array->textures[n_y * texture_array->layout.n_w + n_x]);
}

/*
 * In texturecache.c
 */
extern GLuint
mod_gl_utils_surface2texture_xywh_using_cache (mod_gl_utils_context_t *
					       context, SDL_Surface * surface,
					       int x, int y, int w, int h,
					       char *str_key, int int_key);
extern GLuint
mod_gl_utils_surface2texture_wh_using_cache (mod_gl_utils_context_t * context,
					     SDL_Surface * surface, int w,
					     int h, char *str_key,
					     int int_key);
extern GLuint mod_gl_utils_surface2texture_using_cache (mod_gl_utils_context_t
							* utils_context,
							SDL_Surface * surface,
							char *str_key,
							int int_key);
extern GLuint
mod_gl_utils_map2texture_xywh_using_cache (mod_gl_utils_context_t * context,
					   lw6map_level_t * level, int x,
					   int y, int w, int h, char *str_key,
					   int int_key);
extern GLuint mod_gl_utils_map2texture_wh_using_cache (mod_gl_utils_context_t
						       * context,
						       lw6map_level_t * level,
						       int w, int h,
						       char *str_key,
						       int int_key);
extern GLuint mod_gl_utils_map2texture_using_cache (mod_gl_utils_context_t *
						    utils_context,
						    lw6map_level_t * level,
						    char *str_key,
						    int int_key);
extern void mod_gl_utils_optimize_texturecache (mod_gl_utils_context_t *
						utils_context);
extern void mod_gl_utils_clear_texturecache (mod_gl_utils_context_t *
					     utils_context);

/*
 * In timer.c
 */
extern void mod_gl_utils_timer_init (mod_gl_utils_context_t * utils_context,
				     int ticks);
extern void mod_gl_utils_delay (mod_gl_utils_context_t * utils_context,
				int ms);
extern int mod_gl_utils_get_ticks (mod_gl_utils_context_t * utils_context);

/*
 * In viewport.c
 */
extern void mod_gl_utils_viewport_calc_view_area (mod_gl_utils_context_t *
						  utils_context, int *view_x1,
						  int *view_y1, int *view_x2,
						  int *view_y2);
extern void mod_gl_utils_viewport_calc_map_area (mod_gl_utils_context_t *
						 utils_context, float *map_x1,
						 float *map_y1, float *map_x2,
						 float *map_y2, int shape_w,
						 int shape_h,
						 lw6gui_look_t * look);
extern void mod_gl_utils_viewport_calc_map_pos (mod_gl_utils_context_t *
						utils_context, float *map_x,
						float *map_y, int x, int y,
						int w, int h,
						lw6gui_look_t * look);

/*
 * In zones.c
 */
extern SDL_Surface *mod_gl_utils_create_zones_surface (mod_gl_utils_context_t
						       * utils_context,
						       lw6ker_map_struct_t *
						       map_struct);

#endif
