/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "map.h"

static int
find_first_free_point (lw6map_depth_t * depth, int *found_x, int *found_y)
{
  int ret = 0;
  int x, y;

  *found_x = 0;
  *found_y = 0;
  for (y = 1; y < depth->shape.h - 1 && !ret; ++y)
    {
      for (x = 1; x < depth->shape.w - 1 && !ret; ++x)
	{
	  if (lw6map_depth_get (depth, x, y) > 0)
	    {
	      ret = 1;
	      *found_x = x;
	      *found_y = y;
	    }
	}
    }

  return ret;
}

/*
 * Updates a depth point if needed, returns true if value was changed
 */
static int
update_if_needed (lw6map_depth_t * dst, lw6map_depth_t * src, int x, int y)
{
  int ret = 0;

  if (lw6map_depth_get (dst, x, y) != lw6map_depth_get (src, x, y))
    {
      lw6map_depth_set (dst, x, y, lw6map_depth_get (src, x, y));
      ret = 1;
    }

  return ret;
}

/*
 * Checks the maps for "holes", that is fight zones that wouldn't
 * be connected with the main area, and fills them with walls.
 */
int
lw6map_depth_check_and_fix_holes (lw6map_depth_t * depth,
				  lw6map_rules_t * rules)
{
  int ret = 0;
  lw6map_depth_t fixed_depth;
  int x, y, tx, ty;

  fixed_depth.shape.w = depth->shape.w;
  fixed_depth.shape.h = depth->shape.h;
  fixed_depth.data =
    (unsigned char *) LW6SYS_CALLOC (depth->shape.w * depth->shape.h *
				     sizeof (unsigned char));
  fixed_depth.max = depth->max;

  if (fixed_depth.data)
    {
      int found;

      /*
       * We first set one point, which will spread all over.
       */
      ret = find_first_free_point (depth, &x, &y)
	&& update_if_needed (&fixed_depth, depth, x, y);
      found = ret ? 1 : 0;

      while (found)
	{
	  found = 0;

	  for (y = 0; y < depth->shape.h; ++y)
	    {
	      for (x = 0; x < depth->shape.w; ++x)
		{
		  if (lw6map_depth_get (&fixed_depth, x, y) > 0)
		    {
		      tx = x + 1;
		      ty = y;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		      tx = x + 1;
		      ty = y + 1;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		      tx = x;
		      ty = y + 1;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		      tx = x - 1;
		      ty = y + 1;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		    }
		}
	    }

	  for (y = depth->shape.h - 1; y >= 0; --y)
	    {
	      for (x = depth->shape.w - 1; x >= 0; --x)
		{
		  if (lw6map_depth_get (&fixed_depth, x, y) > 0)
		    {
		      tx = x - 1;
		      ty = y;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		      tx = x - 1;
		      ty = y - 1;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		      tx = x;
		      ty = y - 1;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		      tx = x + 1;
		      ty = y - 1;
		      lw6map_coords_fix (rules, &depth->shape, &tx, &ty);
		      found += update_if_needed (&fixed_depth, depth, tx, ty);
		    }
		}
	    }
	}
      LW6SYS_FREE (depth->data);
      depth->data = fixed_depth.data;
    }

  return ret;
}

static int
rectangle (lw6map_depth_t * depth, int width, int height)
{
  int ret = 0;
  int x, y;

  depth->shape.w = width;
  depth->shape.h = height;
  depth->data =
    (unsigned char *) LW6SYS_CALLOC (depth->shape.w * depth->shape.h *
				     sizeof (unsigned char));

  if (depth->data)
    {
      for (y = 0; y < depth->shape.h; ++y)
	{
	  for (x = 0; x < depth->shape.w; ++x)
	    {
	      lw6map_depth_set (depth, x, y, 0);
	    }
	}

      for (y = 1; y < depth->shape.h - 1; ++y)
	{
	  for (x = 1; x < depth->shape.w - 1; ++x)
	    {
	      lw6map_depth_set (depth, x, y, 1);
	    }
	}
      ret = 1;
    }

  return ret;
}

void
lw6map_depth_defaults (lw6map_depth_t * depth, int w, int h)
{
  lw6map_depth_clear (depth);

  if (!rectangle (depth, w, h))
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to construct default map depth"));
      depth->shape.w = 0;
      depth->shape.h = 0;
    }
}

/*
 * Clears a depth map.
 */
void
lw6map_depth_clear (lw6map_depth_t * depth)
{
  if (depth->data)
    {
      LW6SYS_FREE (depth->data);
    }

  memset (depth, 0, sizeof (lw6map_depth_t));
}

void
lw6map_depth_fill (lw6map_depth_t * depth, unsigned char value)
{
  int x, y;

  for (y = 0; y < depth->shape.h; ++y)
    {
      for (x = 0; x < depth->shape.w; ++x)
	{
	  lw6map_depth_set (depth, x, y, value);
	}
    }
}

int
lw6map_depth_coord_from_texture (lw6map_level_t * level, int *depth_x,
				 int *depth_y, int texture_x, int texture_y)
{
  int ret = 0;

  if (level->texture.shape.w > 0 && level->texture.shape.h > 0)
    {
      (*depth_x) =
	(texture_x * level->depth.shape.w) / level->texture.shape.w;
      (*depth_y) =
	(texture_y * level->depth.shape.h) / level->texture.shape.h;
      (*depth_x) =
	lw6sys_max (0, lw6sys_min (level->depth.shape.w - 1, *depth_x));
      (*depth_y) =
	lw6sys_max (0, lw6sys_min (level->depth.shape.h - 1, *depth_y));
      ret = 1;
    }
  else
    {
      (*depth_x) = 0;
      (*depth_y) = 0;
    }

  return ret;
}

unsigned char
lw6map_depth_get_with_texture_coord (lw6map_level_t * level, int texture_x,
				     int texture_y)
{
  unsigned char ret = 0;
  int depth_x;
  int depth_y;

  if (lw6map_depth_coord_from_texture
      (level, &depth_x, &depth_y, texture_x, texture_y))
    {
      ret = lw6map_depth_get (&level->depth, depth_x, depth_y);
    }

  return ret;
}
