/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "map.h"

static int
push_metadata (lw6sys_hexa_serializer_t * hexa_serializer,
	       lw6map_metadata_t * metadata)
{
  int ret = 1;

  ret = ret
    && lw6sys_hexa_serializer_push_str (hexa_serializer, metadata->title);
  ret = ret
    && lw6sys_hexa_serializer_push_str (hexa_serializer, metadata->readme);

  return ret;
}

static int
push_depth (lw6sys_hexa_serializer_t * hexa_serializer,
	    lw6map_depth_t * depth)
{
  int ret = 1;
  int x, y;

  ret = ret && lw6sys_hexa_serializer_push_wh (hexa_serializer, depth->shape);
  for (y = 0; y < depth->shape.h && ret; ++y)
    {
      for (x = 0; x < depth->shape.w && ret; ++x)
	{
	  ret = ret
	    && lw6sys_hexa_serializer_push_int8 (hexa_serializer,
						 lw6map_depth_get (depth, x,
								   y));
	}
    }
  ret = ret
    && lw6sys_hexa_serializer_push_int32 (hexa_serializer, depth->max);

  return ret;
}

static int
push_texture (lw6sys_hexa_serializer_t * hexa_serializer,
	      lw6map_texture_t * texture)
{
  int ret = 1;
  int x, y;

  ret = ret
    && lw6sys_hexa_serializer_push_wh (hexa_serializer, texture->shape);
  for (y = 0; y < texture->shape.h && ret; ++y)
    {
      for (x = 0; x < texture->shape.w && ret; ++x)
	{
	  ret = ret
	    && lw6sys_hexa_serializer_push_color (hexa_serializer,
						  lw6map_texture_get (texture,
								      x, y));
	}
    }
  ret = ret && lw6sys_hexa_serializer_push_color (hexa_serializer,
						  texture->
						  guessed_color_base.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  texture->guessed_color_base.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  texture->
					  guessed_color_alternate.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  texture->
					  guessed_color_alternate.fg);

  return ret;
}

static int
push_param (lw6sys_hexa_serializer_t * hexa_serializer,
	    lw6map_param_t * param)
{
  int ret = 1;
  int i, value;
  char *key;

  for (i = 0; (key = LW6MAP_RULES_LIST[i]) != NULL && ret; ++i)
    {
      value = lw6map_rules_get_int (&(param->rules), key);
      ret = ret && lw6sys_hexa_serializer_push_int32 (hexa_serializer, value);
    }

  ret = ret
    && lw6sys_hexa_serializer_push_int32 (hexa_serializer,
					  param->style.keep_ratio);
  ret = ret
    && lw6sys_hexa_serializer_push_float (hexa_serializer, param->style.zoom);
  ret = ret
    && lw6sys_hexa_serializer_push_str (hexa_serializer,
					param->style.background_style);
  ret = ret
    && lw6sys_hexa_serializer_push_str (hexa_serializer,
					param->style.hud_style);
  ret = ret
    && lw6sys_hexa_serializer_push_str (hexa_serializer,
					param->style.menu_style);
  ret = ret
    && lw6sys_hexa_serializer_push_str (hexa_serializer,
					param->style.view_style);
  ret = ret
    && lw6sys_hexa_serializer_push_float (hexa_serializer,
					  param->style.animation_density);
  ret = ret
    && lw6sys_hexa_serializer_push_float (hexa_serializer,
					  param->style.animation_speed);
  ret = ret
    && lw6sys_hexa_serializer_push_int32 (hexa_serializer,
					  param->style.colorize);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.color_base.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.color_base.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.color_alternate.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.color_alternate.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.background_color_root.
					  bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.background_color_root.
					  fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.background_color_stuff.
					  bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.background_color_stuff.
					  fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.hud_color_frame.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.hud_color_frame.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.hud_color_text.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.hud_color_text.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.menu_color_default.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.menu_color_default.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.
					  menu_color_selected.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.
					  menu_color_selected.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.
					  menu_color_disabled.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.
					  menu_color_disabled.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.view_color_cursor.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.view_color_cursor.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.view_color_map.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.view_color_map.fg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.system_color.bg);
  ret = ret
    && lw6sys_hexa_serializer_push_color (hexa_serializer,
					  param->style.system_color.fg);
  for (i = 0; i < LW6MAP_NB_TEAM_COLORS && ret; ++i)
    {
      ret = ret
	&& lw6sys_hexa_serializer_push_color (hexa_serializer,
					      param->style.team_colors[i]);
    }

  return ret;
}

/**
 * lw6map_to_hexa
 *
 * @map: the map to convert
 *
 * Converts a map to something that is later readable by @lw6map_from_hexa
 * to reproduce the exact same map. Just a serializer.
 *
 * Return value: a newly allocated pointer, NULL if conversion failed.
 */
char *
lw6map_to_hexa (lw6map_level_t * level)
{
  char *ret = NULL;
  lw6sys_hexa_serializer_t *hexa_serializer = NULL;
  int ok = 1;

  hexa_serializer = lw6sys_hexa_serializer_new (NULL);
  if (hexa_serializer)
    {
      ok = ok && push_metadata (hexa_serializer, &(level->metadata));
      ok = ok && push_depth (hexa_serializer, &(level->depth));
      ok = ok && push_texture (hexa_serializer, &(level->texture));
      ok = ok && push_param (hexa_serializer, &(level->param));
      if (ok)
	{
	  ret = lw6sys_hexa_serializer_as_string (hexa_serializer);
	}
      lw6sys_hexa_serializer_free (hexa_serializer);
    }

  return ret;
}


static int
pop_metadata (lw6sys_hexa_serializer_t * hexa_serializer,
	      lw6map_metadata_t * metadata)
{
  int ret = 1;

  ret = ret
    && lw6sys_hexa_serializer_pop_str (hexa_serializer, &(metadata->title));
  ret = ret
    && lw6sys_hexa_serializer_pop_str (hexa_serializer, &(metadata->readme));

  return ret;
}

static int
pop_depth (lw6sys_hexa_serializer_t * hexa_serializer, lw6map_depth_t * depth)
{
  int ret = 1;
  int x, y;
  int8_t c = 0;

  ret = ret
    && lw6sys_hexa_serializer_pop_wh (hexa_serializer, &(depth->shape));
  if (depth->shape.w >= LW6MAP_MIN_SIDE
      && depth->shape.w <= LW6MAP_MAX_SIDE
      && depth->shape.h >= LW6MAP_MIN_SIDE
      && depth->shape.h <= LW6MAP_MAX_SIDE
      && depth->shape.w * depth->shape.h >= LW6MAP_MIN_SURFACE
      && depth->shape.w * depth->shape.h <= LW6MAP_MAX_SURFACE)
    {
      depth->data =
	(u_int8_t *) LW6SYS_CALLOC (depth->shape.w * depth->shape.h);
      if (depth->data)
	{
	  for (y = 0; y < depth->shape.h && ret; ++y)
	    {
	      for (x = 0; x < depth->shape.w && ret; ++x)
		{
		  ret = ret
		    && lw6sys_hexa_serializer_pop_int8 (hexa_serializer, &c);
		  lw6map_depth_set (depth, x, y, c);
		}
	    }
	}
      else
	{
	  ret = 0;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("map shape out of range (%dx%d)"),
		  depth->shape.w, depth->shape.h);
      ret = 0;
    }
  ret = ret
    && lw6sys_hexa_serializer_pop_int32 (hexa_serializer, &(depth->max));

  return ret;
}

static int
pop_texture (lw6sys_hexa_serializer_t * hexa_serializer,
	     lw6map_texture_t * texture)
{
  int ret = 1;
  int x, y;
  lw6sys_color_8_t color = LW6SYS_COLOR_8_BLACK;

  ret = ret
    && lw6sys_hexa_serializer_pop_wh (hexa_serializer, &(texture->shape));
  if (texture->shape.w >= LW6MAP_MIN_SIDE
      && texture->shape.w <= LW6MAP_MAX_SIDE
      && texture->shape.h >= LW6MAP_MIN_SIDE
      && texture->shape.h <= LW6MAP_MAX_SIDE
      && texture->shape.w * texture->shape.h >= LW6MAP_MIN_SURFACE
      && texture->shape.w * texture->shape.h <= LW6MAP_MAX_SURFACE)
    {
      texture->data =
	(lw6sys_color_8_t *) LW6SYS_CALLOC (texture->shape.w *
					    texture->shape.h *
					    sizeof (lw6sys_color_8_t));
      if (texture->data)
	{
	  for (y = 0; y < texture->shape.h && ret; ++y)
	    {
	      for (x = 0; x < texture->shape.w && ret; ++x)
		{
		  ret = ret
		    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
							 &color);
		  lw6map_texture_set (texture, x, y, color);
		}
	    }
	}
      else
	{
	  ret = 0;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("texture shape out of range (%dx%d)"),
		  texture->shape.w, texture->shape.h);
      ret = 0;
    }
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(texture->guessed_color_base.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(texture->guessed_color_base.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(texture->
					   guessed_color_alternate.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(texture->
					   guessed_color_alternate.fg));

  return ret;
}

static int
pop_param (lw6sys_hexa_serializer_t * hexa_serializer, lw6map_param_t * param)
{
  int ret = 1;
  int32_t value;
  int i;
  char *key;

  for (i = 0; (key = LW6MAP_RULES_LIST[i]) != NULL && ret; ++i)
    {
      ret = ret && lw6sys_hexa_serializer_pop_int32 (hexa_serializer, &value);
      if (ret)
	{
	  lw6map_rules_set_int (&(param->rules), key, value);
	}
    }

  ret = ret
    && lw6sys_hexa_serializer_pop_int32 (hexa_serializer,
					 &(param->style.keep_ratio));
  ret = ret
    && lw6sys_hexa_serializer_pop_float (hexa_serializer,
					 &(param->style.zoom));
  ret = ret
    && lw6sys_hexa_serializer_pop_str (hexa_serializer,
				       &(param->style.background_style));
  ret = ret
    && lw6sys_hexa_serializer_pop_str (hexa_serializer,
				       &(param->style.hud_style));
  ret = ret
    && lw6sys_hexa_serializer_pop_str (hexa_serializer,
				       &(param->style.menu_style));
  ret = ret
    && lw6sys_hexa_serializer_pop_str (hexa_serializer,
				       &(param->style.view_style));
  ret = ret
    && lw6sys_hexa_serializer_pop_float (hexa_serializer,
					 &(param->style.animation_density));
  ret = ret
    && lw6sys_hexa_serializer_pop_float (hexa_serializer,
					 &(param->style.animation_speed));
  ret = ret
    && lw6sys_hexa_serializer_pop_int32 (hexa_serializer,
					 &(param->style.colorize));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.color_base.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.color_base.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.color_alternate.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.color_alternate.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.background_color_root.
					   bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.background_color_root.
					   fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   background_color_stuff.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   background_color_stuff.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.hud_color_frame.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.hud_color_frame.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.hud_color_text.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.hud_color_text.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   menu_color_default.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   menu_color_default.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   menu_color_selected.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   menu_color_selected.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   menu_color_disabled.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   menu_color_disabled.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   view_color_cursor.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.
					   view_color_cursor.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.view_color_map.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.view_color_map.fg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.system_color.bg));
  ret = ret
    && lw6sys_hexa_serializer_pop_color (hexa_serializer,
					 &(param->style.system_color.fg));
  for (i = 0; i < LW6MAP_NB_TEAM_COLORS && ret; ++i)
    {
      ret = ret
	&& lw6sys_hexa_serializer_pop_color (hexa_serializer,
					     &(param->style.team_colors[i]));
    }

  return ret;
}

/**
 * lw6map_from_hexa
 *
 * @hexa: an hexadecimal ASCII string, created by @lw6map_to_hexa
 *
 * Constructs a map from an hexadecimal string generated
 * by @lw6map_to_hexa. Just an un-serializer.
 *
 * Return value: a new map, might be NULL if string isn't correct.
 */
lw6map_level_t *
lw6map_from_hexa (char *hexa)
{
  lw6map_level_t *level = NULL;
  lw6sys_hexa_serializer_t *hexa_serializer;
  int ok = 1;

  hexa_serializer = lw6sys_hexa_serializer_new (hexa);
  if (hexa_serializer)
    {
      lw6sys_hexa_serializer_rewind (hexa_serializer);
      level = lw6map_new ();
      if (level)
	{
	  ok = ok && pop_metadata (hexa_serializer, &(level->metadata));
	  ok = ok && pop_depth (hexa_serializer, &(level->depth));
	  ok = ok && pop_texture (hexa_serializer, &(level->texture));
	  ok = ok && pop_param (hexa_serializer, &(level->param));
	  if (!lw6sys_hexa_serializer_eof (hexa_serializer))
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _("expected EOF in serialized level"));
	      ok = 0;
	    }
	  if (!ok)
	    {
	      lw6map_free (level);
	      level = NULL;
	    }
	}
      lw6sys_hexa_serializer_free (hexa_serializer);
    }

  return level;
}
