/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "map.h"

#define TEST_MAP_WIDTH 72
#define TEST_MAP_HEIGHT 24
#define TEST_RULES_KEY LW6DEF_TOTAL_TIME
#define TEST_RULES_VALUE 3600
#define TEST_RULES_CHECKSUM 0x1b844db5
#define TEST_STYLE_KEY LW6DEF_ZOOM
#define TEST_STYLE_VALUE "1.0"
#define TEST_COORDS_NB 5
#define TEST_COORDS_W 100
#define TEST_COORDS_H 50
#define TEST_COORDS_X1 66
#define TEST_COORDS_Y1 33
#define TEST_COORDS_X2 130
#define TEST_COORDS_Y2 20
#define TEST_COORDS_X3 -30
#define TEST_COORDS_Y3 20
#define TEST_COORDS_X4 -10
#define TEST_COORDS_Y4 -5
#define TEST_COORDS_X5 1000
#define TEST_COORDS_Y5 51

/*
 * Testing basic new & free functions
 */
static int
test_new ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_level_t *level = NULL;
    char *repr = NULL;

    level = lw6map_new ();
    if (level)
      {
	repr = lw6map_repr (level);
	if (repr)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE, _("empty map \"%s\" constructed"),
			repr);
	    LW6SYS_FREE (repr);
	  }
	else
	  {
	    ret = 0;
	  }
	lw6map_free (level);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_color ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_color_couple_t color_couple;

    color_couple.bg = LW6SYS_COLOR_8_WHITE;
    color_couple.fg = LW6SYS_COLOR_8_BLACK;

    lw6map_color_invert (&color_couple);
    if (!lw6map_color_is_same (&color_couple, &color_couple))
      {
	lw6sys_log (LW6SYS_LOG_WARNING,
		    _
		    ("color_is_same pretending colors are different, while they should be equal"));
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_coords ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_rules_t rules;
    lw6sys_wh_t shape;
    int test_x[TEST_COORDS_NB] =
      { TEST_COORDS_X1, TEST_COORDS_X2, TEST_COORDS_X3, TEST_COORDS_X4,
      TEST_COORDS_X5
    };
    int test_y[TEST_COORDS_NB] =
      { TEST_COORDS_Y1, TEST_COORDS_Y2, TEST_COORDS_Y3, TEST_COORDS_Y4,
      TEST_COORDS_Y5
    };
    int i, x, y, px, py;

    lw6map_rules_defaults (&rules);
    shape.w = TEST_COORDS_W;
    shape.h = TEST_COORDS_H;

    for (py = 1; py >= -1; --py)
      {
	for (px = 1; px >= -1; --px)
	  {
	    rules.x_polarity = px;
	    rules.y_polarity = py;
	    lw6sys_log (LW6SYS_LOG_NOTICE, _("polarity set to %d,%d"), px,
			py);
	    for (i = 0; i < TEST_COORDS_NB; ++i)
	      {
		x = test_x[i];
		y = test_y[i];
		lw6sys_log (LW6SYS_LOG_NOTICE, _("coords before fix %d,%d"),
			    x, y);
		lw6map_coords_fix (&rules, &shape, &x, &y);
		lw6sys_log (LW6SYS_LOG_NOTICE, _("coords after fix %d,%d"), x,
			    y);
	      }
	  }
      }

    ret = 1;
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing defaults
 */
static int
test_defaults ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_level_t *level = NULL;
    char *repr = NULL;

    level = lw6map_defaults (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (level)
      {
	repr = lw6map_repr (level);
	if (repr)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE,
			_("default map \"%s\" constructed"), repr);
	    LW6SYS_FREE (repr);
	  }
	else
	  {
	    ret = 0;
	  }
	lw6map_free (level);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing dup
 */
static int
test_dup ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_level_t *level1 = NULL;
    lw6map_level_t *level2 = NULL;
    char *repr1 = NULL;
    char *repr2 = NULL;

    level1 = lw6map_defaults (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (level1)
      {
	repr1 = lw6map_repr (level1);
	if (repr1)
	  {
	    level2 = lw6map_dup (level1);
	    if (level2)
	      {
		repr2 = lw6map_repr (level2);
		if (repr2)
		  {
		    lw6sys_log (LW6SYS_LOG_NOTICE,
				_("map \"%s\" copied to \"%s\""), repr1,
				repr2);
		    LW6SYS_FREE (repr2);
		  }
		else
		  {
		    ret = 0;
		  }
		lw6map_free (level2);
	      }
	    else
	      {
		ret = 0;
	      }
	    LW6SYS_FREE (repr1);
	  }
	else
	  {
	    ret = 0;
	  }
	lw6map_free (level1);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

/*
 * Testing hexa
 */
static int
test_hexa ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_level_t *level = NULL;
    char *repr = NULL;
    char *hexa = NULL;

    level = lw6map_defaults (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (level)
      {
	repr = lw6map_repr (level);
	if (repr)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE, _("map \"%s\" constructed"), repr);
	    LW6SYS_FREE (repr);
	  }
	else
	  {
	    ret = 0;
	  }
	hexa = lw6map_to_hexa (level);
	lw6map_free (level);
	if (hexa)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE, _("hexa is \"%s\""), hexa);
	    level = lw6map_from_hexa (hexa);
	    if (level)
	      {
		repr = lw6map_repr (level);
		if (repr)
		  {
		    lw6sys_log (LW6SYS_LOG_NOTICE,
				_("map \"%s\" re-constructed from hexa"),
				repr);
		    LW6SYS_FREE (repr);
		  }
		else
		  {
		    ret = 0;
		  }
		lw6map_free (level);
	      }
	    else
	      {
		ret = 0;
	      }
	    LW6SYS_FREE (hexa);
	  }
	else
	  {
	    ret = 0;
	  }
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_rules ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_rules_t *rules;
    int value;
    unsigned int checksum = 0;

    rules = (lw6map_rules_t *) LW6SYS_CALLOC (sizeof (lw6map_rules_t));
    if (rules)
      {
	lw6map_rules_set_int (rules, TEST_RULES_KEY, TEST_RULES_VALUE);
	value = lw6map_rules_get_int (rules, TEST_RULES_KEY);
	lw6sys_log (LW6SYS_LOG_NOTICE, _("map rules for key \"%s\" is %d"),
		    TEST_RULES_KEY, value);
	lw6map_rules_clear (rules);

	lw6map_rules_defaults (rules);
	checksum = 0;
	lw6map_rules_update_checksum (rules, &checksum);

	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _
		    ("rules checksum is %08x and should be %08x"),
		    checksum, TEST_RULES_CHECKSUM);
	ret = ret && (checksum == TEST_RULES_CHECKSUM);
	LW6SYS_FREE (rules);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_style ()
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6map_style_t *style;
    char *value;

    style = (lw6map_style_t *) LW6SYS_CALLOC (sizeof (lw6map_style_t));
    if (style)
      {
	lw6map_style_set (style, TEST_STYLE_KEY, TEST_STYLE_VALUE);
	value = lw6map_style_get (style, TEST_STYLE_KEY);
	if (value)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE,
			_("map style for key \"%s\" is \"%s\""),
			TEST_STYLE_KEY, value);
	    LW6SYS_FREE (value);
	  }
	else
	  {
	    ret = 0;
	  }
	lw6map_style_clear (style);
	LW6SYS_FREE (style);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

int
lw6map_test ()
{
  int ret = 0;

  if (lw6sys_false ())
    {
      /*
       * Just to make sure most functions are stuffed in the binary
       */
      lw6sys_test ();
      lw6map_test ();
    }

  ret = test_new () && test_color () && test_coords () && test_defaults ()
    && test_dup () && test_hexa () && test_rules () && test_style ();

  return ret;
}
