/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gfx.h"

#define TEST_WIDTH 640
#define TEST_HEIGHT 480
#define TEST_FULLSCREEN 0
#define TEST_SLEEP 0.01
#define TEST_DURATION_SETUP 100
#define TEST_DURATION_BACKGROUND 2000
#define TEST_DURATION_MENU 3000
#define TEST_MENU_LABEL "My menu"
#define TEST_MENU_ESC "Esc"
#define TEST_MENU_ENABLE_ESC 0
#define TEST_MENU_ALLOW_SCROLL 0
#define TEST_MENU_COLOR 1
#define TEST_MENU_NB_MENUITEMS 5
#define TEST_MENU_SELECTED_MENUITEM 1
#define TEST_MENU_COLORED_MENUITEM 3
#define TEST_MENU_MAX_DISPLAYED_ITEMS 3
#define TEST_DURATION_EVENTS 10000
#define TEST_ARGC 1
#define TEST_ARGV0 "prog"

static int
test_resolution (lw6gfx_backend_t * backend)
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6gui_fullscreen_modes_t modes;

    if (lw6gfx_get_fullscreen_modes (backend, &modes))
      {
	lw6sys_log (LW6SYS_LOG_NOTICE, _("low resolution is %dx%d"),
		    modes.low.width, modes.low.height);
	lw6sys_log (LW6SYS_LOG_NOTICE, _("standard resolution is %dx%d"),
		    modes.standard.width, modes.standard.height);
	lw6sys_log (LW6SYS_LOG_NOTICE, _("high resolution is %dx%d"),
		    modes.high.width, modes.high.height);
      }
    else
      {
	lw6sys_log (LW6SYS_LOG_WARNING, _("unable to get fullscreen modes"));
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static void
resize_callback (lw6gui_video_mode_t * video_mode)
{
  lw6sys_log (LW6SYS_LOG_NOTICE,
	      _("resize callback called %dx%d fullscreen=%d"),
	      video_mode->width, video_mode->height, video_mode->fullscreen);
}

static int
test_init (lw6gfx_backend_t * backend)
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6gui_video_mode_t video_mode;
    int64_t ticks;
    char *repr;

    video_mode.width = TEST_WIDTH;
    video_mode.height = TEST_HEIGHT;
    video_mode.fullscreen = TEST_FULLSCREEN;

    ret = ret && lw6gfx_init (backend, &video_mode, resize_callback);
    if (ret)
      {
	repr = lw6gfx_repr (backend);
	if (repr)
	  {
	    lw6sys_log (LW6SYS_LOG_NOTICE, _("display repr is \"%s\""), repr);
	    LW6SYS_FREE (repr);
	  }
	ticks = lw6sys_uptime ();
	while (lw6sys_uptime () < ticks + TEST_DURATION_SETUP)
	  {
	    lw6sys_sleep (TEST_SLEEP);
	  }
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_background (lw6gfx_backend_t * backend)
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    int64_t ticks;
    lw6gui_look_t *look = NULL;

    look = lw6gui_look_new (NULL);
    if (look)
      {
	ticks = lw6sys_uptime ();
	while (lw6sys_uptime () < ticks + TEST_DURATION_BACKGROUND)
	  {
	    if (!lw6gfx_display (backend,
				 LW6GUI_DISPLAY_BACKGROUND,
				 look,
				 NULL,
				 NULL, NULL, 0, NULL, 0.0f, 0, 0, NULL, 0, 0))
	      {
		lw6sys_log (LW6SYS_LOG_WARNING, _("display error"));
		ret = 0;
	      }
	    lw6sys_sleep (TEST_SLEEP);
	  }
	lw6gui_look_free (look);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_menu (lw6gfx_backend_t * backend)
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6gui_menuitem_t *menuitem = NULL;
    lw6gui_menu_t *menu = NULL;
    int i;
    char *menuitem_label;
    int64_t ticks;
    float progress;

    lw6gui_look_t *look = NULL;

    look = lw6gui_look_new (NULL);
    if (look)
      {
	menu =
	  lw6gui_menu_new (TEST_MENU_LABEL, TEST_MENU_ESC,
			   TEST_MENU_ENABLE_ESC);
	if (menu)
	  {
	    for (i = 0; i < TEST_MENU_NB_MENUITEMS; ++i)
	      {
		menuitem_label =
		  lw6sys_new_sprintf (_("Test menuitem %d"), i);
		if (menuitem_label)
		  {
		    menuitem =
		      lw6gui_menuitem_new (menuitem_label, TEST_MENU_COLOR, 1,
					   0,
					   i == TEST_MENU_COLORED_MENUITEM);
		    if (menuitem)
		      {
			lw6gui_menu_append (menu, menuitem, lw6sys_uptime ());
		      }
		    LW6SYS_FREE (menuitem_label);
		  }
	      }
	    lw6gui_menu_select (menu, TEST_MENU_SELECTED_MENUITEM,
				TEST_MENU_ALLOW_SCROLL, 0);
	    ticks = lw6sys_uptime ();
	    while (lw6sys_uptime () < ticks + TEST_DURATION_MENU)
	      {
		progress = ((float)
			    (lw6sys_uptime () -
			     ticks)) / ((float) TEST_DURATION_MENU);
		if (!lw6gfx_display (backend,
				     LW6GUI_DISPLAY_BACKGROUND |
				     LW6GUI_DISPLAY_MENU |
				     LW6GUI_DISPLAY_PROGRESS, look, NULL,
				     NULL, NULL, 0, menu, progress, 0, 0,
				     NULL, 0, 0))
		  {
		    lw6sys_log (LW6SYS_LOG_WARNING, _("display error"));
		    ret = 0;
		  }
		lw6sys_sleep (TEST_SLEEP);
	      }
	    lw6gui_menu_free (menu);	// should free the menuitem
	  }
	lw6gui_look_free (look);
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

static int
test_events (lw6gfx_backend_t * backend)
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  {
    lw6gui_look_t *look = NULL;
    lw6gui_input_t *input = NULL;
    lw6gui_keypress_t *keypress = NULL;
    char *repr = NULL;
    int64_t ticks = 0;

    look = lw6gui_look_new (NULL);
    if (look)
      {
	ticks = lw6sys_uptime ();
	lw6sys_log (LW6SYS_LOG_NOTICE,
		    _
		    ("now for %d seconds you can move mouse, touch keyboard, punch joystick"),
		    TEST_DURATION_EVENTS / 1000);
	input = lw6gfx_pump_events (backend);
	while (lw6sys_uptime () < ticks + TEST_DURATION_EVENTS
	       && !input->quit)
	  {
	    if (!lw6gfx_display (backend,
				 LW6GUI_DISPLAY_BACKGROUND, look, NULL,
				 NULL, NULL, 0, NULL, 0.0f, 0, 0, NULL, 0, 0))
	      {
		lw6sys_log (LW6SYS_LOG_WARNING, _("display error"));
		ret = 0;
	      }
	    lw6sys_sleep (TEST_SLEEP);
	    input = lw6gfx_pump_events (backend);
	    if (lw6gui_button_pop_press (&(input->mouse.button_left)))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("left mouse button pressed at %dx%d"),
			    input->mouse.x, input->mouse.y);
	      }
	    if (lw6gui_button_pop_press (&(input->mouse.button_right)))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("right mouse button pressed at %d,%d"),
			    input->mouse.x, input->mouse.y);
	      }
	    if (lw6gui_button_pop_press (&(input->mouse.wheel_up)))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("mouse wheel up"));
	      }
	    if (lw6gui_button_pop_press (&(input->mouse.wheel_down)))
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE, _("mouse wheel down"));
	      }
	    keypress = lw6gui_keyboard_pop_keypress (&(input->keyboard));
	    if (keypress)
	      {
		repr = lw6gui_keypress_repr (keypress);
		if (repr)
		  {
		    lw6sys_log (LW6SYS_LOG_NOTICE, _("%s"), repr);
		    LW6SYS_FREE (repr);
		  }
		lw6gui_keypress_free (keypress);
	      }
	    if (input->quit)
	      {
		lw6sys_log (LW6SYS_LOG_NOTICE,
			    _("a QUIT event was detected"));
	      }
	  }
	lw6gui_look_free (look);
      }
    else
      {
	ret = 0;
      }
  }

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}


static int
test_quit (lw6gfx_backend_t * backend)
{
  int ret = 1;
  LW6SYS_TEST_FUNCTION_BEGIN;

  lw6gfx_quit (backend);

  LW6SYS_TEST_FUNCTION_END;
  return ret;
}

int
lw6gfx_test ()
{
  int ret = 0;
  lw6gfx_backend_t *backend = NULL;
  int argc = TEST_ARGC;
  char *argv[TEST_ARGC] = { TEST_ARGV0 };

  if (lw6sys_false ())
    {
      /*
       * Just to make sure most functions are stuffed in the binary
       */
      lw6sys_test ();
      lw6cfg_test ();
      lw6map_test ();
      lw6ker_test ();
      lw6gui_test ();
    }

#ifdef MOD_GL
  backend = lw6gfx_create_backend (argc, argv, "gl");
  if (backend)
    {
      ret = (test_init (backend) && test_resolution (backend)
	     && test_background (backend) && test_menu (backend)
	     && test_events (backend) && test_quit (backend)) || ret;
      lw6gfx_destroy_backend (backend);
    }
#endif

  return ret;
}
