/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "../../gfx.h"
#include "gl-utils.h"

static void
log_event (SDL_Event * event)
{
  switch (event->type)
    {
    case SDL_ACTIVEEVENT:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("SDL event type=SDL_ACTIVEEVENT gain=%d state=%d"),
		  (int) event->active.gain, (int) event->active.state);
      break;
    case SDL_KEYDOWN:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_KEYDOWN state=%d scancode=%d sym=%d unicode=%d"),
		  (int) event->key.state,
		  (int) event->key.keysym.scancode,
		  (int) event->key.keysym.sym,
		  (int) event->key.keysym.mod,
		  (int) event->key.keysym.unicode);
      break;
    case SDL_KEYUP:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_KEYUP state=%d scancode=%d sym=%d unicode=%d"),
		  (int) event->key.state,
		  (int) event->key.keysym.scancode,
		  (int) event->key.keysym.sym,
		  (int) event->key.keysym.mod,
		  (int) event->key.keysym.unicode);
      break;
    case SDL_MOUSEMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_MOUSEMOTION state=%d x=%d y=%d xrel=%d yrel=%d"),
		  (int) event->motion.state, (int) event->motion.x,
		  (int) event->motion.y, (int) event->motion.xrel,
		  (int) event->motion.yrel);
      break;
    case SDL_MOUSEBUTTONDOWN:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_MOUSEBUTTONDOWN button=%d state=%d x=%d y=%d"),
		  (int) event->button.button, (int) event->button.state,
		  (int) event->motion.x, (int) event->motion.y);
      break;
    case SDL_MOUSEBUTTONUP:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_MOUSEBUTTONUP button=%d state=%d x=%d y=%d"),
		  (int) event->button.button, (int) event->button.state,
		  (int) event->motion.x, (int) event->motion.y);
      break;
    case SDL_JOYAXISMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYAXISMOTION which=%d axis=%d value=%d"),
		  (int) event->jaxis.which, event->jaxis.axis,
		  (int) event->jaxis.value);
      break;
    case SDL_JOYBALLMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYBALLMOTION which=%d ball=%d xrel=%d yrel=%d"),
		  (int) event->jball.which, (int) event->jball.ball,
		  (int) event->jball.xrel, (int) event->jball.yrel);
      break;
    case SDL_JOYHATMOTION:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYHATMOTION which=%d hat=%d value=%d"),
		  (int) event->jhat.which, (int) event->jhat.hat,
		  (int) event->jhat.value);
      break;
    case SDL_JOYBUTTONDOWN:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYBUTTONDOWN which=%d button=%d state=%d"),
		  (int) event->jbutton.which, (int) event->jbutton.button,
		  (int) event->jbutton.state);
      break;
    case SDL_JOYBUTTONUP:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _
		  ("SDL event type=SDL_JOYBUTTONUP which=%d button=%d state=%d"),
		  (int) event->jbutton.which, (int) event->jbutton.button,
		  (int) event->jbutton.state);
      break;
    case SDL_QUIT:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("SDL event type=SDL_QUIT"));
      break;
    case SDL_SYSWMEVENT:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("SDL event type=SDL_SYSWMEVENT"));
      break;
    case SDL_VIDEORESIZE:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("SDL event type=SDL_VIDEORESIZE w=%d h=%d"),
		  (int) event->resize.w, (int) event->resize.h);
      break;
    case SDL_VIDEOEXPOSE:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("SDL event type=SDL_VIDEOEXPOSE"));
      break;
    case SDL_USEREVENT:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("SDL event type=SDL_USEREVENT code=%d"),
		  (int) event->user.code);
      break;
    default:
// should never get there, unless API changes...
      lw6sys_log (LW6SYS_LOG_WARNING, _("SDL event with unknown type=%d"),
		  event->type);
    }
}

static void
key_down (lw6gui_keyboard_t * keyboard, SDL_Event * event,
	  mod_gl_utils_const_data_t * const_data, int64_t timestamp)
{
  int sym = 0;

  sym = event->key.keysym.sym;

  lw6gui_keyboard_register_key_down (keyboard, sym,
				     event->key.keysym.unicode,
				     SDL_GetKeyName (sym), timestamp);

  if (sym == const_data->keysym1_up ||
      sym == const_data->keysym2_up ||
      sym == const_data->keysym3_up || sym == const_data->keysym4_up)
    {
      lw6gui_button_register_down (&(keyboard->key_up), timestamp);
    }
  if (sym == const_data->keysym1_down ||
      sym == const_data->keysym2_down ||
      sym == const_data->keysym3_down || sym == const_data->keysym4_down)
    {
      lw6gui_button_register_down (&(keyboard->key_down), timestamp);
    }
  if (sym == const_data->keysym1_left ||
      sym == const_data->keysym2_left ||
      sym == const_data->keysym3_left || sym == const_data->keysym4_left)
    {
      lw6gui_button_register_down (&(keyboard->key_left), timestamp);
    }
  if (sym == const_data->keysym1_right ||
      sym == const_data->keysym2_right ||
      sym == const_data->keysym3_right || sym == const_data->keysym4_right)
    {
      lw6gui_button_register_down (&(keyboard->key_right), timestamp);
    }
  if (sym == const_data->keysym1_enter || sym == const_data->keysym2_enter)
    {
      lw6gui_button_register_down (&(keyboard->key_enter), timestamp);
    }
  if (sym == const_data->keysym1_esc || sym == const_data->keysym2_esc)
    {
      lw6gui_button_register_down (&(keyboard->key_esc), timestamp);
    }
  if (sym == const_data->keysym1_pgup || sym == const_data->keysym2_pgup)
    {
      lw6gui_button_register_down (&(keyboard->key_pgup), timestamp);
    }
  if (sym == const_data->keysym1_pgdown || sym == const_data->keysym2_pgdown)
    {
      lw6gui_button_register_down (&(keyboard->key_pgdown), timestamp);
    }
}

static void
key_up (lw6gui_keyboard_t * keyboard, SDL_Event * event,
	mod_gl_utils_const_data_t * const_data)
{
  int sym = 0;

  sym = event->key.keysym.sym;

  lw6gui_keyboard_register_key_up (keyboard, sym);

  if (sym == const_data->keysym1_up ||
      sym == const_data->keysym2_up ||
      sym == const_data->keysym3_up || sym == const_data->keysym4_up)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_up)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym2_up)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym3_up)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym4_up))
	{
	  lw6gui_button_register_up (&(keyboard->key_up));
	}
    }
  if (sym == const_data->keysym1_down ||
      sym == const_data->keysym2_down ||
      sym == const_data->keysym3_down || sym == const_data->keysym4_down)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_down)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym2_down)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym3_down)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym4_down))
	{
	  lw6gui_button_register_up (&(keyboard->key_down));
	}
    }
  if (sym == const_data->keysym1_left ||
      sym == const_data->keysym2_left ||
      sym == const_data->keysym3_left || sym == const_data->keysym4_left)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_left)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym2_left)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym3_left)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym4_left))
	{
	  lw6gui_button_register_up (&(keyboard->key_left));
	}
    }
  if (sym == const_data->keysym1_right ||
      sym == const_data->keysym2_right ||
      sym == const_data->keysym3_right || sym == const_data->keysym4_right)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_right)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym2_right)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym3_right)
	  && !lw6gui_keyboard_is_pressed (keyboard,
					  const_data->keysym4_right))
	{
	  lw6gui_button_register_up (&(keyboard->key_right));
	}
    }
  if (sym == const_data->keysym1_enter || sym == const_data->keysym2_enter)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_enter)
	  && !lw6gui_keyboard_is_pressed (keyboard,
					  const_data->keysym2_enter))
	{
	  lw6gui_button_register_up (&(keyboard->key_enter));
	}
    }
  if (sym == const_data->keysym1_esc || sym == const_data->keysym2_esc)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_esc)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym2_esc))
	{
	  lw6gui_button_register_up (&(keyboard->key_esc));
	}
    }
  if (sym == const_data->keysym1_pgup || sym == const_data->keysym2_pgup)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_pgup)
	  && !lw6gui_keyboard_is_pressed (keyboard, const_data->keysym2_pgup))
	{
	  lw6gui_button_register_up (&(keyboard->key_pgup));
	}
    }
  if (sym == const_data->keysym1_pgdown || sym == const_data->keysym2_pgdown)
    {
      if (!lw6gui_keyboard_is_pressed (keyboard, const_data->keysym1_pgdown)
	  && !lw6gui_keyboard_is_pressed (keyboard,
					  const_data->keysym2_pgdown))
	{
	  lw6gui_button_register_up (&(keyboard->key_pgdown));
	}
    }
}

static void
mouse_move (lw6gui_mouse_t * mouse, SDL_Event * event, int64_t timestamp)
{
  lw6gui_mouse_register_move (mouse, event->button.x, event->button.y,
			      timestamp);
}

static void
mouse_button_down (lw6gui_mouse_t * mouse, SDL_Event * event,
		   int64_t timestamp)
{
  switch (event->button.button)
    {
    case SDL_BUTTON_LEFT:
      lw6gui_button_register_down (&(mouse->button_left), timestamp);
      break;
    case SDL_BUTTON_RIGHT:
      lw6gui_button_register_down (&(mouse->button_right), timestamp);
      break;
    case SDL_BUTTON_WHEELUP:
      lw6gui_button_register_down (&(mouse->wheel_up), timestamp);
      /*
       * WHEELUP & WHEELDOWN are not ordinary buttons, they never
       * stay pressed so we unpress them right on and only the
       * poll()-like functions which query if it has been pressed
       * will have some effects.
       */
      lw6gui_button_register_up (&(mouse->wheel_up));
      lw6gui_button_register_up (&(mouse->wheel_down));
      break;
    case SDL_BUTTON_WHEELDOWN:
      lw6gui_button_register_down (&(mouse->wheel_down), timestamp);
      /*
       * WHEELUP & WHEELDOWN are not ordinary buttons, they never
       * stay pressed so we unpress them right on and only the
       * poll()-like functions which query if it has been pressed
       * will have some effects.
       */
      lw6gui_button_register_up (&(mouse->wheel_up));
      lw6gui_button_register_up (&(mouse->wheel_down));
      break;
    }
}

static void
mouse_button_up (lw6gui_mouse_t * mouse, SDL_Event * event)
{
  switch (event->button.button)
    {
    case SDL_BUTTON_LEFT:
      lw6gui_button_register_up (&(mouse->button_left));
      break;
    case SDL_BUTTON_RIGHT:
      lw6gui_button_register_up (&(mouse->button_right));
      break;
    case SDL_BUTTON_WHEELUP:
      lw6gui_button_register_up (&(mouse->wheel_up));
      break;
    case SDL_BUTTON_WHEELDOWN:
      lw6gui_button_register_up (&(mouse->wheel_down));
      break;
    }
}

static int
joystick_index (SDL_Event * event, mod_gl_utils_const_data_t * const_data)
{
  int ret = -1;

  if (event->jaxis.which == const_data->joystick1_index)
    {
      ret = LW6GUI_JOYSTICK1_ID;
    }
  if (event->jaxis.which == const_data->joystick2_index)
    {
      ret = LW6GUI_JOYSTICK2_ID;
    }

  return ret;
}

static int
joystick_button_index (SDL_Event * event,
		       mod_gl_utils_const_data_t * const_data, int i)
{
  int ret = -1;

  switch (i)
    {
    case LW6GUI_JOYSTICK1_ID:
      if (event->jbutton.button == const_data->joystick1_button_a_index)
	{
	  ret = 0;
	}
      if (event->jbutton.button == const_data->joystick1_button_b_index)
	{
	  ret = 1;
	}
      if (event->jbutton.button == const_data->joystick1_button_c_index)
	{
	  ret = 2;
	}
      if (event->jbutton.button == const_data->joystick1_button_d_index)
	{
	  ret = 3;
	}
      break;
    case LW6GUI_JOYSTICK2_ID:
      if (event->jbutton.button == const_data->joystick2_button_a_index)
	{
	  ret = 0;
	}
      if (event->jbutton.button == const_data->joystick2_button_b_index)
	{
	  ret = 1;
	}
      if (event->jbutton.button == const_data->joystick2_button_c_index)
	{
	  ret = 2;
	}
      if (event->jbutton.button == const_data->joystick2_button_d_index)
	{
	  ret = 3;
	}
      break;
    }

  /*
   * If button is outside our range we still take it in account but with
   * a non-modifiable binding.
   */
  if (ret < 0 && const_data->joystick_all_buttons)
    {
      ret = event->jbutton.button % LW6GUI_NB_JOYSTICK_BUTTONS;
    }

  return ret;
}

static void
joystick_move (lw6gui_joystick_t * joystick, SDL_Event * event, int limit,
	       int64_t timestamp)
{
  switch (event->jaxis.axis)
    {
    case 0:
      lw6gui_joystick_update_axis_x (joystick, event->jaxis.value, limit,
				     timestamp);
      break;
    case 1:
      lw6gui_joystick_update_axis_y (joystick, event->jaxis.value, limit,
				     timestamp);
      break;
    default:
      lw6sys_log (LW6SYS_LOG_DEBUG,
		  _("joystick %d axis %d with value %d ignored"),
		  event->jaxis.which, event->jaxis.axis, event->jaxis.value);
      break;
    }
}

static void
joystick_button_down (lw6gui_joystick_t * joystick, int b, SDL_Event * event,
		      int64_t timestamp)
{
  switch (b)
    {
    case 0:
      lw6gui_button_register_down (&(joystick->button_a), timestamp);
      break;
    case 1:
      lw6gui_button_register_down (&(joystick->button_b), timestamp);
      break;
    case 2:
      lw6gui_button_register_down (&(joystick->button_c), timestamp);
      break;
    case 3:
      lw6gui_button_register_down (&(joystick->button_d), timestamp);
      break;
    default:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("joystick %d button %d ignored"),
		  event->jbutton.which, event->jbutton.button);
      break;
    }
}

static void
joystick_button_up (lw6gui_joystick_t * joystick, int b, SDL_Event * event)
{
  switch (b)
    {
    case 0:
      lw6gui_button_register_up (&(joystick->button_a));
      break;
    case 1:
      lw6gui_button_register_up (&(joystick->button_b));
      break;
    case 2:
      lw6gui_button_register_up (&(joystick->button_c));
      break;
    case 3:
      lw6gui_button_register_up (&(joystick->button_d));
      break;
    default:
      lw6sys_log (LW6SYS_LOG_DEBUG, _("joystick %d button %d ignored"),
		  event->jbutton.which, event->jbutton.button);
      break;
    }
}

/*
 * Internal poll function.
 */
lw6gui_input_t *
mod_gl_utils_pump_events (mod_gl_utils_context_t * utils_context)
{
  lw6gui_video_mode_t video_mode;
  SDL_Event event;
  int64_t timestamp;
  lw6gui_input_t *input = &(utils_context->input);
  mod_gl_utils_const_data_t *const_data = &(utils_context->const_data);
  int i, b;

  timestamp = mod_gl_utils_timer_timestamp (utils_context);
  memset (&event, 0, sizeof (SDL_Event));
  while (SDL_PollEvent (&event))
    {
      log_event (&event);
      switch (event.type)
	{
	case SDL_KEYDOWN:
	  lw6gui_input_register_change (input);
	  key_down (&(input->keyboard), &event, const_data, timestamp);
	  if (event.key.keysym.sym == const_data->keysym_quit)
	    {
	      input->quit = 1;
	    }
	  break;
	case SDL_KEYUP:
	  lw6gui_input_register_change (input);
	  key_up (&(input->keyboard), &event, const_data);
	  break;
	case SDL_QUIT:
	  lw6gui_input_register_change (input);
	  input->quit = 1;
	  break;
	case SDL_MOUSEMOTION:
	  lw6gui_input_register_change (input);
	  mouse_move (&(input->mouse), &event, timestamp);
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  lw6gui_input_register_change (input);
	  mouse_move (&(input->mouse), &event, timestamp);
	  mouse_button_down (&(input->mouse), &event, timestamp);
	  break;
	case SDL_MOUSEBUTTONUP:
	  lw6gui_input_register_change (input);
	  mouse_move (&(input->mouse), &event, timestamp);
	  mouse_button_up (&(input->mouse), &event);
	  break;
	case SDL_JOYAXISMOTION:
	  lw6gui_input_register_change (input);
	  i = joystick_index (&event, const_data);
	  if (lw6gui_joystick_check_index (i))
	    {
	      joystick_move (&(input->joysticks[i]), &event,
			     const_data->joystick_limit, timestamp);
	    }
	  break;
	case SDL_JOYBUTTONDOWN:
	  lw6gui_input_register_change (input);
	  i = joystick_index (&event, const_data);
	  if (lw6gui_joystick_check_index (i))
	    {
	      b = joystick_button_index (&event, const_data, i);
	      joystick_button_down (&(input->joysticks[i]), b, &event,
				    timestamp);
	    }
	  break;
	case SDL_JOYBUTTONUP:
	  lw6gui_input_register_change (input);
	  i = joystick_index (&event, const_data);
	  if (lw6gui_joystick_check_index (i))
	    {
	      b = joystick_button_index (&event, const_data, i);
	      joystick_button_up (&(input->joysticks[i]), b, &event);
	    }
	  break;
	case SDL_VIDEORESIZE:
	  mod_gl_utils_get_video_mode (utils_context, &video_mode);
	  video_mode.width = event.resize.w;
	  video_mode.height = event.resize.h;
	  mod_gl_utils_resize_video_mode (utils_context, &video_mode);
	  break;
	case SDL_VIDEOEXPOSE:
	  mod_gl_utils_sync_mode (utils_context, 0);
	  break;
	}
    }

  return input;
}
