/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "../../gfx.h"
#include "gl-utils.h"

/*
 * This is where the game map with actual information "there are
 * fighters here" is done.
 */
static void
update_game_surface (mod_gl_utils_context_t * utils_context,
		     SDL_Surface * surface,
		     lw6ker_game_state_t * game_state,
		     lw6gui_look_t * look, int x0, int y0, int w, int h)
{
  int x, y, z, x1, y1, d;
  int safe_x, safe_y;
  int fighter_id, hidden;
  lw6ker_fighter_t *fighter;
  Uint32 color;
  float hidden_layer_alpha;
  int team_color;
  lw6map_rules_t *rules;
  lw6sys_whd_t *shape;
  int bytes_per_pixel;

  hidden_layer_alpha = look->style.hidden_layer_alpha;
  x1 = x0 + w;
  y1 = y0 + h;
  shape = &(game_state->map_state.shape);
  d = shape->d;
  rules = &(game_state->game_struct->rules);
  bytes_per_pixel = surface->format->BytesPerPixel;
  for (x = x0; x < x1; ++x)
    {
      for (y = y0; y < y1; ++y)
	{
	  fighter_id = -1;
	  hidden = 0;
	  safe_x = x;
	  safe_y = y;
	  lw6map_coords_fix_xy (rules, shape, &safe_x, &safe_y);
	  for (z = 0; z < d && fighter_id < 0; ++z)
	    {
	      fighter_id =
		lw6ker_map_state_get_fighter_id (&(game_state->map_state),
						 safe_x, safe_y, z);
	      /*
	       * It's important to test d>1 else a call
	       * to get_zone_id is done when it's not needed,
	       * and this function does cost CPU. When d==1 hidden
	       * just makes no sense.
	       */
	      if (d > 1 && fighter_id < 0)
		{
		  hidden =
		    (lw6ker_map_struct_get_zone_id
		     (game_state->map_state.map_struct, safe_x, safe_y,
		      z) < 0);
		}
	    }
	  if (fighter_id >= 0)
	    {
	      fighter = &(game_state->map_state.armies.fighters[fighter_id]);
	      team_color = fighter->team_color;
	      if (team_color >= 0)
		{
		  /*
		   * We need to cache & check team_color for in dirty read
		   * mode, team_color could be negative even if fighter was
		   * reported active just before.
		   */
		  color = mod_gl_utils_get_shaded_color_for_fighter
		    (utils_context, team_color, fighter->health);
		  if (hidden)
		    {
		      mod_gl_utils_force_color32_alpha (&color,
							hidden_layer_alpha);
		    }
		}
	      if (bytes_per_pixel == 4)
		{
		  mod_gl_utils_putpixel_4_bytes_per_pixel (surface, x - x0,
							   y - y0, color);
		}
	      else
		{
		  mod_gl_utils_putpixel (surface, x - x0, y - y0, color);
		}
	    }
	  else
	    {
	      if (bytes_per_pixel == 4)
		{
		  mod_gl_utils_putpixel_4_bytes_per_pixel (surface, x - x0,
							   y - y0, 0);
		}
	      else
		{
		  mod_gl_utils_putpixel (surface, x - x0, y - y0, 0);
		}
	    }
	}
    }
}

void
mod_gl_utils_update_game_texture (mod_gl_utils_context_t * utils_context,
				  SDL_Surface * surface,
				  GLuint texture,
				  lw6ker_game_state_t * game_state,
				  lw6gui_look_t * look)
{
  update_game_surface (utils_context, surface, game_state, look,
		       0, 0, game_state->map_state.shape.w,
		       game_state->map_state.shape.h);
  /*
   * Using this instead of creating a new texture from a surface using
   * glTexImage2D (surface2texture) is important, for performance.
   * This is called at each display, we don't want a new texture to
   * be created each time: it's always the same size and represents
   * pretty much the same thing/layer -> where fighters are!
   */
  mod_gl_utils_texture_update (utils_context, texture, surface,
			       look->style.pixelize);
}

void
mod_gl_utils_update_game_texture_array (mod_gl_utils_context_t *
					utils_context,
					mod_gl_utils_surface_array_t *
					surface_array,
					mod_gl_utils_texture_array_t *
					texture_array,
					lw6ker_game_state_t * game_state,
					lw6gui_look_t * look)
{
  int n_x, n_y;

  if (utils_context->last_action.game_texture_array_update_id !=
      game_state->id
      || utils_context->last_action.game_texture_array_update_rounds !=
      game_state->rounds)
    {
      utils_context->last_action.game_texture_array_update_id =
	game_state->id;
      utils_context->last_action.game_texture_array_update_rounds =
	game_state->rounds;

      for (n_y = 0; n_y < surface_array->layout.n_h; ++n_y)
	{
	  for (n_x = 0; n_x < surface_array->layout.n_w; ++n_x)
	    {
	      update_game_surface (utils_context,
				   mod_gl_utils_get_surface_array_surface
				   (surface_array, n_x, n_y), game_state,
				   look,
				   surface_array->layout.x0[n_x],
				   surface_array->layout.y0[n_y],
				   surface_array->layout.w[n_x],
				   surface_array->layout.h[n_y]);
	    }
	}

      /*
       * Now transform surfaces into textures.
       */
      mod_gl_utils_texture_array_update (utils_context, texture_array,
					 surface_array, look->style.pixelize);
    }
  else
    {
      // same as before, no need to update
    }
}
