/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <dirent.h>
#include <sys/stat.h>

#include "sys.h"

#define DIR_SEP_CHAR_UNIX '/'
#define DIR_SEP_STR_UNIX "/"
#define DIR_SEP_CHAR_MS_WINDOWS '\\'
#define DIR_SEP_STR_MS_WINDOWS "\\"

#ifdef LW6_MS_WINDOWS
#define DIR_SEP_CHAR DIR_SEP_CHAR_MS_WINDOWS
#define DIR_SEP_STR DIR_SEP_STR_MS_WINDOWS
#else
#define DIR_SEP_CHAR DIR_SEP_CHAR_UNIX
#define DIR_SEP_STR DIR_SEP_STR_UNIX
#endif

static int
is_dir_sep (char c)
{
  return ((c == DIR_SEP_CHAR_UNIX) || (c == DIR_SEP_CHAR_MS_WINDOWS));
}

static void
normalize (char *path)
{
  char *pos = NULL;

  for (pos = path; (*pos); pos++)
    {
      if (is_dir_sep (*pos))
	{
	  (*pos) = DIR_SEP_CHAR;
	}
    }
}

/**
 * lw6sys_file_exists
 *
 * @filename: the file to test
 *
 * Tests the existence of a file on the filesystem. File is considered
 * to exists if it's at least readable.
 *
 * Return value: 1 if OK, 0 if file doesn't exist or can't be read.
 */
int
lw6sys_file_exists (char *filename)
{
  int ret = 0;
  FILE *f;

  f = fopen (filename, "r");
  if (f)
    {
      ret = 1;
      fclose (f);
    }

  return ret;
}

/**
 * lw6sys_dir_exists
 *
 * @dirname: the directory to test
 *
 * Tests the existence of a directory on the filesystem.
 *
 * Return value: 1 if OK, 0 if directory doesn't exist.
 */
int
lw6sys_dir_exists (char *dirname)
{
  int ret = 0;
  DIR *dir = NULL;

  dir = opendir (dirname);
  if (dir)
    {
      closedir (dir);
      dir = NULL;
      ret = 1;
    }

  return ret;
}

static int
create_dir (char *dirname, int verbose)
{
  int ret = 0;

#ifdef LW6_MS_WINDOWS
  if (mkdir (dirname))
#else
  if (mkdir (dirname, S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP) == 0)
#endif
    {
      if (verbose)
	{
	  lw6sys_log (LW6SYS_LOG_INFO, _("creating dir \"%s\""), dirname);
	}
      ret = lw6sys_dir_exists (dirname);
    }
  else
    {
      if (verbose)
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, _("unable to create dir \"%s\""),
		      dirname);
	}
    }

  return ret;
}

/**
 * lw6sys_create_dir
 *
 * @dirname: the directory to create
 *
 * Creates a directory, performing sanity checks such as verifying
 * the directory really exists after being created.
 *
 * Return value: 1 if OK, 0 if error.
 */
int
lw6sys_create_dir (char *dirname)
{
  return create_dir (dirname, 1);
}

/**
 * lw6sys_create_dir_silent
 *
 * @dirname: the directory to create
 *
 * Creates a directory like @lw6sys_create_dir but this function
 * is silent in the sense that it won't log any error. Usefull
 * to create the log directory itself, for instance, and avoid
 * infinite loops on error.
 *
 * Return value: 1 if OK, 0 if error.
 */
int
lw6sys_create_dir_silent (char *dirname)
{
  return create_dir (dirname, 0);
}

/**
 * lw6sys_path_add_slash
 *
 * @path: a path
 *
 * Adds a slash, or in a general manner, a directory separator,
 * at the end of a path, if needed. So /foo/bar will become
 * /foo/bar/ but /bar/foo/ will remain /bar/foo/.
 *
 * Return value: a newly allocated string, must be freed.
 */
char *
lw6sys_path_add_slash (char *path)
{
  char *ret;
  int len;

  len = strlen (path);
  if (len == 0 || (len > 0 && !is_dir_sep (path[len - 1])))
    {
      ret = lw6sys_str_concat (path, DIR_SEP_STR);
    }
  else
    {
      ret = lw6sys_str_copy (path);
    }

  if (ret)
    {
      normalize (ret);
    }

  return ret;
}

/**
 * lw6sys_path_strip_slash
 *
 * @path: a path
 *
 * Strips the slash, or in a general manner, the directory separator,
 * at the end of a path, if needed. So /foo/bar/ will become
 * /foo/bar but /bar/foo will remain /bar/foo.
 *
 * Return value: a newly allocated string, must be freed.
 */
char *
lw6sys_path_strip_slash (char *path)
{
  char *ret;
  int len;

  ret = lw6sys_str_copy (path);
  if (ret)
    {
      len = strlen (ret);
      if (len > 0 && is_dir_sep (ret[len - 1]))
	{
	  ret[len - 1] = '\0';
	}
    }

  if (ret)
    {
      normalize (ret);
    }

  return ret;
}

/**
 * lw6sys_path_concat
 *
 * @path1: left part of the path
 * @path2: right part of the path
 *
 * Concatenates 2 parts of a path. Function will try to avoid
 * stupid "double-slash" when concatenating /foo/ with /bar/
 * and conversely insert a directory separator when concatenating
 * /foo with bar/.
 *
 * Return value: a newly allocated string, must be freed.
 */
char *
lw6sys_path_concat (char *path1, char *path2)
{
  char *path1_stripped = NULL;
  char *path2_stripped = NULL;
  char *path2_start = NULL;
  char *ret = NULL;

  path1_stripped = lw6sys_path_strip_slash (path1);
  if (path1_stripped)
    {
      path2_stripped = lw6sys_path_strip_slash (path2);
      if (path2_stripped)
	{
	  path2_start = path2_stripped;
	  if (is_dir_sep (path2_start[0]))
	    {
	      path2_start++;
	    }
	  if (!lw6sys_path_is_cwd (path2_start))
	    {
	      if (strlen (path1_stripped) > 0)
		{
		  ret =
		    lw6sys_new_sprintf ("%s" DIR_SEP_STR "%s", path1_stripped,
					path2_start);
		}
	      else
		{
		  ret = lw6sys_str_copy (path2_start);
		}
	    }
	  else
	    {
	      ret = lw6sys_str_copy (path1_stripped);
	    }
	  LW6SYS_FREE (path2_stripped);
	}
      LW6SYS_FREE (path1_stripped);
    }

  if (ret)
    {
      normalize (ret);
    }

  return ret;
}

/**
 * lw6sys_path_split
 *
 * @path: a path
 *
 * Splits a path into all its parts. For instance /boo/bar/foo2/bar2
 * returns a 4 elements list. This is more than a plain split, for
 * heading and tailing slashes will be ignored, and various path
 * separators will be interpreted (depends on platform).
 *
 * Return value: a list containing 0-terminated strings.
 */
lw6sys_list_t *
lw6sys_path_split (char *path)
{
  char *stripped_path = NULL;
  lw6sys_list_t *ret = NULL;

  stripped_path = lw6sys_path_strip_slash (path);
  if (stripped_path)
    {
      ret = lw6sys_str_split_no_0 (stripped_path, DIR_SEP_CHAR);
      LW6SYS_FREE (stripped_path);
    }

  return ret;
}

/**
 * lw6sys_path_is_relative
 *
 * @path: a path
 *
 * Checks wether a path is relative or absolute.
 *
 * Return value: 1 if relative, 0 if absolute.
 */
int
lw6sys_path_is_relative (char *path)
{
  int ret = 0;

  if (strcmp (".", path) == 0 ||
      strncmp ("." DIR_SEP_STR_UNIX, path, 2) == 0 ||
      strncmp ("." DIR_SEP_STR_MS_WINDOWS, path, 2) == 0 ||
      strncmp (".." DIR_SEP_STR_UNIX, path, 3) == 0 ||
      strncmp (".." DIR_SEP_STR_MS_WINDOWS, path, 3) == 0)
    {
      ret = 1;
    }

  return ret;
}

/**
 * lw6sys_path_is_cwd
 *
 * @path: a path
 *
 * Checks wether a path is "." or not. Will also trap "" and "./".
 *
 * Return value: 1 if relative, 0 if absolute.
 */
int
lw6sys_path_is_cwd (char *path)
{
  int ret = 0;

  if (strcmp ("", path) == 0 ||
      strcmp (".", path) == 0 ||
      strcmp ("." DIR_SEP_STR_UNIX, path) == 0 ||
      strcmp ("." DIR_SEP_STR_MS_WINDOWS, path) == 0)
    {
      ret = 1;
    }

  return ret;
}

/**
 * lw6sys_path_parent
 *
 * @path: a path
 *
 * Returns the parent path. That will return /foo when given
 * /foo/bar in input. 
 *
 * Return value: a newly allocated string, must be freed.
 */
char *
lw6sys_path_parent (char *path)
{
  char *parent = NULL;
  char *stripped_path = NULL;
  char *pos = NULL;

  stripped_path = lw6sys_path_strip_slash (path);
  if (stripped_path)
    {
      pos = strrchr (stripped_path, DIR_SEP_CHAR);
      if (pos)
	{
	  if (strcmp (pos + 1, "..") == 0)
	    {
	      parent = lw6sys_path_concat (stripped_path, "..");
	    }
	  else
	    {
	      (*pos) = '\0';
	      parent = lw6sys_str_copy (stripped_path);
	    }
	}
      else
	{
	  if (strcmp (stripped_path, "") == 0
	      || strcmp (stripped_path, ".") == 0)
	    {
	      parent = lw6sys_str_copy ("..");
	    }
	  else
	    {
	      parent = lw6sys_str_copy (".");
	    }
	}
      LW6SYS_FREE (stripped_path);
    }

  if (parent)
    {
      normalize (parent);
    }

  return parent;
}

/**
 * lw6sys_path_unparent
 *
 * @path: a path
 *
 * Given the ../foo/bar path, will return foo/bar. Usefull to
 * get rid of heading ../ when a path is known to start with it.
 *
 * Return value: a newly allocated string, must be freed.
 */
char *
lw6sys_path_unparent (char *path)
{
  char *unparent = NULL;
  char *slashed_path = NULL;

  slashed_path = lw6sys_path_add_slash (path);
  if (slashed_path)
    {
      if (strncmp (".." DIR_SEP_STR_UNIX, slashed_path, 3) == 0 ||
	  strncmp (".." DIR_SEP_STR_MS_WINDOWS, slashed_path, 3) == 0)
	{
	  unparent = lw6sys_str_copy (path + 3);
	}
      else
	{
	  unparent = lw6sys_str_copy (path);
	}
      LW6SYS_FREE (slashed_path);
    }

  if (unparent)
    {
      normalize (unparent);
    }

  return unparent;
}

/**
 * lw6sys_path_unparent_no_malloc
 *
 * @path: a path
 *
 * Given the ../foo/bar path, will return foo/bar. Usefull to
 * get rid of heading ../ when a path is known to start with it.
 * This is different from @lw6sys_path_unparent just because
 * the result is not dynamically allocated and copied from source.
 *
 * Return value: a pointer which points somewhere within the
 *   string passed as an argument.
 */
char *
lw6sys_path_unparent_no_malloc (char *path)
{
  char *unparent = NULL;

  if (strncmp (".." DIR_SEP_STR_UNIX, path, 3) == 0 ||
      strncmp (".." DIR_SEP_STR_MS_WINDOWS, path, 3) == 0)
    {
      unparent = path + 3;
    }
  else
    {
      unparent = path;
    }

  return unparent;
}
