// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "listerola.h"
#include "host-list.h"
#include "host.h"
#include "event.h"
#include "hosted-event-list.h"
#include "load-random-listing.h"
#include "load-xmltv-listing.h"
#include "load-mrbs-listing.h"
#include "listerola-options.h"

Listerola::Listerola(ListerolaOptions &o)
  : opts(o)
{
  host_list_loader = NULL;
  EventDateRegion now (opts.get_duration());
  now.decrement_to_nearest_interval(opts.get_interval());
  current_event_time = now;
  //we do this, so that it doesn't start up and then flip ahead right away.
  flip_ahead_to_next_interval();
  opts.num_rows_changed.connect
    (sigc::mem_fun(options_changed, &sigc::signal<void>::emit));
  opts.duration_changed.connect
    (sigc::mem_fun(*this, &Listerola::on_duration_changed));
  opts.interval_changed.connect
    (sigc::mem_fun(options_changed, &sigc::signal<void>::emit));
  opts.listing_file_changed.connect
    (sigc::mem_fun(*this, &Listerola::on_listing_file_changed));
  opts.listing_file_changed.connect
    (sigc::mem_fun(listing_file_changed, &sigc::signal<void>::emit));
  opts.host_numbers_changed.connect
    (sigc::mem_fun(*this, &Listerola::on_host_numbers_changed));
  opts.host_names_changed.connect
    (sigc::mem_fun(*this, &Listerola::on_host_names_changed));
  update_host_list_loader();

}

Listerola::~Listerola()
{
  if (host_list_loader)
    delete host_list_loader;
}

void Listerola::next()
{
  channels.get_next_host();
}

HostedEventList Listerola::get_events(Host *host)
{
  if (host == NULL)
    host = channels.get_active_host();

  
  if (host == NULL)
    return HostedEventList(NULL);

  return host->get_events (current_event_time);
}

double Listerola::overlap_end(Event *event)
{
  if (event->get_date().straddles(current_event_time.get_ending_date()))
    return event->get_date().overlap_end(current_event_time.get_ending_date());
  else
    return 0.0;
}
      
double Listerola::overlap_start(Event *event)
{
  if (event->get_date().straddles(current_event_time))
    return event->get_date().overlap_start(current_event_time);
  else
    return 0.0;
}

bool Listerola::flip_ahead_to_next_interval()
{
  double then = get_current_date().get_time().as_double();

  EventDate nowdate;
  double now = nowdate.get_time().as_double();

  double diff = now - then;

  if (diff >= (get_interval() * 60 * 60) - (get_flipahead_interval() * 60))
    {
      EventDateRegion newnow = get_current_date();
      newnow.add_hours(get_interval());
      set_current_date(newnow);
      return true;
    }
  return false;
}

void Listerola::set_listing_file(std::string file, ListerolaOptions::Source kind)
{
  switch (kind)
    {
    case ListerolaOptions::SOURCE_XMLTV:
      opts.set_xmltv_listing_file(file);
      break;
    case ListerolaOptions::SOURCE_MRBS:
      opts.set_mrbs_listing_file(file);
      break;
    case ListerolaOptions::SOURCE_RANDOM:
      opts.set_random_listing_file(file);
      break;
    default:
      break;
    }

  if (host_list_loader)
    {
      if (host_list_loader->get_kind() == kind)
        host_list_loader->set_listing_file(file);
      else
        update_host_list_loader();
    }
  else
    update_host_list_loader();
}

void Listerola::update_host_list_loader()
{
  refresh_date = EventDate();
  if (host_list_loader)
    delete host_list_loader;
  channels.lClear();
  bool broken = false;
  switch (opts.get_listing_kind())
    {
    case ListerolaOptions::SOURCE_XMLTV:
      host_list_loader = new LoadXmlTvListing(get_host_list(), 
                                              opts.get_listing_file(), 
                                              current_event_time, broken);
      break;
    case ListerolaOptions::SOURCE_MRBS:
      host_list_loader = new LoadMrbsListing(get_host_list(), 
                                             opts.get_listing_file(), 
                                             current_event_time, broken);
      break;
    case ListerolaOptions::SOURCE_RANDOM:
      host_list_loader = new LoadRandomListing(get_host_list(), 
                                               opts.get_listing_file(), 
                                               current_event_time, broken);
      break;
    case ListerolaOptions::SOURCE_NONE:
      host_list_loader = NULL;
    default:
      break;
    }
  sort_host_list();
  if (host_list_loader)
    {
      host_list_loader->bad_file_format.connect
        (sigc::mem_fun(bad_listing_file, &sigc::signal<void>::emit));
      if (broken)
        bad_listing_file.emit();
    }
}

void Listerola::reload_listing_file()
{
  if (host_list_loader)
    {
      host_list_loader->reload();
      refresh_date = EventDate();
    }
}
    
void Listerola::on_duration_changed()
{
  current_event_time.set_duration(opts.get_duration());
  options_changed.emit();
}
    
void Listerola::on_listing_file_changed()
{
  update_host_list_loader();
  options_changed.emit();
}

std::list<std::string> Listerola::get_genres()
{
  return channels.get_genres();
}

bool Listerola::sort_by_numeric_id(const Host *lhs, const Host *rhs)
{
  int left = lhs->get_numeric_identifier();
  if (left == -1)
    left = 999999;
  int right = rhs->get_numeric_identifier();
  if (right == -1)
    right = 999999;
  return  left < right;
}

void Listerola::sort_host_list()
{
  for (HostList::iterator it = channels.begin(); it != channels.end(); it++)
    {
      Host *host = *it;
      int number = opts.get_host_number(host->get_name());
      if (number >= 0)
        host->set_numeric_identifier((guint32)number);
      std::string name = opts.get_host_name(host->get_name());
      host->set_friendly_name(name);
    }
  channels.sort(sort_by_numeric_id);
}

std::list<std::string> Listerola::get_host_names()
{
  std::list<std::string> host_names = channels.get_names();
  host_names.sort();
  return host_names;
}
    
void Listerola::on_host_numbers_changed()
{
  sort_host_list();
}

void Listerola::on_host_names_changed()
{
  sort_host_list();
}
