// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "load-mrbs-listing.h"
#include "CsvReader.h"
#include <time.h>
#include <assert.h>

using namespace std;
LoadMrbsListing::LoadMrbsListing(HostList &hosts, std::string file, EventDate starting_time, bool &broken)
  : HostListLoader(hosts, file, starting_time, ListerolaOptions::SOURCE_MRBS,
                   broken)
{
  broken = !reload();
}

LoadMrbsListing::~LoadMrbsListing()
{
}

static int
is_month (char *arg)
{
  time_t now;
  int retval = -1;
  int start_month;
  char month[128];
  struct tm when;
  now = time (NULL);
  localtime_r (&now, &when);
  start_month = when.tm_mon;
  while (1)
    {
      //check string
      strftime (month, sizeof(month), "%B", &when);

      if (strcasecmp (month, arg) == 0)
	retval = when.tm_mon + 1;

      //move to the next fortnight, bail if we've wrapped around.
      //i'm guessing that some locales might have more months,
      //but none shorter than 2 of our anglo-weeks.
      now += 60 * 60 * 24 * 31;
      localtime_r (&now, &when);
      if (when.tm_mon == start_month)
	break;

      //bail if we're done
      if (retval != -1)
	break;
    }
  return retval;
}

EventDateRegion LoadMrbsListing::get_event_time(std::string time)
{
  char *month = NULL;
  char *units = NULL;
  float duration = 0.0;
  struct tm tm;
  memset(&tm, 0, sizeof (tm));
  tm.tm_isdst = -1;
  int retval = sscanf(time.c_str(), "%*s %d %as %d %d:%d:%d - %f %as",
                      &tm.tm_mday, &month, &tm.tm_year, &tm.tm_hour, 
                      &tm.tm_min, &tm.tm_sec, &duration, &units);
  //"Tuesday 05 January 2010 18:30:00"
  retval = is_month(month);
  free(month);
  if (retval > -1)
    tm.tm_mon = retval - 1;
  tm.tm_year -= 1900;
  time_t starting_time = timelocal(&tm);
  int duration_secs = 0;
  std::string u = units;
  if (u == "minutes")
    duration_secs = duration * 60;
  else if (u == "hours")
    duration_secs = duration * 60 * 60;
  else if (u == "days")
    duration_secs = duration * 60 * 60 * 24;
  else if (u == "weeks")
    duration_secs = duration * 60 * 60 * 24 * 7;
  else
    {
      printf("unhandled `%s'\n", units);
    assert(0);
    }
  free (units);
  time_t ending_time = starting_time + duration_secs;
  EventDateRegion then(starting_time, ending_time);
  return then;
}


bool LoadMrbsListing::reload()
{
  HostList &channels = hostlist;
  channels.lClear();
  std::ifstream fileStream(filename.c_str());
  IGame::CsvReader reader(fileStream, ",");
  reader.InitTitle();
  int count = 0;
  while(reader.ReadLine())
    {
      std::string room;
      std::string name;
      std::string time;
      reader.Bind("Area - Room", room);
      if (room == "")
        continue;
      reader.Bind("Brief Description", name);
      Host *host = channels.get_host_with_name(name);
      if (host == NULL)
        {
          count++;
          host = new Host(name);
          channels.add(host);
        }
      reader.Bind("Start Time - Duration", time);

      EventDateRegion then = get_event_time(time);
      std::string genre = name;
      host->add(new Event(name, genre, then));
    }
  fileStream.close();
  return true;
}
