/*
 * @file xmltv.lib/xmltv.c XMLTV parser class
 *
 * atvguide -- a collection of helper libraries and a GTK+ frontend for XMLTV
 * Copyright (C) 2004  Andrew Ayer
 * Copyright (C) 2010  Ben Asselstine
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 * 02110-1301, USA.
*/


#include <iostream>
#include <libxml/tree.h>
#include <time.h>
#include <vector>
#include "xmltv.h"
#include "xmltv-program.h"
#include "xmltv-channel.h"
#include "xmltv-date.h"

using namespace XmlTv;

xmltv::xmltv ()
{
	// This does nothing.
	// Programmer must call init() to initialize shit.
	// Check return value of init() and make sure it's true.
}

xmltv::~xmltv ()
{
	xmlFreeDoc(xml_doc);
}

xmltv&	xmltv::operator= (const xmltv& x)
{
	// FIXME: This is dangerous because it's not copying the xml_doc.
	// When the old xmltv object is destroyed, the xml_doc will be freed
	// making the new xmltv object stuffed.
	
	// I'll fix this as soon as I learn more about libxml.  Too bad 
	// libxml's not object-oriented!

	xml_doc = x.xml_doc;
	xml_root = x.xml_root;

	return *this;
}

bool	xmltv::init (const char* file)
{
	xml_doc = xmlParseFile(file);
	if (!xml_doc)
		return false;

 	xml_root = xmlDocGetRootElement(xml_doc);

	if (!xml_root)
		return false;

	if (xmlStrcmp(xml_root->name, BAD_CAST "tv") != 0) {
		xmlFreeDoc(xml_doc);
		return false;
	}

	return true;
}

// To get a list of available channels.
void	xmltv::get_channels(std::vector<channel*>& channels)
{
	xmlNodePtr cur = xml_root->xmlChildrenNode;

	while (cur) {
		if (xmlStrcasecmp(cur->name, BAD_CAST "channel") == 0) {
			xmlChar*	id = xmlGetProp( cur, BAD_CAST "id" );
			xmlChar*	name = 0;
			xmlNodePtr	child = cur->xmlChildrenNode;

			while (child) {
				if (xmlStrcasecmp(child->name, BAD_CAST "display-name") == 0) {
					name = xmlNodeGetContent(child);
					break;
				}
				child = child->next;
			}

			if (id && name) {
				channel*	ch = new channel((const char*) id, (const char*) name);
				channels.push_back(ch);
			}

			if (id) xmlFree(id);
			if (name) xmlFree(name);
		}
		cur = cur->next;
	}
}

// To get all the programs on a channel.
void	xmltv::get_programs_by_channel(const char* channelid, std::vector<program*>& programs)
{
	xmlNodePtr cur = xml_root->xmlChildrenNode;

	while (cur) {
		if (xmlStrcasecmp(cur->name, BAD_CAST "programme") == 0) {
			xmlChar* channel = xmlGetProp( cur, BAD_CAST "channel" );
			if (channel && xmlStrcasecmp(channel, BAD_CAST channelid) == 0) {
				xmlChar* start = xmlGetProp(cur, BAD_CAST "start");
				if (start) {
					//long start_time = (long) tvdate::parse_xmltv_date((char*) start);

					program*	prgm = new program;
					prgm->init(cur);

					programs.push_back(prgm);
					xmlFree( start );
				}
			}
			if (channel)
				xmlFree(channel);
		}
		cur = cur->next;
	}
}

// To get all the programs on ALL channels that are airing at time
void	xmltv::get_programs_by_time(time_t time, std::vector<program*>& programs)
{
	xmlNodePtr cur = xml_root->xmlChildrenNode;

	while (cur) {
		if (xmlStrcasecmp(cur->name, BAD_CAST "programme") == 0) {
			xmlChar* channel = xmlGetProp(cur, BAD_CAST "channel");
			xmlChar* start = xmlGetProp(cur, BAD_CAST "start");
			xmlChar* end = xmlGetProp(cur, BAD_CAST "stop");
			if (channel && start) {
				time_t start_time = tvdate::parse_xmltv_date((char*) start);
				time_t end_time = end ? tvdate::parse_xmltv_date((char*) end) : start_time + 1800;
				if (time >= start_time && time < end_time) {
					program*	prgm = new program;
					prgm->init(cur);

					programs.push_back(prgm);
				}

				xmlFree(channel);
				xmlFree(start);
				if (end)
					xmlFree(end);
			}
		}
		cur = cur->next;
	}
}

// Get all programs on ALL channels that are playing in between time1 and time2
void	xmltv::get_programs_by_start_time(time_t time1, time_t time2, std::vector<program*>& programs)
{
	xmlNodePtr cur = xml_root->xmlChildrenNode;

	while (cur) {
		if (xmlStrcasecmp(cur->name, BAD_CAST "programme") == 0) {
			xmlChar* start = xmlGetProp(cur, BAD_CAST "start");
			xmlChar* channel = xmlGetProp(cur, BAD_CAST "channel");
			if (channel && start) {
				long start_time = (long) tvdate::parse_xmltv_date((char*) start);
				if (start_time >= time1 && start_time < time2) {
					program*	prgm = new program;
					prgm->init(cur);

					programs.push_back(prgm);
				}

				xmlFree(channel);
				xmlFree(start);
			}
		}
		cur = cur->next;
	}
}
