/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* square.h -- type definitions for magic squares */

#ifndef _SQUARE_H
#define _SQUARE_H

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stddef.h>
#include <stdint.h>

/* The output file version */
#define FILE_VERSION 0

/* The magic number we write out when we write a binary file */
#define MAGIC_NUMBER 0x80b9faf4

/* Check if a `cellval_t' is valid */
#define cellval_valid(cellval)				\
  ((cellval).type == INT || (cellval).type == FLOAT)

/* Compare two `cellval_t's */
#define cellval_equal(a, b)			\
  (cellval_valid(a) && cellval_valid(b) &&	\
   (((a).type == INT) ? (a).i : (a).f) ==	\
   (((b).type == INT) ? (b).i : (b).f))

/* An `int' type for cells */
typedef long long cellvali_t;
typedef long double cellvalf_t;

/* A single cell value; it should be signed since we support negative
   numbers */
typedef struct cellval {
  union {
    cellvali_t i;
    cellvalf_t f;
  };
  enum {INT, FLOAT} type;
} cellval_t;

/* A single cell */
typedef struct cell {
  cellval_t val;
  char mutable; /* Whether this cell is permitted to change */
} cell_t;

/* A magic square */
typedef struct square {
  cell_t **cells;
  cellval_t *nums; /* Other valid numbers */
  size_t size, /* The size of the magic square */
    nums_size; /* The size of `nums' */
  char *description; /* An optional description of the magic square */
} square_t;

/* A list of magic squares */
typedef struct square_list {
  struct square_list *next;
  square_t square;
} square_list_t;

int square_solve(square_t *, square_t **, size_t *, char, char, FILE *
#ifdef HAVE_PTHREAD
		 , unsigned long
#endif
		 );
square_t square_dup(square_t);
square_t square_dup_nodesc(square_t);
void square_destroy(square_t *);
void square_destroy_nodesc(square_t *);

#endif /* !_SQUARE_H */
