(* Apache interface for mod_caml programs.
 * Copyright (C) 2003 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: apache.ml,v 1.11 2004/08/03 16:54:31 rwmj Exp $
 *)

(* Must appear in this order! *)
type result_type = OK | DECLINED | DONE

(* List of methods must match order in <httpd.h>. *)
type method_type = M_GET
                 | M_PUT
		 | M_POST
		 | M_DELETE
		 | M_CONNECT
		 | M_OPTIONS
		 | M_TRACE
		 | M_PATCH
		 | M_PROPFIND
		 | M_PROPPATCH
		 | M_MKCOL
		 | M_COPY
		 | M_MOVE
		 | M_LOCK
		 | M_UNLOCK
		 | M_INVALID

exception HttpError of int		(* Exceptions thrown by API functions*)

let cHTTP_CONTINUE =                      100
let cHTTP_SWITCHING_PROTOCOLS =           101
let cHTTP_PROCESSING =                    102
let cHTTP_OK =                            200
let cHTTP_CREATED =                       201
let cHTTP_ACCEPTED =                      202
let cHTTP_NON_AUTHORITATIVE =             203
let cHTTP_NO_CONTENT =                    204
let cHTTP_RESET_CONTENT =                 205
let cHTTP_PARTIAL_CONTENT =               206
let cHTTP_MULTI_STATUS =                  207
let cHTTP_MULTIPLE_CHOICES =              300
let cHTTP_MOVED_PERMANENTLY =             301
let cHTTP_MOVED_TEMPORARILY =             302
let cHTTP_SEE_OTHER =                     303
let cHTTP_NOT_MODIFIED =                  304
let cHTTP_USE_PROXY =                     305
let cHTTP_TEMPORARY_REDIRECT =            307
let cHTTP_BAD_REQUEST =                   400
let cHTTP_UNAUTHORIZED =                  401
let cHTTP_PAYMENT_REQUIRED =              402
let cHTTP_FORBIDDEN =                     403
let cHTTP_NOT_FOUND =                     404
let cHTTP_METHOD_NOT_ALLOWED =            405
let cHTTP_NOT_ACCEPTABLE =                406
let cHTTP_PROXY_AUTHENTICATION_REQUIRED = 407
let cHTTP_REQUEST_TIME_OUT =              408
let cHTTP_CONFLICT =                      409
let cHTTP_GONE =                          410
let cHTTP_LENGTH_REQUIRED =               411
let cHTTP_PRECONDITION_FAILED =           412
let cHTTP_REQUEST_ENTITY_TOO_LARGE =      413
let cHTTP_REQUEST_URI_TOO_LARGE =         414
let cHTTP_UNSUPPORTED_MEDIA_TYPE =        415
let cHTTP_RANGE_NOT_SATISFIABLE =         416
let cHTTP_EXPECTATION_FAILED =            417
let cHTTP_UNPROCESSABLE_ENTITY =          422
let cHTTP_LOCKED =                        423
let cHTTP_FAILED_DEPENDENCY =             424
let cHTTP_INTERNAL_SERVER_ERROR =         500
let cHTTP_NOT_IMPLEMENTED =               501
let cHTTP_BAD_GATEWAY =                   502
let cHTTP_SERVICE_UNAVAILABLE =           503
let cHTTP_GATEWAY_TIME_OUT =              504
let cHTTP_VERSION_NOT_SUPPORTED =         505
let cHTTP_VARIANT_ALSO_VARIES =           506
let cHTTP_INSUFFICIENT_STORAGE =          507
let cHTTP_NOT_EXTENDED =                  510

let cDOCUMENT_FOLLOWS =    cHTTP_OK
let cPARTIAL_CONTENT =     cHTTP_PARTIAL_CONTENT
let cMULTIPLE_CHOICES =    cHTTP_MULTIPLE_CHOICES
let cMOVED =               cHTTP_MOVED_PERMANENTLY
let cREDIRECT =            cHTTP_MOVED_TEMPORARILY
let cUSE_LOCAL_COPY =      cHTTP_NOT_MODIFIED
let cBAD_REQUEST =         cHTTP_BAD_REQUEST
let cAUTH_REQUIRED =       cHTTP_UNAUTHORIZED
let cFORBIDDEN =           cHTTP_FORBIDDEN
let cNOT_FOUND =           cHTTP_NOT_FOUND
let cMETHOD_NOT_ALLOWED =  cHTTP_METHOD_NOT_ALLOWED
let cNOT_ACCEPTABLE =      cHTTP_NOT_ACCEPTABLE
let cLENGTH_REQUIRED =     cHTTP_LENGTH_REQUIRED
let cPRECONDITION_FAILED = cHTTP_PRECONDITION_FAILED
let cSERVER_ERROR =        cHTTP_INTERNAL_SERVER_ERROR
let cNOT_IMPLEMENTED =     cHTTP_NOT_IMPLEMENTED
let cBAD_GATEWAY =         cHTTP_BAD_GATEWAY
let cVARIANT_ALSO_VARIES = cHTTP_VARIANT_ALSO_VARIES

let is_http_info s =
  s >= 100 && s < 200
let is_http_success s =
  s >= 200 && s < 300
let is_http_redirect s =
  s >= 300 && s < 400
let is_http_error s =
  s >= 400 && s < 600
let is_http_client_error s =
  s >= 400 && s < 500
let is_http_server_error s =
  s >= 500 && s < 600

module Table = struct			(* Table functions. *)
  type t
  external get : t -> string -> string
      = "mod_caml_table_get"
  external set : t -> string -> string -> unit
      = "mod_caml_table_set"
  external add : t -> string -> string -> unit
      = "mod_caml_table_add"
(*
  Non-copying version. Not a great idea to allow access to this.
  external setn : t -> string -> string -> unit
      = "mod_caml_table_setn"
*)

  external unset : t -> string -> unit
      = "mod_caml_table_unset"

      (* ... etc ... *)
end

module Server = struct			(* Server_rec functions. *)
  type t				(* Actual server_rec structure. *)
  external hostname : t -> string
      = "mod_caml_server_hostname"

      (* ... etc ... *)
end

module Connection = struct		(* Conn_rec functions. *)
  type t				(* Actual conn_rec structure. *)

  external remote_ip : t -> string
      = "mod_caml_connection_remote_ip"

      (* ... etc ... *)
end

module Request = struct			(* Request_rec functions. *)
  type t				(* Actual request_rec structure. *)

  type read_policy = REQUEST_NO_BODY
                   | REQUEST_CHUNKED_ERROR
		   | REQUEST_CHUNKED_DECHUNK
		   | REQUEST_CHUNKED_PASS

  external connection : t -> Connection.t
      = "mod_caml_request_connection"
  external server : t -> Server.t
      = "mod_caml_request_server"
  external next : t -> t
      = "mod_caml_request_next"
  external prev : t -> t
      = "mod_caml_request_prev"
  external main : t -> t
      = "mod_caml_request_main"
  external the_request : t -> string
      = "mod_caml_request_the_request"
  external assbackwards : t -> bool
      = "mod_caml_request_assbackwards"

  external header_only : t -> bool
      = "mod_caml_request_header_only"
  external protocol : t -> string
      = "mod_caml_request_protocol"
  external proto_num : t -> int
      = "mod_caml_request_proto_num"
  external hostname : t -> string
      = "mod_caml_request_hostname"
  external request_time : t -> float
      = "mod_caml_request_request_time"
  external status_line : t -> string
      = "mod_caml_request_status_line"
  external set_status_line : t -> string -> unit
      = "mod_caml_request_set_status_line"
  external status : t -> int
      = "mod_caml_request_status"
  external set_status : t -> int -> unit
      = "mod_caml_request_set_status"
  external method_name : t -> string
      = "mod_caml_request_method"
  external method_number : t -> method_type
      = "mod_caml_request_method_number"

  external headers_in : t -> Table.t
      = "mod_caml_request_headers_in"
  external headers_out : t -> Table.t
      = "mod_caml_request_headers_out"
  external err_headers_out : t -> Table.t
      = "mod_caml_request_err_headers_out"
  external subprocess_env : t -> Table.t
      = "mod_caml_request_subprocess_env"
  external notes : t -> Table.t
      = "mod_caml_request_notes"
  external content_type : t -> string
      = "mod_caml_request_content_type"
  external set_content_type : t -> string -> unit
      = "mod_caml_request_set_content_type"

  external user : t -> string
      = "mod_caml_request_user"

  external uri : t -> string
      = "mod_caml_request_uri"
  external set_uri : t -> string -> unit
      = "mod_caml_request_set_uri"
  external filename : t -> string
      = "mod_caml_request_filename"
  external set_filename : t -> string -> unit
      = "mod_caml_request_set_filename"
  external path_info : t -> string
      = "mod_caml_request_path_info"
  external set_path_info : t -> string -> unit
      = "mod_caml_request_set_path_info"
  external args : t -> string
      = "mod_caml_request_args"
  external set_args : t -> string -> unit
      = "mod_caml_request_set_args"
  external finfo : t -> Unix.stats option
      = "mod_caml_request_finfo"

  external send_http_header : t -> unit
      = "mod_caml_request_send_http_header"

  external setup_client_block : t -> read_policy -> unit
      = "mod_caml_request_setup_client_block"
  external should_client_block : t -> bool
      = "mod_caml_request_should_client_block"
  external get_client_block : t -> string
      = "mod_caml_request_get_client_block"
  external discard_request_body : t -> unit
      = "mod_caml_request_discard_request_body"

  external note_auth_failure : t -> unit
      = "mod_caml_request_note_auth_failure"
  external note_basic_auth_failure : t -> unit
      = "mod_caml_request_note_basic_auth_failure"
  external note_digest_auth_failure : t -> unit
      = "mod_caml_request_note_digest_auth_failure"
  external get_basic_auth_pw : t -> string option
      = "mod_caml_request_get_basic_auth_pw"

  external internal_redirect : string -> t -> unit
      = "mod_caml_request_internal_redirect"
  external internal_redirect_handler : string -> t -> unit
      = "mod_caml_request_internal_redirect_handler"

  external print_char : t -> char -> unit
      = "mod_caml_request_print_char"
  external print_string : t -> string -> unit
      = "mod_caml_request_print_string"
  let print_int r i =
    print_string r (string_of_int i)
  let print_float r f =
    print_string r (string_of_float f)
  let print_newline r =
    print_string r "\n\r"
  let print_endline r s =
    print_string r s;
    print_newline r

      (* ... etc ... *)

  external register_cleanup : t -> (unit -> unit) -> unit
      = "mod_caml_request_register_cleanup"
end

type handler_t = Request.t -> result_type

(* Override normal printing and reading functions to read/write from the
 * client connection. Each of these functions takes an additional Request.t
 * parameter, so we're not likely to get them confused with the functions in
 * Pervasives.
 *)
let print_char = Request.print_char
let print_string = Request.print_string
let print_int = Request.print_int
let print_float = Request.print_float
let print_newline = Request.print_newline
let print_endline = Request.print_endline

(* This is the "functorized" Dbi pooling code.  See apache.mli for an example
 * of how to use this.
 *)

module type DbiDriverT = sig
  type connection
  val connect : ?host:string -> ?port:string ->
    ?user:string -> ?password:string -> string ->
    connection
  val close : connection -> unit
  val closed : connection -> bool
  val commit : connection -> unit
  val ping : connection -> bool
  val rollback : connection -> unit
end

module type DbiPoolT = sig
  type connection
  val get : Request.t -> ?host:string -> ?port:string ->
    ?user:string -> ?password:string -> string -> connection
end

module DbiPool (Dbi_driver : DbiDriverT) = struct

  type connection = Dbi_driver.connection

  (* List of pools. The key is the unique combination of host/port/etc. and
   * the value is a list of unused connections for that pool.
   *
   * This code ought to work even for a multi-threaded Apache server.
   *)
  let pools = Hashtbl.create 8

  let key ?host ?port ?user ?password database_name =
    host, port, user, password, database_name

  let get r ?host ?port ?user ?password database_name =
    let key = key ?host ?port ?user ?password database_name in

    (* Get the pool (a connection list). *)
    let dbh_list = try Hashtbl.find pools key with Not_found -> [] in

    (* Search for an unused connection. We actually iterate over the pool
     * testing the handles (in case they have timed out or something).
     *)
    let rec loop = function
	[] ->
	  (* No handles left. Need to create a new connection. *)
	  let dbh =
	    Dbi_driver.connect ?host ?port ?user ?password database_name in
	  dbh, []
      | dbh :: dbhs ->
	  (* Test if dbh is a working handle. If so, return it. *)
	  if Dbi_driver.ping dbh then
	    dbh, dbhs
	  else (
	    Dbi_driver.close dbh;
	    loop dbhs
	  )
    in
    let dbh, remainder = loop dbh_list in

    (* Update the pool. *)
    Hashtbl.replace pools key remainder;

    (* Register a callback so that we return this handle to the pool
     * when the request finishes.
     *)
    Request.register_cleanup r
      (fun () ->
	 if not (Dbi_driver.closed dbh) then (
	   Dbi_driver.rollback dbh;
	   let dbh_list = Hashtbl.find pools key in
	   Hashtbl.replace pools key (dbh_list @ [dbh])
	 ));

    dbh

end

(*----------------------------------------------------------------------*)

(* Unless we actually reference the external C functions, OCaml doesn't
 * load them into the primitive table and we won't be able to access them.
 * Duh!
 *)
let _table_get = Table.get
let _table_set = Table.set
let _table_add = Table.add
let _table_unset = Table.unset
let _server_hostname = Server.hostname
let _connection_remote_ip = Connection.remote_ip
let _request_connection = Request.connection
let _request_server = Request.server
let _request_next = Request.next
let _request_prev = Request.prev
let _request_main = Request.main
let _request_the_request = Request.the_request
let _request_assbackwards = Request.assbackwards
let _request_header_only = Request.header_only
let _request_protocol = Request.protocol
let _request_proto_num = Request.proto_num
let _request_hostname = Request.hostname
let _request_request_time = Request.request_time
let _request_status_line = Request.status_line
let _request_set_status_line = Request.set_status_line
let _request_status = Request.status
let _request_set_status = Request.set_status
let _request_method_name = Request.method_name
let _request_method_number = Request.method_number
let _request_headers_in = Request.headers_in
let _request_headers_out = Request.headers_out
let _request_err_headers_out = Request.err_headers_out
let _request_subprocess_env = Request.subprocess_env
let _request_notes = Request.notes
let _request_content_type = Request.content_type
let _request_set_content_type = Request.set_content_type
let _request_user = Request.user
let _request_uri = Request.uri
let _request_set_uri = Request.set_uri
let _request_filename = Request.filename
let _request_set_filename = Request.set_filename
let _request_path_info = Request.path_info
let _request_set_path_info = Request.set_path_info
let _request_args = Request.args
let _request_set_args = Request.set_args
let _request_finfo = Request.finfo
let _request_send_http_header = Request.send_http_header
let _request_setup_client_block = Request.setup_client_block
let _request_should_client_block = Request.should_client_block
let _request_get_client_block = Request.get_client_block
let _request_discard_request_body = Request.discard_request_body
let _request_note_auth_failure = Request.note_auth_failure
let _request_note_basic_auth_failure = Request.note_basic_auth_failure
let _request_note_digest_auth_failure = Request.note_digest_auth_failure
let _request_get_basic_auth_pw = Request.get_basic_auth_pw
let _request_internal_redirect = Request.internal_redirect
let _request_internal_redirect_handler = Request.internal_redirect_handler
let _request_register_cleanup = Request.register_cleanup

;;

Callback.register_exception "mod_caml_http_error" (HttpError 0)
