(**
  * Library for writing CGIs
  *
  * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
  *
  * Copyright (C) 1997 Daniel de Rauglaudre, INRIA
  *
  * Copyright (C) 1998 Jean-Christophe FILLIATRE
  * 
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Library General Public
  * License as published by the Free Software Foundation; either
  * version 2 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  * Library General Public License for more details.
  *
  * You should have received a copy of the GNU Library General Public
  * License along with this library; if not, write to the Free
  * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  *
  * $Id: cgi.mli,v 1.10 2004/03/05 13:11:01 rwmj Exp $
  *)

open Apache

module Cookie :
  sig
    class cookie :
      name:string -> value:string -> expires:string ->
      domain:string -> path:string -> secure:bool ->
    object
      method name : string
	(** Return the name of the cookie. *)
      method value : string
	(** Return the value of the cookie. *)
      method expires : string
	(** Return the expires string of the cookie, or "" if not set. *)
      method domain : string
	(** Return the domain of the cookie, or "" if not set. *)
      method path : string
	(** Return the path of the cookie, or "" if not set. *)
      method secure : bool
	(** Return true if the cookie is a secure cookie. *)
      method set_name : string -> unit
	(** Set the name of the cookie. *)
      method set_value : string -> unit
	(** Set the value of the cookie. *)
      method set_expires : string -> unit
	(** Set the expires string of the cookie. *)
      method set_domain : string -> unit
	(** Set the domain of the cookie. *)
      method set_path : string -> unit
	(** Set the path of the cookie. *)
      method set_secure : bool -> unit
	(** Set if the cookie is a secure cookie. *)
      method as_string : string
	(** Return the string representation of the cookie. *)
    end

    val cookie : name:string -> value:string -> ?expires:string ->
      ?domain:string -> ?path:string -> ?secure:bool -> unit -> cookie
    (** Create a cookie. *)

    val parse : string -> cookie list
    (** Parse zero or more cookies, normally from a Cookie: header. *)

  end

val set_post_max : int -> unit
(** Set the maximum size for POSTed requests. This is a security feature to
 * prevent clients from overrunning the server with data. The default is
 * [max_int], meaning no limit.
 *)

val post_max : unit -> int
(** Get the maximum size for POSTed requests. *)

type upload_data = {
  upload_value: string;
  upload_filename: string;
  upload_content_type: string
}
(** Type returned by [cgi#upload] method. *)

class cgi :
  Request.t ->
  object
    method header : ?content_type:string -> ?cookie:Cookie.cookie ->
      ?cookies:Cookie.cookie list -> unit -> unit
    (** Emit the header. The default content type is "text/html". *)

    method template : ?content_type:string -> ?cookie:Cookie.cookie ->
      ?cookies:Cookie.cookie list -> Template.template -> unit
    (** Emit the header followed by the given template.  This is a short cut
      * for: [q#header (); print_string r template#to_string;]
      *)

    method redirect : ?cookie:Cookie.cookie -> ?cookies:Cookie.cookie list ->
      string -> unit
    (** Emit a redirect header for the given URL. *)

    method url : unit -> string
    (** Return the URL of the script. *)

    method param : string -> string
    (** Return the value of a parameter, or throw [Not_found] *)

    method param_all : string -> string list
    (** Return all the values of a parameter. *)

    method param_exists : string -> bool
    (** Return true iff the named parameter exists. *)

    method param_true : string -> bool
    (** This method returns false if the named parameter is missing, is an
      empty string, or is the string ["0"]. Otherwise it returns true. Thus
      the intent of this is to return true in the Perl sense of the word.
      If a parameter appears multiple times, then this uses the first
      definition and ignores the others. *)

    method params : (string * string) list
    (** Return an assoc-list mapping name -> value for all parameters.
      Note that CGI scripts may have multiple values for a single name. *)

    method is_multipart : bool
      (** Returns true iff the request was a [multipart/form-data] [POST]
	* request.  Such requests are used when you need to upload files
	* to a CGI script.
        *)

    method upload : string -> upload_data
      (** For multipart forms only.  Returns the full upload data passed
	* by the browser for a particular parameter.  Throws [Not_found]
	* is no such parameter exists.
        *)

    method upload_all : string -> upload_data list
      (** For multipart forms only.  As for [#upload], but returns all
	* uploads.
        *)

    method cookie : string -> Cookie.cookie
    (** Return the named cookie, or throw [Not_found]. *)

    method cookies : Cookie.cookie list
    (** Return a list of all cookies passed to the script. *)

    method request : Request.t
    (** Returns the original request object (pass in the constructor). *)
  end

module StdPages :
  sig
    type button = {
      label : string;
      link : string;
      method_ : string option;
      params : (string * string) list;
    }

    val dialog : ?cookie:Cookie.cookie ->
      ?cookies:Cookie.cookie list ->
      ?template:Template.template ->
      ?title:string ->
      ?icon:string ->
      ?icon_alt:string ->
      ?back_button:bool ->
      ?close_button:bool ->
      ?buttons:button list ->
      cgi -> string -> unit
      (** [StdPages.dialog] is used to generate a standard dialog box,
	* which can be variously used for asking questions, displaying
	* error messages, presenting confirmation interstitial pages,
	* etc. (Note that there are specialised versions of this function
	* for OK messages and error pages. See {!Cgi.StdPages.ok} and
	* {!Cgi.StdPages.error} for simple functions for these tasks.)
	*
	* A dialog looks like this (in left-to-right locales such as
	* US english):
	*
	* {v
	* +--------------------------------------------------+
	* |                                                  |
	* |             TITLE .............................  |
        * | +--------+                                       |
        * | | ICON   |                                       |
        * | |        |  MESSAGE                              |
        * | |        |                                       |
        * | +--------+                                       |
	* |            +--------+ +--------+ +--------+      |
	* |            | BUTTON | | BUTTON | | BUTTON |      |
	* |            +--------+ +--------+ +--------+      |
	* |                                                  |
	* +--------------------------------------------------+
	* v}
	*
	* Only the message element is required. All other elements are
	* optional (even the buttons, although it won't be a very usable
	* dialog if there are no buttons at all).
	*
	* The optional parameters control which elements are displayed:
	*
	* [?title] Sets the title element.
	*
	* [?icon] Sets the path to the icon element. [?icon_alt] sets the
	* ALT text.
	*
	* [?back_button] Controls whether a Javascript back button is
	* displayed (default: [true]).
	*
	* [?close_button] Controls whether a Javascript close window button
	* is displayed (default: [false]).
	*
	* [?buttons] is a list of buttons to display. (Each button has type
	* {!Cgi.StdPages.button}).
	*
	* [?template] can be used to override the template used. The [Cgi]
	* library comes with a suitable default template, but you can
	* override it if you want one which is more suitable for the style
	* of your website.
        *)

    val error : ?cookie:Cookie.cookie -> ?cookies:Cookie.cookie list ->
      ?template:Template.template ->
      ?title:string ->
      ?icon:string ->
      ?icon_alt:string ->
      ?back_button:bool ->
      ?close_button:bool ->
      cgi -> string -> unit
      (** This is a limited form of [StdPages.dialog] which displays a
	* simple error page.
        *)

    val ok : ?cookie:Cookie.cookie -> ?cookies:Cookie.cookie list ->
      ?template:Template.template ->
      ?title:string ->
      ?icon:string ->
      ?icon_alt:string ->
      ?back_button:bool ->
      ?close_button:bool ->
      ?buttons:button list ->
      cgi -> string -> unit
      (** This is a limited form of [StdPages.dialog] which displays a
	* simple ok/success page.
        *)

  end

module Sendmail :
  sig

    exception Sendmail_failure of string
      (** This exception may be thrown by any of the functions in this
	* library, to indicate some problem running sendmail.
        *)

    val sendmail : string ref
      (** This contains the path to sendmail (or the binary which acts
	* like sendmail, as in the case of exim).  Normally this is set
	* to the correct value for your OS, eg. ["/usr/sbin/sendmail"].
        *)
    val sendmail_args : string ref
      (** This contains the arguments passed to sendmail, normally
	* ["-t -i"].  You could add the ["-f"] option to specify the
	* sender email address, provided that the current user is one
	* of sendmail's trusted users.
        *)

    val send : unit -> out_channel
      (** Begin sending an email.  This returns a channel on which you
	* should write the mail headers, followed by a blank line, followed
	* by the message body.  After writing the mail you must call
	* {!Cgi.Sendmail.close}.
	*
	* [send] does not perform any sort of processing or escaping on
	* the message.
        *)

    val close : out_channel -> unit
      (** Close the output channel.  You must call this on the channel
	* returned from {!Cgi.Sendmail.send}.
        *)

    val send_mail : ?subject:string ->
      ?to_addr:string list -> ?cc:string list -> ?bcc:string list ->
      ?from:string -> ?content_type:string ->
      ?headers:(string * string) list ->
      body:string -> unit -> unit
      (** This is a less flexible, but simpler interface to sending mail.
	* You specify, optionally, Subject line, To, Cc, Bcc, From
	* and Content-Type headers, and a body, and a mail is generated.
	*
	* [send_mail] will correctly escape headers, provided they
	* are in strict 7 bit US-ASCII.  Its behaviour on non-7 bit
	* sequences is currently undefined (and probably wrong).
	* [send_mail] will not process or escape any part of the body.
        *)

  end

val random_sessionid : unit -> string
  (** Generates a 128 bit (32 hex digits) random string which can be used
    * for session IDs, random cookies and so on.  The string returned will
    * be very random and hard to predict, if the OS supports this feature.
    *)
