(* mod_caml module for executing CGI scripts written in OCaml.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: registry.ml,v 1.3 2004/01/24 18:02:13 rwmj Exp $
 *
 * CGI scripts are loaded dynamically into memory, and stay there unless
 * you restart the server or the file changes on disk.
 *
 * To use:
 *
 * CamlLoad /usr/lib/ocaml/3.06/apache/registry.cmo
 * Alias /caml-bin/ /path/to/your/scripts/
 * <Location /caml-bin>
 *   SetHandler ocaml-bytecode
 *   CamlHandler Registry.handler
 *   Options ExecCGI
 *   Allow from all
 * </Location>
 *
 * This is called the "registry" in honour of mod_perl's Apache::Registry
 * which does much the same thing for Perl programs.
 *)

open Unix
open Apache
open Mod_caml

(* Table of scripts we have already loaded. Maps
 * filename -> (handler, mtime) so we can reload the file if the mtime has
 * changed.
 *)
let registry = Hashtbl.create 32

let reg_hand = ref None

(* Callback from the loaded script. *)
let register_script handler =
  if !reg_hand <> None then
    failwith ("Registry.register_script should only be called once " ^
	      "during script initialization");
  reg_hand := Some handler

(* Load a file. *)
let load_file filename =
  reg_hand := None;
  (try
    Dynlink.loadfile_private filename;
  with
    Dynlink.Error (e) ->
      failwith (Dynlink.error_message e));
  match !reg_hand with
    None -> failwith (filename ^ ": forgot to call Registry.register_script!")
  | Some handler -> handler

(* Handler for requests. *)
let handler r =
  let filename = Request.filename r in
  let finfo =
    match Request.finfo r with
	None -> failwith (filename ^ ": no request->finfo: probably means " ^
			  "the file does not exist.")
      | Some finfo -> finfo
  in
  let mtime = finfo.st_mtime in

  (* Get the handler for this file. *)
  let handler =
    try
      let (handler, old_mtime) = Hashtbl.find registry filename in
      if old_mtime < mtime then (
        (* Reload the file. *)
	let handler = load_file filename in
	Hashtbl.replace registry filename (handler, mtime);
	handler
      ) else
	handler
    with
      Not_found ->
	(* Load the file. *)
	let handler = load_file filename in
	Hashtbl.add registry filename (handler, mtime);
	handler in

  (* Run the handler. XXX Handle exceptions here or pass them up? *)
  let () = handler r in

  (* Return OK (actually the return value is ignored, I think). *)
  OK

let () =
  register_handler handler "handler"
