(* Comments system.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: comments.ml,v 1.7 2004/08/03 16:54:31 rwmj Exp $
 *
 * See examples/comments/README
 *)

open Apache
open Registry
open Cgi

module Pool = DbiPool (Dbi_postgres)

let (//) = Filename.concat

(* Wrapper around the Cgi.Template.template function which loads the
 * template from a pre-defined path and sets up some default variables.
 *)
let get_template filename =
  let path =
    try Sys.getenv "COMMENTS_TEMPLATESDIR"
    with Not_found -> "/usr/lib/mod_caml/templates" in

  let template =
    Template.template (path // "comments" // filename) in

  (* Set some default variables shared by all templates. *)
  template#set "bindir" "/caml-bin";

  (* Return the template. *)
  template

(* mod_caml doesn't have any explicit support for logins / session cookies.
 * However it is simple to implement this on top of cookies, as shown below.
 * CGI scripts in the comments system register themselves here instead of
 * calling [Registry.register_script] directly.
 *)
let register_script run =
  (* Actually register the script with the [Registry] module. *)
  register_script
    (fun r ->
       (* This is the function which gets wrapped around the normal "run"
	* function. *)
       let q = new cgi r in
       (* Get a database connection from the pool. *)
       let dbh = Pool.get r "comments" in
       (* Get the userid from the cookie. *)
       let userid =
         try
           let cookie = q#cookie "comments_auth" in
           let sessionid = cookie#value in
           let sth = dbh#prepare_cached "select userid from comments_sessions
                                          where sessionid = ?" in
           sth#execute [`String sessionid];
	   sth#fetch1int ()
         with
             (* Cookie not found OR session expired. *)
             Not_found -> 0
       in
       (* Call the actual CGI script. *)
       run r q dbh userid
    )

(* Produce a printable representation of a Dbi.interval type. *)
let printable_interval (date, time) =
  let years, months, days, hours, mins =
    date.Dbi.year, date.Dbi.month, date.Dbi.day,
    time.Dbi.hour, time.Dbi.min in
  let plural str = function
      0 -> ""
    | 1 -> "1 " ^ str ^ " "
    | n -> (string_of_int n) ^ " " ^ str ^ "s "
  in
  if years = 0 && months = 0 && days = 0 && hours = 0 && mins = 0
  then
    "under 1 minute"
  else
    String.concat ""
      [ plural "year" years;
	plural "month" months;
	plural "day" days;
	plural "hour" hours;
	plural "minute" mins ]

(* "Trim" a string (remove leading and trailing whitespace characters. *)
let string_trim =
  let re_before = Pcre.regexp "^\\s+" in
  let re_after = Pcre.regexp "\\s+$" in
  let itempl = Pcre.subst "" in
  fun str ->
    Pcre.replace ~rex:re_before ~itempl str;
    Pcre.replace ~rex:re_after ~itempl str
