(*
 * Library for writing templates.
 * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: template.ml,v 1.2 2004/04/16 15:07:45 rwmj Exp $
 *)

open Cgi_escape

(* Find content in str, starting at index i. *)
let string_find_from str i content =
  let first = content.[0] in
  let len = String.length str in
  let clen = String.length content in
  let rec loop i =
    let r = try String.index_from str i first with Not_found -> -1 in
    if r >= i && r <= len-clen then (
      let sub = String.sub str r clen in
      if sub = content then
	r				(* Found at index r. *)
      else
	loop (r+1)			(* Start looking again at r+1. *)
    ) else
      -1				(* Not found. *)
  in
  loop i

(* Find content in str, starting at beginning. *)
let string_find str content =
  string_find_from str 0 content

(* Return two strings by splitting str at character offset i. *)
let string_partition str i =
  String.sub str 0 i, String.sub str i ((String.length str) - i)

(* Return true if the string starts with the prefix. *)
let string_starts_with s pref =
  String.length s >= String.length pref &&
  String.sub s 0 (String.length pref) = pref

let read_whole_chan chan =
  let buf = Buffer.create 4096 in
  let rec loop () =
    let line = input_line chan in
    Buffer.add_string buf line;
    Buffer.add_char buf '\n';
    loop ()
  in
  try
    loop ()
  with
      End_of_file -> Buffer.contents buf

(* This is the type used to store the compiled template. *)
type node_t = Plain of string	(* Just some text. *)
              | Tag of string		(* ::tag:: *)
	      | If of string * compiled_t * compiled_t
					(* ::if(..):: .. ::else:: .. ::end:: *)
	      | Table of string * compiled_t
                                        (* ::table(..):: .. ::end:: *)
	      | Call of string * string list
					(* ::call(fname,arg,arg...):: *)
and compiled_t = node_t list

let tag_re = Pcre.regexp "^::(\\w+)::$"
let if_re = Pcre.regexp "^::if\\((.*)\\)::$"
let table_re = Pcre.regexp "^::table\\((.*)\\)::$"
let call_re = Pcre.regexp "^::call\\((.*)\\)::$"
let escape_url_re = Pcre.regexp "^(.*)_url$"
let escape_html_re = Pcre.regexp "^(.*)_html$"
let escape_html_tag_re = Pcre.regexp "^(.*)_html_tag$"
let escape_html_textarea_re = Pcre.regexp "^(.*)_html_textarea$"
let comma_split_re = Pcre.regexp ","

(* Return Some $1 if string matches regular expression. *)
let matches str rex =
  try
    let subs = Pcre.exec ~rex str in
    Some (Pcre.get_substring subs 1)
  with
      Not_found -> None

let compile_template source =
  (* This function splits up source into a list like this:
   * source = "some text ::tag:: more text ::end::" would become:
   * [ "some text "; "::tag::"; " more text "; "::end::" ]
   *)
  let split_up source =
    let rec loop source list =
      let i = string_find source "::" in
      match i with
	  -1 -> (* No "::" in the whole string. *)
	    source :: list
	| 0 ->  (* String starts with "::". *)
	    let i = string_find_from source 2 "::" in
	    if i > 2 then (
	      let tag, remainder = string_partition source (i+2) in
	      loop remainder (tag :: list)
	    ) else
	      source :: list
	| i -> (* String followed by possible tag start. *)
	    let str, remainder = string_partition source i in
	    loop remainder (str :: list)
    in
    List.rev (loop source [])
  in

  (* Return Some filename if the string has the form ::include(filename):: *)
  let is_include str =
    let len = String.length str in
    if string_starts_with str "::include(" &&
      str.[len-3] = ')' && str.[len-2] = ':' && str.[len-1] = ':' then
	Some (String.sub str 10 (len - 13))
    else
      None
  in

  (* Load included files. *)
  let rec load_includes = function
      [] -> []
    | x :: xs ->
	(match is_include x with
	     None -> [x]
	   | Some filename ->
	       let chan = open_in filename in
	       let source = read_whole_chan chan in
	       close_in chan;
	       load_includes (split_up source)) @ load_includes xs
  in

  (* Convert the flat list of strings into a list of (type, string)
   * tuples. So for example "::table(foo)::" would become ("table",
   * "foo").
   *)
  let typify =
    (* Return true if string is not a ::tag::. *)
    let is_plain str =
      let len = String.length str in
      len < 2 || str.[0] != ':' || str.[1] != ':'
    in

    List.map (fun str ->
		if is_plain str then
		  ("plain", str)
		else if str = "::else::" then
		  ("else", "")
		else if str = "::end::" then
		  ("end", "")
		else match matches str tag_re with
		    Some tag ->
		      ("tag", tag)
		  | None ->
		      match matches str if_re with
			  Some cond ->
			    ("if", cond)
			| None ->
			    match matches str table_re with
				Some name ->
				  ("table", name)
			      | None ->
				  match matches str call_re with
				      Some call ->
					("call", call)
				    | None ->
					failwith ("Template: " ^
						  "unknown tag in template: " ^
						  str))
  in

  (* Combine plain text parts of the list (these might have been split
   * naturally or across include files). This is for efficiency.
   *)
  let rec combine_plains = function
      [] -> []
    | [x] -> [x]
    | ("plain", x) :: ("plain", y) :: xs ->
	combine_plains (("plain", x ^ y) :: xs)
    | x :: xs ->
	x :: combine_plains xs
  in

  (* Split up the original source and load all included files to produce
   * a big flat list of (type, string) pairs.
   *)
  let flat = combine_plains (typify (load_includes (split_up source))) in

  (* This creates the final structure - a compiled tree (ct) - from the
   * flat list of (type, string) pairs.
   *)
  let rec build_tree = function
      [] -> []
    | ("plain", text) :: xs ->
	Plain text :: build_tree xs
    | ("tag", tag) :: xs ->
	Tag tag :: build_tree xs
    | ("if", cond) :: xs ->
	let then_clause, else_clause, remainder = build_if_stmt xs in
	If (cond, then_clause, else_clause) :: build_tree remainder
    | ("table", name) :: xs ->
	let body, remainder = build_table_stmt xs in
	Table (name, body) :: build_tree remainder
    | ("call", call) :: xs ->
	let fname, args = split_call_args call in
	Call (fname, args) :: build_tree xs
    | (typ, _) :: xs ->
	failwith ("Template: unexpected tag ::" ^ typ ^ "::")
  and build_if_stmt xs =
    let rec then_part = function
	_, _, [] ->
	  failwith "Template: missing ::end:: in ::if:: statement"
      | 0, part, ("else", _) :: xs ->
	  List.rev part, xs, true
      | 0, part, ("end", _) :: xs ->
	  List.rev part, xs, false
      | lvl, part, ("if", cond) :: xs ->
	  then_part (lvl+1, (("if", cond) :: part), xs)
      | lvl, part, ("table", name) :: xs ->
	  then_part (lvl+1, (("table", name) :: part), xs)
      | lvl, part, ("end", _) :: xs ->
	  then_part (lvl-1, (("end", "") :: part), xs)
      | lvl, part, x :: xs ->
	  then_part (lvl, (x :: part), xs)
    in
    let rec else_part = function
	_, _, [] ->
	  failwith "Template: missing ::end:: in ::if:: statement"
      | 0, part, ("else", _) :: xs ->
	  failwith "Template: multiple ::else:: in ::if:: statement"
      | 0, part, ("end", _) :: xs ->
	  List.rev part, xs
      | lvl, part, ("if", cond) :: xs ->
	  else_part (lvl+1, (("if", cond) :: part), xs)
      | lvl, part, ("table", name) :: xs ->
	  else_part (lvl+1, (("table", name) :: part), xs)
      | lvl, part, ("end", _) :: xs ->
	  else_part (lvl-1, (("end", "") :: part), xs)
      | lvl, part, x :: xs ->
	  else_part (lvl, (x :: part), xs)
    in
    let then_part, remainder, has_else_clause = then_part (0, [], xs) in
    let then_clause = build_tree then_part in
    let else_part, remainder =
      if has_else_clause then else_part (0, [], remainder)
      else [], remainder in
    let else_clause = build_tree else_part in
    then_clause, else_clause, remainder
  and build_table_stmt xs =
    let rec body_part = function
	_, _, [] ->
	  failwith "Template: missing ::end:: in ::table:: statement"
      | 0, part, ("end", _) :: xs ->
	  List.rev part, xs
      | lvl, part, ("if", cond) :: xs ->
	  body_part (lvl+1, (("if", cond) :: part), xs)
      | lvl, part, ("table", name) :: xs ->
	  body_part (lvl+1, (("table", name) :: part), xs)
      | lvl, part, ("end", _) :: xs ->
	  body_part (lvl-1, (("end", "") :: part), xs)
      | lvl, part, x :: xs ->
	  body_part (lvl, (x :: part), xs)
    in
    let body_part, remainder = body_part (0, [], xs) in
    let body_clause = build_tree body_part in
    body_clause, remainder
  and split_call_args call =
    (* Split string on commas. *)
    let args = Pcre.split ~rex:comma_split_re call in
    List.hd args, List.tl args
  in

  (* Build the tree from the flat list. *)
  build_tree flat

(* Type of variables, either a simple ::tag:: or a set of row definitions
 * in a table.
 *)
type table_row_t = (string * var_t) list
and var_t = VarString of string
            | VarTable of table_row_t list
	    | VarConditional of bool
	    | VarCallback of (string list -> string)

(* Convert compiled template to string, substituting set variables and
 * tables.
 *)
let ct_to_buffer buffer vars compiled =
  let rec loop ctx compiled =
    List.iter (function
		   Plain text ->
		     Buffer.add_string buffer text
		 | Tag name ->
		     Buffer.add_string buffer (resolve_variable ctx name)
		 | If (cond, then_clause, else_clause) ->
		     if eval_condition ctx cond then
		       loop ctx then_clause
		     else
		       loop ctx else_clause
		 | Table (name, body) ->
		     let rows = resolve_table ctx name in
		     List.iter (fun new_ctx ->
				  let ctx = new_ctx :: ctx in
				  loop ctx body
			       ) rows;
		 | Call (fname, args) ->
		     let f = resolve_callback ctx fname in
		     let result = f args in
		     Buffer.add_string buffer result
	      ) compiled
  and resolve_variable ctx name =
    let name, escaping = get_escaping_from_name name in
    let value = resolve ctx name in
    match value with
	VarString str -> escape escaping str
      | _ ->
	  failwith ("Template: ::" ^ name ^ ":: should be a "^
		    "simple string tag.")
  and resolve_table ctx name =
    let value = resolve ctx name in
    match value with
	VarTable tbl -> tbl
      | _ ->
	  failwith ("Template: ::table(" ^ name ^ "):: should be a "^
		    "table tag.")
  and resolve_conditional ctx name =
    let value = resolve ctx name in
    match value with
	VarConditional b -> b
      | _ ->
	  failwith ("Template: ::if(" ^ name ^ "):: should be a "^
		    "conditional tag.")
  and resolve_callback ctx name =
    let value = resolve ctx name in
    match value with
	VarCallback f -> f
      | _ ->
	  failwith ("Template: ::call(" ^ name ^ "[,...]):: should be a "^
		    "callback function.")
  and resolve ctx name =
    try
      let assoc = List.find (fun assoc -> List.mem_assoc name assoc) ctx in
      List.assoc name assoc
    with
	Not_found ->
	  failwith ("Template: tag/table ::" ^ name ^ ":: was not " ^
		    "assigned any value.");
  and eval_condition ctx cond =
    (* This is only a very simple implementation. We can extend it with
     * operators, etc. later.
     *)
    try
      resolve_conditional ctx cond
    with
	Not_found ->
	  failwith ("Template: conditional " ^ cond ^ " was not "^
		    "assigned any value.");
  and get_escaping_from_name name =
    (* If name has the form "name_html" (or one of the similar cases),
     * return bare name and the form of the escaping that needs to be
     * applied.
     *)
    match matches name escape_url_re with
	Some name -> name, EscapeUrl
      | None ->
	  match matches name escape_html_re with
	      Some name -> name, EscapeHtml
	    | None ->
		match matches name escape_html_tag_re with
		    Some name -> name, EscapeHtmlTag
		  | None ->
		      match matches name escape_html_textarea_re with
			  Some name -> name, EscapeHtmlTextarea
			| None -> name, EscapeNone
  in
  (* Convert vars hash table to assoc list. *)
  let vars =
    Hashtbl.fold (fun key value xs -> (key, value) :: xs) vars [] in

  let initial_ctx = [vars] in
  loop initial_ctx compiled

class template source =
  (* This will store the values of set variables/tables. The mutable
   * properties of the hash table make this more useful inside the
   * class. When we print the template we convert this top-level
   * hashtbl into an assoc list for compatibility with internal
   * contexts.
   *)
  let vars = Hashtbl.create 16 in

  (* Compile the source. *)
  let compiled = compile_template source in

object (self)
  method set name value =
    Hashtbl.replace vars name (VarString value)

  method table name tbl =
    Hashtbl.replace vars name (VarTable tbl)

  method conditional name cond =
    Hashtbl.replace vars name (VarConditional cond)

  method callback name f =
    Hashtbl.replace vars name (VarCallback f)

  method to_string =
    let buffer = Buffer.create 4096 in
    ct_to_buffer buffer vars compiled;
    Buffer.contents buffer

  method to_channel chan =
    let buffer = Buffer.create 4096 in
    ct_to_buffer buffer vars compiled;
    Buffer.output_buffer chan buffer

  method source =
    source

end

let template_from_string source =
  new template source

let template_from_channel chan =
  let source = read_whole_chan chan in
  template_from_string source

let template filename =
  let chan = open_in filename in
  let template = template_from_channel chan in
  close_in chan;
  template
