(* Comments system.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: show_comments.ml,v 1.14 2004/03/02 16:21:44 rwmj Exp $
 *
 * See examples/comments/README
 *)

open Apache
open Cgi
open Comments

(* Load and compile the template. *)
let template = get_template "show_comments.html"

let hostname_re = Pcre.regexp ".*\\.([^.]*\\.(com|net|org))$"
let hostname_subst = Pcre.subst "$1"

let run r (q : cgi) (dbh : Dbi.connection) _ =
  (* If we were correctly embedded into a SSI page using something like
   * <!--#include virtual="/caml-bin/show_comments.cmo" --> then mod_include
   * is running this script as a subrequest. We navigate through r->main
   * to get back to the main request, and thence to the original request URI
   * (so we know the original page, of course). We may have to go "up"
   * through several levels of subrequest.
   *)
  let hostname, uri =
    let rec get_uri r =
      try
	let r = Request.main r in
	get_uri r
      with
	  Not_found -> Request.hostname r, Request.uri r
    in
    get_uri r
  in

  (* XXX The following is a simple heuristic which works for me, but
   * may not work for you.  If someone goes to annexia.org vs.
   * www.annexia.org then we want them to see the same comments.  To this
   * end, try to ensure that the hostname is the shortest possible
   * distinguishing hostname for this organization.
   *)
  let hostname = String.lowercase
		   (Pcre.replace
		      ~rex:hostname_re ~itempl:hostname_subst hostname) in

  (* Add the supposed URI to the template for debugging purposes. *)
  template#set "hostname" hostname;
  template#set "uri" uri;

  let sth = dbh#prepare_cached "select id from comments_pages
                                 where hostname = ? and uri = ?" in
  sth#execute [`String hostname; `String uri];

  (* Does this URI exist yet? If not, we'll have to create it. *)
  let id =
    try
      sth#fetch1int ()
    with
	Not_found ->
	  let sth =
	    dbh#prepare_cached "insert into comments_pages (hostname, uri)
                                values (?, ?)" in
	  sth#execute [`String hostname; `String uri];
	  let id = sth#serial "comments_pages_id_seq" in
	  dbh#commit ();
	  id
  in

  (* Set the ID. *)
  template#set "id" (string_of_int id);

  (* Get the comments. *)
  let sth = dbh#prepare_cached
	      "select c.body,
                      current_timestamp - c.posted_date,
                      u.name
                 from comments as c
                      left outer join comments_users as u on c.author = u.id
                where c.page_id = ?
             order by 2 desc" in
  sth#execute [`Int id];

  let comments =
    sth#map (function
		 [`String body; `Interval iv; (`Null | `String _) as name] ->
		   let name =
		     match name with
			 `Null -> "Anonymous User"
		       | `String name -> name in
		   let interval = printable_interval iv in
		   [ "name", Template.VarString name;
		     "interval", Template.VarString interval;
		     "body", Template.VarString body ]
	       | _ -> assert false) in

  template#table "comments" comments;

  (* Display the page. *)
  q#template template

let () =
  register_script run
