(* Add users to a mailing list.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: maillist.ml,v 1.6 2004/09/11 16:31:44 rwmj Exp $
 *
 * To get this working you need to locate registry.cmo and create a caml-bin
 * directory containing maillist.cmo. Then set up Apache like this:
 *
 * CamlLoad /the/path/to/registry.cmo
 * Alias /caml-bin/ /path/to/your/caml-bin/
 * <Location /caml-bin>
 *   SetHandler ocaml-bytecode
 *   CamlHandler Registry.handler
 *   Options ExecCGI
 *   Allow from all
 * </Location>
 *
 * Create a PostgreSQL database called 'maillist' and populate it using
 * 'maillist.sql'.
 *
 * Add links to your pages like this:
 *
 * <a href="/caml-bin/maillist.cmo?section=foo">Send me mail when this section
 * is updated.</a>
 *
 * Populate the database with the appropriate section(s). Users will be able
 * to add themselves to the list. You will need to mail them manually when
 * the section is actually updated.
 *)

open Apache
open Registry
open Cgi

module Pool = DbiPool (Dbi_postgres)

let (//) = Filename.concat

let path =
  try Sys.getenv "MAILLIST_TEMPLATESDIR"
  with Not_found -> "/usr/share/mod_caml/templates"

let template = Template.template (path // "maillist.html")
let body = Template.template (path // "maillist-body.txt")

type action = Form | Add | Remove | Confirm

let run r =
  let q = new cgi r in

  let action =
    try
      match q#param "action" with
	  "add" -> Add
	| "remove" -> Remove
	| "confirm" -> Confirm
	| str -> failwith ("unknown action: " ^ str)
    with
	Not_found ->
	  if q#param_exists "r" then Confirm else Form in

  let dbh = Pool.get r "maillist" in

  match action with
      Form ->
	let section = q#param "section" in

	template#set "section" section;
	q#template template

    | Add ->				(* Send confirm email. *)
	let section = q#param "section" in
	let email = q#param "email" in

	(* Use this opportunity to delete old cookies in the database. *)
	let sth =
	  dbh#prepare_cached "delete from cookies
                 where entered_time < current_timestamp - interval '1 day'" in
	sth#execute [];

	(* Make up a random number and send back to the user. *)
	let rand = random_sessionid () in

	let sth =
	  dbh#prepare_cached "insert into cookies (section, email, rand)
                                   values (?, ?, ?)" in
	sth#execute [`String section; `String email; `String rand];

	let hostname = Request.hostname r in

	body#set "rand" rand;
	body#set "url" (q#url ());
	body#set "hostname" hostname;
	let body = body#to_string in

	let subject = "[" ^ hostname ^ "] Email address confirmation" in

	Sendmail.send_mail ~to_addr:[email] ~subject ~body ();

	dbh#commit ();

	StdPages.ok q
	  ("An email was sent to your address, to confirm that " ^
	   "your address is correct.  When you get this email, " ^
	   "click on the link to finish subscribing to this mailing list.")

    | Confirm ->			(* Confirm email received. *)
	let rand = q#param "r" in

	(* In the database? *)
	let sth =
	  dbh#prepare_cached "select section, email from cookies
                               where rand = ?" in
	sth#execute [`String rand];
	(try
	   let section, email = match sth#fetch1 () with
	       [ `String section; `String email ] -> section, email
	     | _ -> assert false in

	   let sth = dbh#prepare_cached "delete from cookies where rand = ?" in
	   sth#execute [`String rand];

	   (* Subscribe them. *)
	   let sth = dbh#prepare_cached
		       "select id from users where email = ?" in
	   sth#execute [`String email];

	   let userid =
	     try
	       sth#fetch1int ()
	     with
		 Not_found ->
		   let sth = dbh#prepare_cached
			       "insert into users (email) values (?)" in
		   sth#execute [`String email];
		   sth#serial "users_id_seq" in

	   let sth = dbh#prepare_cached
		       "select 1 from subscriptions
                         where userid = ? and section = ?" in
	   sth#execute [`Int userid; `String section];

	   (try
	      sth#fetch1 (); ()
	    with
		Not_found ->
		  let sth = dbh#prepare_cached
			      "insert into subscriptions (userid, section)
                               values (?, ?)" in
		  sth#execute [`Int userid; `String section]);

	   dbh#commit ();

	   StdPages.ok q (* XXX ~buttons *)
	     "You have been subscribed to the mailing list."
	 with
	     Not_found ->
	       StdPages.error q
	       ("Email address confirmation failed.  If you typed in the " ^
		"link by hand, or copied and pasted it, make sure you " ^
		"copied the full link correctly.  This may also happen " ^
		"if more than 24 hours has passed since you tried to " ^
		"subscribe.  If you are still sure this wrong, you should " ^
		"report this as a bug to the mod_caml maintainers, at " ^
		"http://www.merjis.com/developers/"))

    | Remove ->				(* Remove right away. *)
	let section = q#param "section" in
	let email = q#param "email" in

	let sth = dbh#prepare_cached
		    "select id from users where email = ?" in
	sth#execute [`String email];

	(try
	   let userid = sth#fetch1int () in
	   let sth = dbh#prepare_cached
		       "delete from subscriptions
                         where userid = ? and section = ?" in
	   sth#execute [`Int userid; `String section]
	 with
	     Not_found ->
	       (* Don't give any visual indication that the user didn't exist.
		* This could be used as a possible exploit to find out which
		* email addresses are subscribed.
		*)
	       ());

	dbh#commit ();

	StdPages.ok q
	  "You have been unsubscribed from this mailing list."
;;

let () =
  register_script run
