(** Library for escaping and decoding. *)

(* Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
 *
 * Copyright (C) 1997 Daniel de Rauglaudre, INRIA
 *
 * Copyright (C) 1998 Jean-Christophe FILLIATRE
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: cgi_escape.mli,v 1.5 2005/01/07 17:36:24 ChriS Exp $
 *)

(** Types of escaping performed by the escape function. *)
type escape_t = EscapeNone | EscapeUrl | EscapeHtml | EscapeHtmlTag
	      | EscapeHtmlTextarea

val escape : escape_t -> string -> string
(** Perform text escaping on the given string.

    [escape EscapeNone] leaves the string unchanged.

    [escape EscapeUrl] does URL-encoding.

    [escape EscapeHtml] escapes '&', '<', '>', and '"' with the
    appropriate HTML escape sequence. It also turns end of lines into
    "<br>".

    [escape EscapeHtmlTag] escapes '&', '<', '>', and '"' with the
    appropriate HTML escape sequence. It also turns end of lines into
    "&#13;&#10".  This function is suitable when placing text between
    the quotes in an HTML tag, eg. [<input value="text">].

    [escape EscapeHtmlTextarea] escapes '&', '<', '>', and '"' with
    the appropriate HTML escape sequence. It leaves everything else
    unchanged. This is a suitable function to use when placing text in
    an HTML [<textarea>...</textarea>].
*)

val escape_url : string -> string
  (** Same as [escape EscapeUrl]. *)
val escape_html : string -> string
  (** Same as [escape EscapeHtml] *)
val escape_html_tag : string -> string
  (** Same as [escape EscapeHtmlTag] *)
val escape_html_textarea : string -> string
  (** Same as [escape EscapeHtmlTextarea] *)

val decode : string -> string
  (** Decode URL-encoded string.  Note that this is already done for
      you by the functions in the {!Cgi.cgi} class.  *)
val encode : string -> string
  (** [encode s] returns [s] URL-encoded, the same as [escape EscapeUrl s]. *)



(**/**)

val decode_range : string -> int -> int -> string
  (** [decode_range s low up] decodes the substring [s.[low .. up-1]]
      i.e., in the returned string the '%XX' and '+' are converted
      into their chars.  The range [s.[low .. up-1]] is overwritten.
      It returns the decoded string, stripped of white space at the
      beginning and end.

      It is ASSUMED that the range is valid i.e., [0 <= low] and [up <=
      String.length s].  Invalid '%XX' are left unchanged. *)

val encode_wrt : (char -> bool) -> string -> string
  (** [encode_wrt is_special s] encode the string [s] escaping special
      characters reported by [is_special] in the form "%XX" (' ' is
      always encoded as '+' unless [is_special ' ' = true]). *)
