/***************************************************************
 *                 Mathematical Object Library                 *
 *	      class matrix:  declarations for matrix	       *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 1.9.12                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright  1995,1996,1999,2000,2001,2002,2003,2004,2005,2006 COLLARD Christophe
 * copyright  2002,2003,2004 CREUSE Emmanuel
 * copyright  2003,2004 MERCIER Denis
 * copyright  2002,2003,2004,2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2002,2003,2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class matrix
    \brief matrix library \n

    \htmlonly 
    <FONT color="#838383">

    matrix belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 1995, 1996, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006 Christophe COLLARD \n
             copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; \endhtmlonly 2003, 2004 Denis MERCIER \n
	     copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 1.9.12
    \date 1995-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type matrix
#endif

#if !defined(_MATRIX_H)
#define _MATRIX_H


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATH_H)
#include <math.h>
#endif

#if !defined(__MATHS_H)
#include "maths.h"
#endif

#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

using namespace std;


//=============================
template <class T> class matrix
//=============================
{
  // matrix are stored with row vectors

  private:

  protected :
  int nbRows, nbColumns;  // number of rows & columns of the matrix
  vector<T> *mrows;

  public:
    matrix (); 			        // default constructor
    matrix (int, int, bool=true, T=0);  // this constructor allocates memory
    matrix (const matrix<T>&);          // copy constructor
    matrix (matrix<T>*);                // copy constructor for temporary objects
    ~matrix (); 		        // destructor

    virtual int Rows ()    const {return nbRows;}        // returns the # of rows
    virtual int Columns () const {return nbColumns;}     // returns the # of columns
    virtual void create (int,int);  // partial memory allocation (don't use this function except if you know what you're doing !!)
    virtual void assign (int,int);  // allocates memory without initilizing the matrix components to zero (so use it with great care)

    virtual T&   operator () (int,int) const;
    virtual vector<T>& operator [] (int) const;            // returns a whole row
    virtual matrix<T>& operator = (const matrix<T>&);      // defines equal operator for matrix
    virtual matrix<T>& operator = (matrix<T>*);  // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual matrix<T>& operator &= (matrix<T>&); // allocates the data from the right hand member to the left one (deletes right hand member) - no size check - no copy (use this operator with great care)
    virtual matrix<T>& operator &= (matrix<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class M> friend matrix<M> operator + (const matrix<M>&, const matrix<M>&); // overloads plus operator for matrix
    template <class M> friend matrix<M> operator - (const matrix<M>&, const matrix<M>&); // overloads minus operator for matrix
    template <class M> friend matrix<M> operator * (const matrix<M>&, const matrix<M>&); // overloads multiplicate operator for matrix or tensorial product (A.B)ij = Aik Bkj = Tij
    template <class M> friend matrix<M> operator * (const M&, const matrix<M>&);         // overloads multiplication between scalar & matrix
    template <class M> friend matrix<M> operator * (const matrix<M>&, const M&);         // overloads multiplication between matrix & scalar
    template <class M> friend vector<M> operator * (const matrix<M>&, const vector<M>&); // overloads multiplication between matrix & vector
    template <class M> friend vector<M> operator * (const vector<M>&, const matrix<M>&); // overloads multiplication between vector & matrix
    template <class M> friend matrix<M> operator / (const matrix<M>&, const matrix<M>&); // overloads division operator for matrix
    template <class M> friend matrix<M> operator / (const M&, const matrix<M>&);         // overloads division between scalar & matrix
    template <class M> friend matrix<M> operator / (const matrix<M>&, const M&);         // overloads division between scalar & matrix
    template <class M> friend M         operator | (const matrix<M>&, const matrix<M>&); // A|B = Aij Bij (for matrix & tensor2)
    template <class M> friend matrix<M> operator ^ (const vector<M>&, const vector<M>&); // overloads operator ^ for matrix = vector*t(vector) (tensorial product for 2 vectors)
    void operator += (const matrix<T>&);                                    // overloads += operator for matrix
    void operator -= (const matrix<T>&);                                    // overloads -= operator for matrix
    void operator *= (const matrix<T>&);                                    // overloads *= operator for matrix
    void operator *= (const T&);                                            // overloads *= operator between matrix and scalar
    void operator /= (const matrix<T>&);                                    // overloads /= operator for matrix
    void operator /= (const T&);	                                    // overloads /= operator between matrix and scalar
    friend bool operator ! (const matrix<T>& a) {return !a.nbRows;}
    template <class M> friend bool operator == (const matrix<M>&, const matrix<M>&);  // compares 2 matrices
    template <class M> friend bool operator != (const matrix<M>&, const matrix<M>&);
    template <class M> friend ostream& operator << (ostream&, const matrix<M>&);      // overloads output stream for matrix
    template <class M> friend istream& operator >> (istream&, const matrix<M>&);      // overloads input stream for matrix
    template <class M> friend M det (const matrix<M>&);                     // computes the determinant operator for a matrix
    template <class M> friend matrix<M> t (const matrix<M>&);               // defines tranposition operator for matrix
    virtual matrix<T> inv ();	                                            // computes inversion operator for matrix
    template <class M> friend matrix<M> gauss (matrix<M>);                  // computes matrix inversion with gauss algorithm
    template <class M> friend int inv (matrix<M>&);                         // same function as gauss but returns the inversion result 0/1
    //    friend matrix<T> gauss_pt <T> (matrix<T>);                              // computes matrix inversion with gauss algorithm
    virtual void save (const char*); // writes to disk
    virtual void read (const char*); // reads from disk
    //    friend matrix<T> kronecker <T> (int); // return the identity matrix (ie. i==j -> 1 and i!=j -> 0)
    matrix<T>& approximation ();
    virtual void permute_rows (int,int); // permutes matrix rows
    virtual void permute_columns (int,int); // permutes matrix columns
    template <class M> friend matrix<M> abs (const matrix<M>&);  // gives the absolute value of a matrix
    template <class M> friend M max (const matrix<M>&);  // gives the greatest element of a matrix
    template <class M> friend M tr (const matrix<M>&);   // gives the trace of a matrix
    template <class M> friend vector<M> diagonal (const matrix<M>&);
    template <class M> friend matrix<M> submatrix (const matrix<M>&, int, int, int, int);
    template <class M> friend matrix<M> merge_rows (const matrix<M>&, const matrix<M>&);
    template <class M> friend matrix<M> merge_columns (const matrix<M>&, const matrix<M>&);
};


//=====Private methods for matrix=============================================


//=====Public methods for matrix==============================================


//-------------------------------------
template <class T> matrix<T>::matrix ()
//-------------------------------------
{
  nbRows = nbColumns = 0;
}


//------------------------------------------------------------------------------
template <class T> matrix<T>::matrix (int rows, int columns, bool init, T value)
//------------------------------------------------------------------------------
{
  // computes matrix rows as vectors
  assert (rows>0 && columns>0);

  nbColumns = columns;
  mrows = new vector<T> [nbRows=rows];
  for (int i=0; i<nbRows; i++)
    mrows[i] =& vector<T> (nbColumns,init,value);
}


//---------------------------------------------------------
template <class T> matrix<T>::matrix (const matrix<T>& mat)
//---------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);

  mrows = new vector<T> [nbRows=mat.nbRows]; // allocates memory for rows (no initialization)
  nbColumns = mat.nbColumns;

  for (int i=1; i<=mat.nbRows; i++)
   { mrows[i-1] = nbColumns;    // allocates memory for columns (no initialization)
     for (int j=1; j<=nbColumns; j++)
       (*this)(i,j) = mat(i,j); // allocates element by element for symmatrix compatility
   }
}


//---------------------------------------------------
template <class T> matrix<T>::matrix (matrix<T>* mat)
//---------------------------------------------------
{
  assert ((*mat)[1].dim() == (*mat).nbColumns); // forbids the acces to symmetric matrices

  nbRows = (*mat).nbRows;                // copy the size of the temporary objet
  nbColumns = (*mat).nbColumns;          // copy the size of the temporary objet
  mrows = (*mat).mrows;                  // copy the address of the temporary objet
  (*mat).nbRows = (*mat).nbColumns = 0;  // set temporary object size to zero (so the data stored at p won't be delete by the destructor)
}


//--------------------------------------
template <class T> matrix<T>::~matrix ()
//--------------------------------------
{
  // destruction of the memory allocated for the matrix

  if (nbRows) delete [] mrows; // free memory only if it's been affected
  nbRows = nbColumns = 0;
}


//-----------------------------------------------------------------
template <class T> void matrix<T>::create (int nrows, int ncolumns)
//-----------------------------------------------------------------
{
  assert (nrows>0 && ncolumns>0);
  assert (!nbRows);

  mrows = new vector<T> [nbRows = nrows];  // allocates memory for rows only
  nbColumns = ncolumns;
}


//-----------------------------------------------------------------
template <class T> void matrix<T>::assign (int nrows, int ncolumns)
//-----------------------------------------------------------------
{
  assert (nrows>0 && ncolumns>0);
  assert (!nbRows);

  nbColumns = ncolumns;
  mrows = new vector<T> [nbRows = nrows];  // allocates memory with the = operator for vector (no initialization)
  for (int i=0; i<nbRows; i++)
    mrows[i] =& vector<T> (nbColumns,false);
}


//----------------------------------------------------------------------
template <class T> T& matrix<T>::operator () (int row, int column) const
//----------------------------------------------------------------------
{
  assert ((row>0)&&(row<=nbRows));
  assert ((column>0)&&(column<=nbColumns));
  return mrows[row-1][column];
}


//------------------------------------------------------------------
template <class T> vector<T>& matrix<T>::operator [] (int row) const
//------------------------------------------------------------------
{
  assert ((row>0)&&(row<=nbRows));
  return mrows[row-1];
}


//------------------------------------------------------------------------
template <class T> matrix<T>& matrix<T>::operator = (const matrix<T>& mat)
//------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);

  if (!nbRows) assign(mat.nbRows,mat.nbColumns);
  else assert ( (nbRows==mat.nbRows) && (nbColumns==mat.nbColumns) );

  for (int i=1; i<=mat.nbRows; i++)
    for (int j=1; j<=nbColumns; j++)
      (*this)(i,j) = mat(i,j);

  return *this;
}


//------------------------------------------------------------------
template <class T> matrix<T>& matrix<T>::operator = (matrix<T>* mat)
//------------------------------------------------------------------
{
  assert ((*mat).nbRows && (*mat).nbColumns);
  assert ((*mat)[1].dim() == (*mat).nbColumns); // forbids the acces to symmetric matrices

  if (nbRows)
    { assert ( (nbRows == (*mat).nbRows) && (nbColumns == (*mat).nbColumns) );
      delete [] mrows;
    }
  else
    { nbRows = (*mat).nbRows;
      nbColumns = (*mat).nbColumns;
    }
  mrows = (*mat).mrows;
  (*mat).nbRows = (*mat).nbColumns = 0;

  return *this;
}


//-------------------------------------------------------------------
template <class T> matrix<T>& matrix<T>::operator &= (matrix<T>& mat)
//-------------------------------------------------------------------
{
  if (!mat.nbRows) nbRows = nbColumns = 0;
  else
    { assert (mat[1].dim() == mat.nbColumns); // forbids the acces to symmetric matrices
      if (nbRows)
	delete [] mrows;
      nbRows = mat.nbRows;
      nbColumns = mat.nbColumns;
      mrows = mat.mrows;
      mat.nbRows = mat.nbColumns = 0;
    }

  return *this;
}


//-------------------------------------------------------------------
template <class T> matrix<T>& matrix<T>::operator &= (matrix<T>* mat)
//-------------------------------------------------------------------
{
  if (!(*mat).nbRows) nbRows = nbColumns = 0;
  else
    { assert ((*mat)[1].dim() == (*mat).nbColumns); // forbids the acces to symmetric matrices
      if (nbRows)
	delete [] mrows;
      nbRows = (*mat).nbRows;
      nbColumns = (*mat).nbColumns;
      mrows = (*mat).mrows;
      (*mat).nbRows = (*mat).nbColumns = 0;
    }

  return *this;
}


//------------------------------------------------------------------------------------
template <class T> matrix<T> operator + (const matrix<T>& mat1, const matrix<T>& mat2)
//------------------------------------------------------------------------------------
{
  assert ((mat1.nbRows==mat2.nbRows) && (mat1.nbColumns==mat2.nbColumns)); // matrices must have the same size
  assert((mat1.nbRows)&&(mat1.nbColumns));

  matrix<T> mat = mat1;
  mat += mat2;

  return mat;
}


//------------------------------------------------------------------------------------
template <class T> matrix<T> operator - (const matrix<T>& mat1, const matrix<T>& mat2)
//------------------------------------------------------------------------------------
{
  assert ((mat1.nbRows==mat2.nbRows) && (mat1.nbColumns==mat2.nbColumns)); // matrices must have the same size
  assert((mat1.nbRows)&&(mat1.nbColumns));

  matrix<T> mat = mat1;
  mat -= mat2;

  return mat;
}


//------------------------------------------------------------------------------------
template <class T> matrix<T> operator * (const matrix<T>& mat1, const matrix<T>& mat2)
//------------------------------------------------------------------------------------
{
  assert (mat1.nbColumns==mat2.nbRows); // matrix can't be multiplicated

  matrix<T> mt(mat1.nbRows,mat2.nbColumns); // allocates memory and set matrix components to zero

  for (int i=1; i<=mat1.nbRows; i++)
    for (int j=1; j<=mat2.nbColumns; j++)
      for (int k=1; k<=mat2.nbRows; k++)
  	mt(i,j) += mat1(i, k) * mat2(k, j);

  return mt;
}


//--------------------------------------------------------------------------
template <class T> matrix<T> operator * (const matrix<T>& mat, const T& elt)
//--------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);
  matrix<T> mp = mat;
  mp *= elt;
  return mp;
}


//--------------------------------------------------------------------------
template <class T> matrix<T> operator * (const T& elt, const matrix<T>& mat)
//--------------------------------------------------------------------------
{
  assert (mat.nbRows&&mat.nbColumns);
  matrix<T> mp = mat;
  mp *= elt;
  return mp;
}


//--------------------------------------------------------------------------------
template <class T> vector<T> operator * (const matrix<T>& mat, const vector<T>& v)
//--------------------------------------------------------------------------------
{ 
  assert (mat.nbColumns == v.dim()); // checks validity of the operation
  vector<T> u(mat.nbRows); // allocates memory and set vector components to zero
  for (int i=1; i<=mat.nbRows; i++)
      for (int j=1; j<=mat.nbColumns;j++)
           u[i] += mat(i,j) * v[j];
  return u;
}


//--------------------------------------------------------------------------------
template <class T> vector<T> operator * (const vector<T>& v, const matrix<T>& mat)
//--------------------------------------------------------------------------------
{ 
  assert (mat.nbRows == v.dim());
  vector<T> u(mat.nbColumns); // allocates memory and set vector components to zero
  for (int i=1; i<=mat.nbColumns; i++)
      for (int j=1; j<=mat.nbRows; j++)
          u[i] += v[j] * mat(j,i);
  return u;
}


//------------------------------------------------------------------------------------
template <class T> matrix<T> operator / (const matrix<T>& mat1, const matrix<T>& mat2)
//------------------------------------------------------------------------------------
{
  matrix<T> mt = mat1;
  mt *= gauss(mat2);
  return mt;
}


//--------------------------------------------------------------------------
template <class T> matrix<T> operator / (const T& elt, const matrix<T>& mat)
//--------------------------------------------------------------------------
{
  matrix<T> mt = gauss(mat);
  mt*= elt;
  return mt;
}


//--------------------------------------------------------------------------
template <class T> matrix<T> operator / (const matrix<T>& mat, const T& elt)
//--------------------------------------------------------------------------
{
  assert (elt!=0); // can't divide by zero !
  matrix<T> mp = mat;
  mp /= elt;
  return mp;
}


//----------------------------------------------------------------------------
template <class T> T operator | (const matrix<T>& mat1, const matrix<T>& mat2)
//----------------------------------------------------------------------------
{
  assert (mat1.nbRows==mat2.nbRows); // matrices must have the same size
  assert (mat1.nbRows);

  T sum=0;
  for (int i=1; i<=mat1.nbRows; i++)
    for (int j=1; j<=mat1.nbColumns; j++)
      sum += mat1(i,j) * mat2(i,j);

  return sum;
}


//------------------------------------------------------------------------------
template <class T> matrix<T> operator ^ (const vector<T>& v, const vector<T>& w)
//------------------------------------------------------------------------------
{
  assert (v.dim() && w.dim());
  matrix<T> mat(v.dim(),w.dim(),false); // allocates memory for columns (no initialization)
  for (int i=1; i<=v.dim(); i++)
    for (int j=1; j<=w.dim(); j++)
      mat (i,j) = v[i]*w[j];
  return mat;
}


//-------------------------------------------------------------------
template <class T> void matrix<T>::operator += (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  assert(nbRows && nbColumns);
  assert ((nbRows==mat.nbRows) && (nbColumns==mat.nbColumns)); // matrices must have the same size
  // the following command allows the following operation : matrix += symatrix, so use it instead of mrows[i] += mat[i+1];
  for (int i=1; i<=nbRows; i++)
    for (int j=1; j<=nbColumns; j++)
      (*this)(i,j) += mat(i,j);
}


//-------------------------------------------------------------------
template <class T> void matrix<T>::operator -= (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  assert(nbRows && nbColumns);
  assert ((nbRows==mat.nbRows) && (nbColumns==mat.nbColumns)); // matrices must have the same size
  // the following command allows the following operation : matrix += symatrix, so use it instead of mrows[i] += mat[i+1];
  for (int i=1; i<=nbRows; i++)
    for (int j=1; j<=nbColumns; j++)
      (*this)(i,j) -= mat(i,j);
}


//-------------------------------------------------------------------
template <class T> void matrix<T>::operator *= (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  assert ((nbColumns==mat.nbRows)&&(nbColumns==mat.nbColumns)); // A(x,y) * B(y,y)
  assert (nbColumns==(*this)[1].dim()); // this operator is not defined for symmatrix on the left side of the equation
  matrix<T> mt(nbRows,nbColumns); // allocates memory and set matrix components to zero

  for (int i=1; i<=nbRows; i++)
    for (int j=1; j<=mat.nbColumns; j++)
      for (int k=1; k<=mat.nbRows; k++)
	mt(i,j) += (*this)(i, k) * mat(k, j);
  // next lines are faster than *this = mt; because we use the temporary object (no copy needed)
  delete [] mrows;
  mrows = mt.mrows;
  mt.nbRows = mt.nbColumns = 0;
}


//-----------------------------------------------------------
template <class T> void matrix<T>::operator *= (const T& elt)
//-----------------------------------------------------------
{
  assert (nbRows && nbColumns);
  for (int i=1; i<=nbRows; i++)
      (*this)[i] *= elt;
}


//-------------------------------------------------------------------
template <class T> void matrix<T>::operator /= (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  // a modifier et optimiser - cf operateur /

  *this *= gauss(mat);
}


//-----------------------------------------------------------
template <class T> void matrix<T>::operator /= (const T& elt)
//-----------------------------------------------------------
{
  assert (elt);

  for (int i=1; i<=nbRows; i++)
    (*this)[i] /= elt;
}


//--------------------------------------------------------------------------------
template <class T> bool operator == (const matrix<T>& mat1, const matrix<T>& mat2)
//--------------------------------------------------------------------------------
{
  bool result=( (mat1.nbRows==mat2.nbRows) && (mat2.nbColumns==mat2.nbColumns) );

  for (int i=1; i<=mat1.nbRows && result; i++)
    for (int j=1; (j<=mat1.nbColumns) && (result); j++)
      result *= abs(mat1(i,j) - mat2(i,j)) < epsilon;    // where epsilon is the computer approximation

  return result;
}


//--------------------------------------------------------------------------------
template <class T> bool operator != (const matrix<T>& mat1, const matrix<T>& mat2)
//--------------------------------------------------------------------------------
{
  return !(mat1 == mat2);
}


//------------------------------------------------------------------------
template <class T> ostream& operator << (ostream& s, const matrix<T>& mat)
//------------------------------------------------------------------------
{
  for (int i=1; i<=mat.nbRows; i++)
      s << mat[i];
  s << endl;
  return s;
}


//------------------------------------------------------------------------
template <class T> istream& operator >> (istream& s, const matrix<T>& mat)
//------------------------------------------------------------------------
{
  for (int i=1; i<=mat.nbRows; i++)
      s >> mat[i];
  return s;
}


//---------------------------------------------
template <class T> T det (const matrix<T>& mat)
//---------------------------------------------
{
  assert (mat.nbRows == mat.nbColumns); // matrix must be squared
  T value = 0;
  if (mat.nbRows==1) return mat(1,1);
  matrix<T> m(mat.nbRows-1,mat.nbColumns-1);

  for (int i=1; i<=mat.nbRows; i++)
  {   for (int j=2; j<=mat.nbColumns; j++)
	  for (int k=1,l=1; k<=mat.nbRows; k++)
	      if (k!=i)
		 m(l++,j-1) = mat(k,j);
      if (mat(i,1))
      {
         if ( ((float)(i)/2) == int((float) (i)/2) )
	    value -= mat(i,1)*det(m);
	 else
	    value += mat(i,1)*det(m);
      }
  }
  return value;
}


//---------------------------------------------------
template <class T> matrix<T> t (const matrix<T>& mat)
//---------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);
  matrix<T> mt(mat.nbColumns,mat.nbRows,false); // allocates memory (no initialization)
  for (int i=1; i<=mat.nbRows; i++)
      for (int j=1; j<=mat.nbColumns; j++)
	  mt(j,i) = mat(i,j);
  return mt;
}


//--------------------------------------------
template <class T> matrix<T> matrix<T>::inv ()
//--------------------------------------------
{
  assert (nbRows==nbColumns); // matrix must be squared
  assert (nbRows); // matrix must exists
  T dt = det(*this);
  matrix<T> mat(nbRows-1,nbColumns-1), newmat (nbRows,nbColumns);
  assert (dt); // matrix can't be inverted if the determinant = 0
  // computes the cofactors matrix and divides it by the determinant
  for (int i=1; i<=nbRows; i++)
      for (int j=1; j<=nbColumns; j++)
      {   for (int k=1,x=1; k<=nbRows; k++)
	      if (k!=i)
	      {	 for (int l=1,y=1; l<=nbColumns; l++)
		     if (l!=j)
			mat(x,y++) = (*this)(k,l);
		 x++;
	      }
	  if ( ((float)(i+j)/2) == int ((float)(i+j)/2) )
	     newmat(i,j) = det(mat)/dt;
	  else
	     newmat(i,j) = -det(mat)/dt;
      }
  // tranposes the cofactors matrix
  newmat = t(newmat);
  // divides the transposed cofactors matrix by the determinant
  //  newmat = newmat/dt;
  return newmat;
}


//------------------------------------------------
template <class T> matrix<T> gauss (matrix<T> mat)
//------------------------------------------------
{
  assert (mat.nbRows == mat.nbColumns); // matrix must be squared
  matrix <T> matinv (mat.nbRows, mat.nbColumns); // allocates memory and set matrix components to zero
  int posx, dim = mat.nbRows;
  T pivot, elt;

  for (int i=1; i<=dim; i++) matinv(i,i) = 1; // initialises matinv (matinv - Id)

  for (int k=1; k<=dim; k++)
    { pivot = 0;
      // search the biggest element of the ith line and kth column
      for (int i=k; i<=dim; i++)
	if (abs(mat(i,k)) > abs(pivot))
	  { pivot = mat(i,k);
	    posx = i;
	  }

      assert (abs(pivot) > epsilon);

      // Rows permutation : row k <-> row posx
      if (posx != k)
      	{ mat.permute_rows(k,posx);
      	  matinv.permute_rows(k,posx);
      	}

      //  starts the resolution at step k
      vector<T> *mat_k, *matinv_k;
      mat_k = &mat[k];
      matinv_k = &matinv[k];
      T mat_kk = (*mat_k)[k];

      for (int i=1; i<k; i++)
      	{ vector<T> *mat_i, *matinv_i;
	  mat_i = &mat[i];
	  matinv_i = &matinv[i];
	  elt = (*mat_i)[k] / mat_kk;
	  for (int j=k+1; j<=dim; j++) (*mat_i)[j] -= (*mat_k)[j] * elt;
      	  for (int j=1; j<=dim; j++) (*matinv_i)[j] -= (*matinv_k)[j] * elt;
	}
      for (int i=k+1; i<=dim; i++)
	{ vector<T> *mat_i, *matinv_i;
	  mat_i = &mat[i];
	  matinv_i = &matinv[i];
	  elt = (*mat_i)[k] / mat_kk;
	  for (int j=k+1; j<=dim; j++) (*mat_i)[j] -= (*mat_k)[j] * elt;
	  for (int j=1;   j<=dim; j++) (*matinv_i)[j] -= (*matinv_k)[j] * elt;
	}
      for (int j=k+1; j<=dim; j++) (*mat_k)[j] /= mat_kk;
      for (int j=1; j<=dim; j++) (*matinv_k)[j] /= mat_kk;
    }

  return matinv;
}


//-----------------------------------------
template <class T> int inv (matrix<T>& mat)
//-----------------------------------------
{
  assert (mat.nbRows == mat.nbColumns); // matrix must be squared
  matrix <T> matinv (mat.nbRows, mat.nbColumns); // allocates memory and set matrix components to zero
  int posx, dim = mat.nbRows;
  T pivot, elt;

  for (int i=1; i<=dim; i++) matinv(i,i) = 1; // initialises matinv (matinv - Id)

  for (int k=1; k<=dim; k++)
    { pivot = 0;
      // search the biggest element of the ith line and kth column
      for (int i=k; i<=dim; i++)
	if (abs(mat(i,k)) > abs(pivot))
	  { pivot = mat(i,k);
	    posx = i;
	  }

      if (abs(pivot) < epsilon) return 0;

      // Rows permutation : row k <-> row posx
      if (posx != k)
      	{ mat.permute_rows(k,posx);
      	  matinv.permute_rows(k,posx);
      	}

      // Rows permutation : row k <-> row posx
      if (posx != k)
      	{ mat.permute_rows(k,posx);
      	  matinv.permute_rows(k,posx);
      	}

      //  starts the resolution at step k
      vector<T> *mat_k, *matinv_k;
      mat_k = &mat[k];
      matinv_k = &matinv[k];
      T mat_kk = (*mat_k)[k];

      for (int i=1; i<k; i++)
      	{ vector<T> *mat_i, *matinv_i;
	  mat_i = &mat[i];
	  matinv_i = &matinv[i];
	  elt = (*mat_i)[k] / mat_kk;
	  for (int j=k+1; j<=dim; j++) (*mat_i)[j] -= (*mat_k)[j] * elt;
      	  for (int j=1; j<=dim; j++) (*matinv_i)[j] -= (*matinv_k)[j] * elt;
	}
      for (int i=k+1; i<=dim; i++)
	{ vector<T> *mat_i, *matinv_i;
	  mat_i = &mat[i];
	  matinv_i = &matinv[i];
	  elt = (*mat_i)[k] / mat_kk;
	  for (int j=k+1; j<=dim; j++) (*mat_i)[j] -= (*mat_k)[j] * elt;
	  for (int j=1;   j<=dim; j++) (*matinv_i)[j] -= (*matinv_k)[j] * elt;
	}
      for (int j=k+1; j<=dim; j++) (*mat_k)[j] /= mat_kk;
      for (int j=1; j<=dim; j++) (*matinv_k)[j] /= mat_kk;
    }

  mat =& matinv;
  return 1;
}

/* FAUX - A REVOIR
//---------------------------------------------------
template <class T> matrix<T> gauss_pt (matrix<T> mat)
//---------------------------------------------------
{
  assert (mat.nbRows == mat.nbColumns); // matrix must be squared
  matrix <T> matinv (mat.nbRows, mat.nbColumns); // allocates memory and set matrix components to zero
  int posx, posy, dim = mat.nbRows;
  T pivot, elt;

  for (int i=1; i<=dim; i++) matinv(i,i) = 1; // initialises matinv (matinv - Id)

  for (int k=1; k<=dim; k++)
    { pivot = 0;
      // search the biggest element of the ith line and kth column
      for (int i=k; i<=dim; i++)
	for (int j=k; j<=dim; j++)
	  if (abs(mat(i,j)) > abs(pivot))
	  { pivot = mat(i,j);
	    posx = i;
	    posy = j;
	  }

      assert (abs(pivot)>epsilon);

      // Rows permutation : row k <-> row posx
      if (posx != k)
	{ mat.permute_rows(k,posx);
	  matinv.permute_rows(k,posx);
	}

      // Columns permutation : column k <-> column posy
      if (posy!=k)
	{ mat.permute_columns(k,posy);
	  matinv.permute_columns(k,posy);
	}

      //  starts the resolution at step k
      for (int i=1; i<k; i++)
	{ elt = mat(i,k) / mat(k,k);
	  for (int j=k+1; j<=dim; j++) mat(i,j) -= mat(k,j) * elt;
	  for (int j=1;   j<=dim; j++) matinv(i,j) -= matinv(k,j) * elt;
	}
      for (int i=k+1; i<=dim; i++)
	{  elt = mat(i,k) / mat(k,k);
	   for (int j=k+1; j<=dim; j++) mat(i,j) -= mat(k,j) * elt;
	   for (int j=1;   j<=dim; j++) matinv(i,j) -= matinv(k,j) * elt;
	}
      for (int j=k+1; j<=dim; j++) mat(k,j) /= mat(k,k);
      for (int j=1;   j<=dim; j++) matinv (k,j) /= mat(k,k);
    }

  return matinv;
}
*/

//--------------------------------------------------------
template <class T> void matrix<T>::save (const char* path)
//--------------------------------------------------------
{
  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


//--------------------------------------------------------
template <class T> void matrix<T>::read (const char* path)
//--------------------------------------------------------
{
  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


//-------------------------------------------------------
template <class T> matrix<T>& matrix<T>::approximation ()
//-------------------------------------------------------
{
  for (int i=0; i<nbRows; i++)
    mrows[i].approximation();
  return *this;
}


//--------------------------------------------
template <class T> matrix<T> kronecker (int n)
//--------------------------------------------
{
  matrix<T> mat(n,n); // allocates memory and set matrix components to zero
  for (int i=1; i<=n; i++)
    mat(i,i) = 1;
  return mat;
}


//------------------------------------------------------------
template <class T> void matrix<T>::permute_rows (int i, int j)
//------------------------------------------------------------
{  
  assert (nbColumns==(*this)[1].dim()); // does not allow permutation for symmatrix or sparse matrix
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=nbRows);

  if (i!=j) permute (mrows[i-1],mrows[j-1]);
}


//---------------------------------------------------------------
template <class T> void matrix<T>::permute_columns (int i, int j)
//---------------------------------------------------------------
{  
  assert (nbColumns==(*this)[1].dim()); // does not allow permutation for symmatrix or sparse matrix
  assert (i>0 && i<=nbColumns);
  assert (j>0 && j<=nbColumns);
  T elt;

  if (i!=j)
    for (int k=1; k<=nbRows; k++)
      { elt=(*this)(k,i);
        (*this)(k,i)=(*this)(k,j);
	(*this)(k,j)=elt;
      }
}


//-----------------------------------------------------
template <class T> matrix<T> abs (const matrix<T>& mat)
//-----------------------------------------------------
{
  assert (mat.nbColumns);
  matrix<T> mtx;
  mtx.create(mat.nbRows,mat.nbColumns);

  for (int i=1; i<=mat.nbRows; i++)
    mtx[i] =& abs(mat[i]);

  return mtx;
}


//---------------------------------------------
template <class T> T max (const matrix<T>& mat)
//---------------------------------------------
{
  assert (mat.nbColumns);
  T val = mat(1,1);

  for (int i=1; i<=mat.nbRows; i++)
    for (int j=1; j<=mat.nbColumns; j++)
      if (val<mat(i,j)) val = mat(i,j);

  return val;
}


//--------------------------------------------
template <class T> T tr (const matrix<T>& mat)
//--------------------------------------------
{ 
  assert (mat.nbRows);
  assert (mat.nbRows==mat.nbColumns); // matrix must be squared

  T val = 0;

  for (int i=1; i<=mat.nbRows; i++)
    val += mat(i,i);

  return val;
}


//----------------------------------------------------------
template <class T> vector<T> diagonal (const matrix<T>& mat)
//----------------------------------------------------------
{
  assert (mat.nbRows);
  assert (mat.Rows()==mat.Columns());

  vector<T> v(mat.nbRows,false);

  for (int i=1; i<=mat.nbRows; i++)
    v[i] = mat(i,i);

  return v;
}


//---------------------------------------------------------------------------------------
template <class T> matrix<T> submatrix (const matrix<T>& mat, int m, int n, int p, int q)
//---------------------------------------------------------------------------------------
{
  assert (m>0 && m<=n && n<=mat.nbRows);
  assert (p>0 && p<=q && q<=mat.nbColumns);

  matrix<T> sub_mat;
  sub_mat.create (n-m+1,q-p+1);

  for (int i=m; i<=n; i++)
    sub_mat[i-m+1] =& subvector(mat[i],p,q);

  return sub_mat;
}


//------------------------------------------------------------------------------------
template <class T> matrix<T> merge_rows (const matrix<T>& mat1, const matrix<T>& mat2)
//------------------------------------------------------------------------------------
{
  assert (mat1.nbColumns==mat1[1].dim()); // excludes symmatrix
  assert (mat1.nbColumns == mat2.nbColumns);

  matrix<T> mat;
  mat.create (mat1.nbRows+mat2.nbRows,mat1.nbColumns);

  for (int i=1; i<=mat1.nbRows; i++)
    mat[i] = mat1[i];
  for (int i=1; i<=mat2.nbRows; i++)
    mat[mat1.nbRows+i] = mat2[i];

  return mat;
}


//---------------------------------------------------------------------------------------
template <class T> matrix<T> merge_columns (const matrix<T>& mat1, const matrix<T>& mat2)
//---------------------------------------------------------------------------------------
{
  assert (mat1.nbColumns==mat1[1].dim()); // excludes symmatrix
  assert (mat1.nbRows == mat2.nbRows);

  matrix<T> mat;
  mat.create (mat1.nbRows,mat1.nbColumns+mat2.nbColumns);

  for (int i=1; i<=mat.nbRows; i++)
    mat[i] =& merge(mat1[i],mat2[i]);

  return mat;
}


#endif
