/***************************************************************
 *                 Mathematical Object Library                 *
 *class symtensor4: declaration for symmetric 4th order tensors*
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2009,2010 COLLARD Christophe
 * copyright © 2009,2010 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::symtensor4
    \brief library for symmetric \f$4^\text{th}\f$ order tensor \n

    \htmlonly 
    <FONT color="#838383">

    symtensor4 belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    When fourth order tensors have the following symmetry properties \f$ T_{ijkl} = T_{jikl} = T_{ijlk} \f$, the storage and the operations on these tensors can be improved by using a matrix storage instead of the one given by the tensor4 class (sets of matrices). To do so, we associate with the tensor \f$ \boldsymbol{T} \f$ in \f$ \Bbb{R}^n \times \Bbb{R}^n \times \Bbb{R}^m \times \Bbb{R}^m \f$ the matrix \f$ \boldsymbol{M} \f$ in \f$ \Bbb{R}^{2n} \times \Bbb{R}^{2m} \f$ defined by: \n


    \authors copyright \htmlonly &#169; \endhtmlonly 2009, 2010 Christophe COLLARD \n
	     copyright \htmlonly &#169; 2009, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
    \version 2.4.0
    \date 2009-2010
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type symtensor4
#endif

#if !defined(_SYMTENSORS4_H)
#define _SYMTENSORS4_H


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATH_H)
#include <math.h>
#endif

#if !defined(__PARAMETERS_H)
#include "parameters.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__TENSORS2_H)
#include "MOL++/tensors2.h"
#endif

#if !defined(__SYMTENSORS2_H)
#include "MOL++/symtensors2.h"
#endif

#if !defined(__TENSORS3_H)
#include "MOL++/tensors3.h"
#endif

#if !defined(__TENSORS4_H)
#include "MOL++/tensors4.h"
#endif

#if !defined(__ABAQUS_CONVERSION_TOOLS_H)
#include "MOL++/abaqus conversion tools.h"
#endif

using namespace std;

namespace mol
{


//=================================
template <class T> class symtensor4
//=================================
{
  private:
    matrix<T> mat;

  public:
    symtensor4 () {}
    symtensor4 (int, bool=true, T=0);
    symtensor4 (int, int, int, int, bool=true, T=0);

    symtensor4 (const matrix<T>& mt) {mat = mt;}       // cast constructor (converts matrix    -> symtensor4)
    symtensor4 (const symmatrix<T>& smt) {mat = smt;}  // cast constructor (converts symmatrix -> symtensor4)

    symtensor4 (symtensor4<T>* st) {mat = &(st->mat);}   // copy constructor for temporary objects
    symtensor4 (matrix<T>* st) {mat = st;}               // copy constructor for temporary objects
    symtensor4 (symmatrix<T>* smat) {mat = smat;}        // copy constructor for temporary objects

    ~symtensor4 () {}       // destructor

    virtual int dim1 () const {return .5 * (sqrt(1.+8*mat.Rows()) - 1);}     // returns the tensor 1st component size : i
    virtual int dim2 () const {return .5 * (sqrt(1.+8*mat.Rows()) - 1);}     // returns the tensor 2nd component size : j
    virtual int dim3 () const {return .5 * (sqrt(1.+8*mat.Columns()) - 1);}  // returns the tensor 3rd component size : k
    virtual int dim4 () const {return .5 * (sqrt(1.+8*mat.Columns()) - 1);}  // returns the tensor 3rd component size : l
    virtual void create (int);              // partial memory allocation (don't use this function except if you know what you're doing !)
    virtual void assign (int,int,int,int);  // allocates memory without initilizing the tensor components to zero
    friend bool operator ! (const symtensor4<T>& st) {return !st.mat.Rows();}

    tensor4<T> operator () () {return abaqus::matrix2symtensor (mat);}
    virtual T& operator () (int, int, int, int) const;     // returns the element Pijkl : P(tensor3,tensor2,line,column)
    virtual vector<T>& operator [] (int i) const {return mat[i];}
    //    virtual symtensor4<T>& operator = (symtensor4<T>* st) {mat =& st->mat;}  // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual symtensor4<T>& operator &= (const symtensor4<T>& st) {mat &= st.mat;}  // copies symtensor4 without size check (use this operator with great care)
    virtual symtensor4<T>& operator &= (symtensor4<T>* st) {mat &=& (st->mat);}    // allocates the data from the right hand temporary object to the left hand object (deletes  temporary object)
    template <class Tf> friend symtensor4<Tf> operator +  (const symtensor4<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend symtensor4<Tf> operator -  (const symtensor4<Tf>&, const symtensor4<Tf>&);
    template <class Tf> friend Tf             operator |  (const symtensor4<Tf>&, const symtensor4<Tf>&); // T_ijkl  R_ijkl
    template <class Tf> friend symtensor4<Tf> operator || (const symtensor4<Tf>&, const symtensor4<Tf>&); // P_ijkl = T_ijpq R_pqkl
    symtensor4<T>& operator += (const symtensor4<T>&);
    symtensor4<T>& operator -= (const symtensor4<T>&);
    template <class Tf> friend bool operator == (const symtensor4<Tf>& st1, const symtensor4<Tf>& st2) {return st1.mat == st2.mat;}
    template <class Tf> friend bool operator != (const symtensor4<Tf>& st1, const symtensor4<Tf>& st2) {return st1.mat != st2.mat;}

    //------------------------
    // operations with scalars
    //------------------------
    symtensor4<T>& operator *= (const T&);
    template <class Tf> friend symtensor4<Tf> operator * (const symtensor4<Tf>&, const Tf&);
    template <class Tf> friend symtensor4<Tf> operator * (const Tf&, const symtensor4<Tf>&);

    //-------------------------
    // operations withs vectors
    //-------------------------
    template <class Tf> friend tensor3<Tf> operator * (const symtensor4<Tf>&, const vector<Tf>&);   // P_ijk = T_ijkl * V_l
    template <class Tf> friend tensor3<Tf> operator * (const vector<Tf>&, const symtensor4<Tf>&);   // P_jkl = T_ijkl * V_i
    template <class Tf> friend tensor3<Tf> operator | (const symtensor4<Tf>&, const vector<Tf>&);   // P_ijl = T_ijkl * V_k
    template <class Tf> friend tensor3<Tf> operator | (const vector<Tf>&, const symtensor4<Tf>&);   // P_ikl = T_ijkl * V_j
    template <class Tf> friend tensor3<Tf> tensor_sum (const symtensor4<Tf>&, const vector<Tf>&, int); // P = T : v

    //---------------------------------
    // operation with 2nd order tensors
    //---------------------------------
    template <class Tf> friend symtensor2<Tf> operator || (const symtensor4<Tf>&, const tensor2<Tf>&);//P_ij = T_ijkl * M_kl
    template <class Tf> friend symtensor2<Tf> operator || (const tensor2<Tf>&, const symtensor4<Tf>&);//P_kl = M_ij * T_ijkl

    //-------------------------------------------
    // operation with symmetric 2nd order tensors
    //-------------------------------------------
    template <class Tf> friend symtensor4<Tf> operator ^  (const symtensor2<Tf>&, const symtensor2<Tf>&);  // T_ijkl = M1_ij M2_kl

    //--------------------------------
    // operation with 3rd order tensor
    //--------------------------------
    template <class Tf> friend tensor3<Tf> operator || (const tensor3<Tf>&, const symtensor4<Tf>&);// P_n,ij = R_n,kl T_klij
    template <class Tf> friend tensor3<Tf> operator || (const symtensor4<Tf>&, const tensor3<Tf>&);// P_n,ij = T_ijkl R_n,kl

    //----------------------------------
    // operations with 4th order tensors
    //----------------------------------
    symtensor4 (const tensor4<T>& tsr) { mat = abaqus::symtensor2matrix<T> (tsr);}  // cast constructor (converts tensor4   -> symtensor4)
    symtensor4 (tensor4<T>* tsr) {mat = &(abaqus::symtensor2matrix<T> (*tsr));}     // copy constructor for temporary objects

    virtual symtensor4<T>& operator &= (const tensor4<T>& tsr);  // copies symtensor4 without size check (use this operator with great care)
    virtual symtensor4<T>& operator &= (tensor4<T>* tsr);        // allocates the data from the right hand temporary object to the left hand object (deletes temporary object)
    template <class Tf> friend bool operator == (const symtensor4<Tf>& st4, const tensor4<Tf>& t4) {return st4 == (symtensor4<T>) t4;}
    template <class Tf> friend bool operator == (const tensor4<Tf>& t4, const symtensor4<Tf>& st4) {return st4 == (symtensor4<T>) t4;}
    template <class Tf> friend bool operator != (const symtensor4<Tf>& st4, const tensor4<Tf>& t4) {return st4 != (symtensor4<T>) t4;}
    template <class Tf> friend bool operator != (const tensor4<Tf>& t4, const symtensor4<Tf>& st4) {return st4 != (symtensor4<T>) t4;}

    template <class Tf> friend tensor4<Tf> operator +  (const symtensor4<Tf>& st, const tensor4<Tf>& t4) {return abaqus::matrix2symtensor (st.mat) +  t4;}
    template <class Tf> friend tensor4<Tf> operator -  (const symtensor4<Tf>& st, const tensor4<Tf>& t4) {return abaqus::matrix2symtensor (st.mat) -  t4;}
    template <class Tf> friend Tf          operator |  (const symtensor4<Tf>& st, const tensor4<Tf>& t4) {return abaqus::matrix2symtensor (st.mat) |  t4;}
    template <class Tf> friend tensor4<Tf> operator || (const symtensor4<Tf>& st, const tensor4<Tf>& t4) {return abaqus::matrix2symtensor (st.mat) || t4;}

    template <class Tf> friend tensor4<Tf> operator +  (const tensor4<Tf>& t4, const symtensor4<Tf>& st) {return t4 +  abaqus::matrix2symtensor (st.mat);}
    template <class Tf> friend tensor4<Tf> operator -  (const tensor4<Tf>& t4, const symtensor4<Tf>& st) {return t4 -  abaqus::matrix2symtensor (st.mat);}
    template <class Tf> friend Tf          operator |  (const tensor4<Tf>& t4, const symtensor4<Tf>& st) {return t4 |  abaqus::matrix2symtensor (st.mat);}
    template <class Tf> friend tensor4<Tf> operator || (const tensor4<Tf>& t4, const symtensor4<Tf>& st) {return t4 || abaqus::matrix2symtensor (st.mat);}

    template <class Tf> friend symtensor4<Tf> Id4sym (int); // returns identity tensor
    //    template <class Tf> friend tensor4<Tf> change_basis (const matrix<Tf>&, const tensor4<Tf>&);
    template <class Tf> friend ostream& operator << (ostream&, const symtensor4<Tf>&);
    template <class Tf> friend istream& operator >> (istream&, const symtensor4<Tf>&);
    virtual symtensor4<T> inv () {return mat.inv();}  // computes inversion operator (do not use this one since the algorithm is too slow)
    template <class Tf> friend symtensor4<Tf> gauss (symtensor4<Tf> tsr) {return gauss(tsr.mat);}  // computes matrix inversion with gauss algorithm
    template <class Tf> friend bool inv (symtensor4<Tf>& tsr, long double eps=epsilon) {return mol::inv (tsr.mat, eps);}  // same function as gauss but returns the inversion result 0/1

    // i/o disk
    virtual void save (const char*);   // writes to disk
    virtual void save (const string&); // writes to disk
    virtual void read (const char*);   // reads from disk
    virtual void read (const string&);   // reads from disk

    symtensor4<T>& approximation ();
};


//=====Private methods for symtensor4==========================================


//=====Public methods for symtensor4===========================================


/*!
  \brief Constructor

  \param n space dimension for the symmetric \f$ 4^\text{th} \f$ order tensor
  \param init boolean that indicates if the constructor has to initialize the tensor coordinates (default value is true)
  \param value value for tensor coordinates (default value is 0)
*/

//----------------------------------------------------------------------
template <class T> symtensor4<T>::symtensor4 (int n, bool init, T value)
//----------------------------------------------------------------------
{
  assert (n > 0);

  mat = matrix<T> (.5*n*(n+1), .5*n*(n+1), init, value);
}


/*!
  \brief Constructor

  \param i space dimension for the first component of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param j space dimension for the second component of the symmetric \f$ 4^\text{th} \f$ order tensor (i and j must be equal)
  \param k space dimension for the third component of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param l space dimension for the fourth component of the symmetric \f$ 4^\text{th} \f$ order tensor (k and l must be equal)
  \param init boolean that indicates if the constructor has to initialize the tensor coordinates (default value is true)
  \param value value for tensor coordinates (default value is 0)
*/

//-------------------------------------------------------------------------------------------
template <class T> symtensor4<T>::symtensor4 (int i, int j, int k, int l, bool init, T value)
//-------------------------------------------------------------------------------------------
{
  assert (i>0 && k>0);
  assert (i==j && k==l);

  mat = matrix<T> (.5*i*(i+1), .5*k*(k+1), init, value);
}


/*!
  \brief Partial memory allocation

  Allocates memory for the first components only. \n
  (don't use this function except if you know what you're doing !!)

  \param ntensor3 space dimension for the first component of the symmetric \f$ 4^\text{th} \f$ order tensor
*/

//----------------------------------------------------------
template <class T> void symtensor4<T>::create (int ntensor3)
//----------------------------------------------------------
{
  assert (ntensor3>0);

  int p = .5 * ntensor3 * (ntensor3 + 1);
  mat.create (p,p);
}

/*!
  \brief Full memory allocation without initialization of the tensor coordinates

  Allocates memory without initilizing the tensor components to zero.
  \param ntensor3 space dimension for the first component of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param ntensor2 space dimension for the second component of the symmetric \f$ 4^\text{th} \f$ order tensor (ntensor2 and ntensor3 must be equal)
  \param nrows space dimension for the third component of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param ncolumns space dimension for the fourth component of the symmetric \f$ 4^\text{th} \f$ order tensor (ncolumns and nrows must be equal)
*/

//-------------------------------------------------------------------------------------------------
template <class T> void symtensor4<T>::assign (int ntensor3, int ntensor2, int nrows, int ncolumns)
//-------------------------------------------------------------------------------------------------
{
  assert (ntensor3>0 && nrows>0);
  assert (ntensor3==ntensor2 && nrows==ncolumns);

  int p = .5 * ntensor3 * (ntensor3 + 1);
  int q = .5 * nrows * (nrows + 1);
  mat.assign (p,q);
}


/*!
  \brief Returns the reference of a coordinate for a symmetric \f$ 4^\text{th} \f$ order tensor

  Warning : we use the mathematical notations so the tensor coordinates are from 1 to the space dimension.
  \param i first coordinate of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param j second coordinate of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param k third coordinate of the symmetric \f$ 4^\text{th} \f$ order tensor
  \param l fourth coordinate of the symmetric \f$ 4^\text{th} \f$ order tensor

  \return coordinate \f$ T(i,j,k,l) \f$ of the symmetric \f$ 4^\text{th} \f$ order tensor \f$ \bold{T} \f$
*/

//---------------------------------------------------------------------------------
template <class T> T& symtensor4<T>::operator () (int i, int j, int k, int l) const
//---------------------------------------------------------------------------------
{
  assert (i>0 && i<=dim1());
  assert (j>0 && j<=dim2());
  assert (k>0 && k<=dim3());
  assert (l>0 && l<=dim4());

  int p, q;

  if (i==j) p = i;
  else 
    if (i < j) p = mat.Rows() - 0.5 * (dim1() -i) * (dim1() -i+1) + (j-i);
    else p = mat.Rows() - 0.5 * (dim1() -j) * (dim1() -j+1) + (i-j);

  if (k==l) q = k;
  else
    if (k < l) q = mat.Columns() - 0.5 * (dim3() -k) * (dim3() -k+1) + (l-k);
    else q = mat.Columns() - 0.5 * (dim3() -l) * (dim3() -l+1) + (k-l);

  return mat (p,q);
}


/*!
  \brief Addition for symmetric \f$ 4^\text{th} \f$ order tensors

  Computes the addition of the symmetric tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{ijkl} = a_{ijkl} + b_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.
  \param st1 symmetric tensor \f$ \boldsymbol{a} \f$
  \param st2 symmetric tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} + \boldsymbol{b} \f$
*/

//--------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator + (const symtensor4<Tf>& st1, const symtensor4<Tf>& st2)
//--------------------------------------------------------------------------------------------------
{
  symtensor4<Tf> st;
  st.mat = st1.mat + st2.mat;

  return st;
}


/*!
  \brief Subtraction for symmetric \f$ 4^\text{th} \f$ order tensors

  Computes the subtraction of the symmetric tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{ijkl} = a_{ijkl} - b_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.
  \param st1 symmetric tensor \f$ \boldsymbol{a} \f$
  \param st2 symmetric tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} - \boldsymbol{b} \f$
*/

//--------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator - (const symtensor4<Tf>& st1, const symtensor4<Tf>& st2)
//--------------------------------------------------------------------------------------------------
{
  symtensor4<Tf> st;
  st.mat = st1.mat - st2.mat;

  return st;
}


/*!
  \brief Add a symmetric \f$ 4^\text{th} \f$ order tensor

  Adds the symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} = (R_i) \f$ to the current symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} = (T_i) \f$ : \n
  \f$ T_i += R_i \f$  (which is equivalent to \f$ T_i = T_i + R_i \f$ but faster).

  \param st symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} \f$

  \return \f$  \boldsymbol{T} =  \boldsymbol{T} +  \boldsymbol{R} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf>& symtensor4<Tf>::operator += (const symtensor4<Tf>& st)
//----------------------------------------------------------------------------------------
{
  mat += st.mat;

  return *this;
}


/*!
  \brief Subtract a symmetric \f$ 4^\text{th} \f$ order tensor

  Subtracts the symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} = (R_i) \f$ to the current symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} = (T_i) \f$ : \n
  \f$ T_i -= R_i \f$  (which is equivalent to \f$ T_i = T_i - R_i \f$ but faster).

  \param st symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{R} \f$

  \return \f$  \boldsymbol{T} =  \boldsymbol{T} -  \boldsymbol{R} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf>& symtensor4<Tf>::operator -= (const symtensor4<Tf>& st)
//----------------------------------------------------------------------------------------
{
  mat -= st.mat;

  return *this;
}


/*!
  \brief Contracted product for symmetric \f$ 4^\text{th} \f$ order tensors

  Computes the contracted product of the symmetric tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c = a_{ijkl} b_{ijkl} \f$, where the result \f$ c \f$ is a scalar.

  \param st1 symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param st2 symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ c = \boldsymbol{a} | \boldsymbol{b} \f$
*/

//--------------------------------------------------------------------------------------
template <class Tf> Tf operator | (const symtensor4<Tf>& st1, const symtensor4<Tf>& st2)
//--------------------------------------------------------------------------------------
{
  assert (st1.dim1() && st1.dim3());
  assert (st1.dim1()==st2.dim1()); // tensors must have the same size
  assert (st1.dim3()==st2.dim3());

  Tf sum=0, subsum2=0, subsum4=0;

  for (int i=1; i<=st1.dim1(); i++)
    { for (int j=1; j<=st1.dim3(); j++)
	sum += st1.mat(i,j) * st2.mat(i,j);
      for (int j=st1.dim3()+1; j<=st1.mat.Columns(); j++)
	subsum2 += st1.mat(i,j) * st2.mat(i,j);
    }

  for (int i=st1.dim1()+1; i<=st1.mat.Rows(); i++)
    { for (int j=1; j<=st1.dim3(); j++)
	subsum2 += st1.mat(i,j) * st2.mat(i,j);
      for (int j=st1.dim3()+1; j<=st1.mat.Columns(); j++)
	subsum4 += st1.mat(i,j) * st2.mat(i,j);
    }

  return sum + 2*subsum2 + 4*subsum4;
}


/*!
  \brief Contracted product for symmetric \f$ 4^\text{th} \f$ order tensors

  Computes the contracted product of the symmetric tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{ijkl} = a_{ijpq} b_{pqkl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param st1 symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param st2 symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//---------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator || (const symtensor4<Tf>& st1, const symtensor4<Tf>& st2)
//---------------------------------------------------------------------------------------------------
{
  assert (st1.dim1() && st2.dim1());
  assert (st1.dim3());
  assert (st1.dim3() == st2.dim1());

  symtensor4<Tf> st (st1.dim1(), st1.dim1(), st2.dim3(), st2.dim3());

  for (int i=1; i<=st1.dim1(); i++)
    { for (int k=1; k<=st2.dim3(); k++)
	{ for (int j=1; j<=st1.dim3(); j++)
	    st.mat(i,k) += st1.mat(i,j) * st2.mat(j,k);
	  for (int j=st1.dim3()+1; j<=st1.mat.Columns(); j++)
	    st.mat(i,k) += 2 * st1.mat(i,j) * st2.mat(j,k);
	}

      for (int k=st2.dim3()+1; k<=st2.mat.Columns(); k++)
	{ for (int j=1; j<=st1.dim3(); j++)
	    st.mat(i,k) += st1.mat(i,j) * st2.mat(j,k);
	  for (int j=st1.dim3()+1; j<=st1.mat.Columns(); j++)
	    st.mat(i,k) += 2 * st1.mat(i,j) * st2.mat(j,k);
	}
    }

  for (int i=st1.dim1()+1; i<=st1.mat.Rows(); i++)
    { for (int k=1; k<=st2.dim3(); k++)
	{ for (int j=1; j<=st1.dim3(); j++)
	    st.mat(i,k) += st1.mat(i,j) * st2.mat(j,k);
	  for (int j=st1.dim3()+1; j<=st1.mat.Columns(); j++)
	    st.mat(i,k) += 2 * st1.mat(i,j) * st2.mat(j,k);
	}

      for (int k=st2.dim3()+1; k<=st2.mat.Columns(); k++)
	{ for (int j=1; j<=st1.dim3(); j++)
	    st.mat(i,k) += st1.mat(i,j) * st2.mat(j,k);
	  for (int j=st1.dim3()+1; j<=st1.mat.Columns(); j++)
	    st.mat(i,k) += 2 * st1.mat(i,j) * st2.mat(j,k);
	}
    }

  return st;
}


//=====Public methods for symtensor4 and scalars===============================


/*!
  \brief Multiplication by a scalar

  Multiply the symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ by a scalar a:
  \f$ t_{ijkl} *= a \f$  (which is equivalent to \f$ t_{ijkl} = t_{ijkl} * a \f$ but faster).

  \param elt scalar \f$ a \f$

  \return \f$  \boldsymbol{T} =  a \boldsymbol{T} \f$
*/

//-------------------------------------------------------------------------
template <class T> symtensor4<T>& symtensor4<T>::operator *= (const T& elt)
//-------------------------------------------------------------------------
{
  mat *= elt;

  return *this;
}


/*!
  \brief Multiplication for symmetric \f$ 4^\text{th} \f$ order tensor with scalar

  Computes the multiplication of the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ by a scalar \f$ a \f$ : \n
  \f$ \boldsymbol{T} * a = t_{ijkl} * a \f$.

  \param st symmetric  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param elt scalar \f$ a \f$

  \return \f$ \boldsymbol{T} * a \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator * (const symtensor4<Tf>& st, const Tf& elt)
//-------------------------------------------------------------------------------------
{
  symtensor4<Tf> stsr = st;

  return  stsr *= elt;
}


/*!
  \brief Multiplication for symmetric \f$ 4^\text{th} \f$ order tensor with scalar

  Computes the multiplication of the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ by a scalar \f$ a \f$ : \n
  \f$ a * \boldsymbol{T} = a * t_{ijkl} \f$.

  \param elt scalar \f$ a \f$
  \param st symmetric  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$

  \return \f$ a * \boldsymbol{T} \f$
*/

//-------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator * (const Tf& elt, const symtensor4<Tf>& st)
//-------------------------------------------------------------------------------------
{
  symtensor4<Tf> stsr = st;

  return  stsr *= elt;
}


//=====Public methods for symtensor4 and vectors===============================


/*!
  \brief Multiplication for symmetric \f$ 4^\text{th} \f$ order tensor with vector

  Computes the multiplication of the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ with the vector \f$ \boldsymbol{v} = v_i \f$ : \n
  \f$ \boldsymbol{T} * \boldsymbol{v} = t_{ijkl} * v_l \f$.

  \param st symmetric  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param v vector \f$ \boldsymbol{v} \f$

  \return \f$ \boldsymbol{T} * \boldsymbol{v} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const symtensor4<Tf>& st, const vector<Tf>& v)
//----------------------------------------------------------------------------------------
{
  assert (st.dim1() && st.dim3());
  assert (v.dim() == st.dim3());

  tensor3<Tf> t3 (st.dim1(), st.dim1(), st.dim3());

  for (int i=1; i<=st.dim1(); i++)
    for (int j=1; j<=st.dim1(); j++)
      for (int k=1; k<=st.dim3(); k++)
	for (int l=1; l<=st.dim3(); l++)
	  t3[i][j][k] += st (i,j,k,l) * v[l];

  return t3;
}


/*!
  \brief Multiplication for vector with symmetric \f$ 4^\text{th} \f$ order tensor

  Computes the multiplication of the vector \f$ \boldsymbol{v} = v_i \f$ with the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ : \n
  \f$ \boldsymbol{v} * \boldsymbol{T} = v_i * t_{ijkl} \f$.

  \param v vector \f$ \boldsymbol{v} \f$
  \param st symmetric  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$

  \return \f$ \boldsymbol{v} * \boldsymbol{T} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const vector<Tf>& v, const symtensor4<Tf>& st)
//----------------------------------------------------------------------------------------
{
  assert (st.dim1() && st.dim3());
  assert (v.dim() == st.dim1());

  tensor3<Tf> t3 (st.dim1(), st.dim3(), st.dim3());

  for (int j=1; j<=st.dim1(); j++)
    for (int k=1; k<=st.dim3(); k++)
      for (int l=1; l<=st.dim3(); l++)
	for (int i=1; i<=st.dim1(); i++)
	  t3[j][k][l] += st (i,j,k,l) * v[i];

  return t3;
}


/*!
  \brief Multiplication for symmetric \f$ 4^\text{th} \f$ order tensor with vector

  Computes the multiplication of the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ with the vector \f$ \boldsymbol{v} = v_i \f$ : \n
  \f$ \boldsymbol{T} | \boldsymbol{v} = t_{ijkl} * v_k = t_{ijkl} v_l\f$, by using the symmetry property \f$ t_{ijkl} = t_{ijlk} \f$.

  \param st symmetric  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param v vector \f$ \boldsymbol{v} \f$

  \return \f$ \boldsymbol{T} | \boldsymbol{v} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator | (const symtensor4<Tf>& st, const vector<Tf>& v)
//----------------------------------------------------------------------------------------
{
  return st * v;
}


/*!
  \brief Multiplication for vector with symmetric \f$ 4^\text{th} \f$ order tensor

  Computes the multiplication of the vector \f$ \boldsymbol{v} = v_i \f$ with the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ : \n
  \f$ \boldsymbol{v} | \boldsymbol{T} = v_j * t_{ijkl} = v_i * t_{ijkl} \f$, by using the symmetry property \f$ t_{ijkl} = t_{jikl} \f$.

  \param v vector \f$ \boldsymbol{v} \f$
  \param st symmetric  \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$

  \return \f$ \boldsymbol{v} | \boldsymbol{T} \f$
*/

//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator | (const vector<Tf>& v, const symtensor4<Tf>& st)
//----------------------------------------------------------------------------------------
{
  return v * st;
}


/*!
  \brief Multiplication for symmetric \f$ 4^\text{th} \f$ order tensor and vector

  Computes the multiplication of the symmetric tensor \f$ \boldsymbol{T} = (t_{ijkl}) \f$ with the vector \f$ \boldsymbol{v} = v_i \f$ for a given component \f$ n \f$ of the tensor.

  \param ts  symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{T} \f$
  \param v vector \f$ \boldsymbol{v} \f$
  \param i component n

  \return \f$ \boldsymbol{v} * \boldsymbol{T} \f$ (if n=1), \f$ \boldsymbol{v} | \boldsymbol{T} \f$ (if n=2), \f$ \boldsymbol{T} | \boldsymbol{v} \f$ (if n=3), \f$ \boldsymbol{T} * \boldsymbol{v} \f$ (if n=4)
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> tensor_sum (const symtensor4<Tf>& st, const vector<Tf>& v, int i)
//-----------------------------------------------------------------------------------------------
{
  assert (i>=1 && i<=4);

  tensor3<Tf> tr;
  if (i<3) tr =& (v*st);
  else tr =& (v*st);

  return tr;
}


//=====Public methods for symtensor4 and tensor2===============================


/*!
  \brief Contracted product for symmetric \f$ 4^\text{th} \f$ order tensor with \f$ 2^\text{nd} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{kl}) \f$ : \n
  \f$ c_{ij} = a_{ijkl} b_{kl} \f$, where \f$ \boldsymbol{c} = (c_{ij}) \f$.

  \param st symmetric\f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param ts2 \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> symtensor2<Tf> operator || (const symtensor4<Tf>& st, const tensor2<Tf>& ts2)
//-----------------------------------------------------------------------------------------------
{
  assert (st.dim1());
  assert (st.dim3() == ts2.dim1());
  assert (ts2.dim1() == ts2.dim2());

  int n = ts2.dim1();

  // symtensor2 to vector conversion
  vector<Tf> v (st.mat.Columns(), false);
  for (int i=1,p=1; i<=n; i++)
    { v[i] = ts2(i,i);
      for (int j=i+1; j<=n; j++, p++)
	v[n+p] = ts2(i,j) + ts2(j,i);
    }

  // matrix vector product
  vector<Tf> vv (st.mat.Rows(), false);
  vv = st.mat * v;

  // vector to symtensor2 conversion
  int m = st.dim1();
  symtensor2<Tf> sts2 (m, false);
  for (int i=1,p=1; i<=m; i++)
    { sts2(i,i) = vv[i];
      for (int j=i+1; j<=m; j++, p++)
	sts2(j,i) = vv[m+p];
    }

  return sts2;
}


/*!
  \brief Contracted product for \f$ 2^\text{nd} \f$ order tensor with symmetric \f$ 4^\text{th} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{ij}) \f$ : \n
  \f$ c_{kl} = b_{ij} a_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{kl}) \f$.

  \param t2 \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{b} \f$
  \param st symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{b} :: \boldsymbol{a} \f$
*/

//-----------------------------------------------------------------------------------------------
template <class Tf> symtensor2<Tf> operator || (const tensor2<Tf>& ts2, const symtensor4<Tf>& st)
//-----------------------------------------------------------------------------------------------
{
  assert (st.dim1());
  assert (st.dim1() == ts2.dim1());
  assert (ts2.dim1() == ts2.dim2());

  int n = ts2.dim1();

  // symtensor2 to vector conversion
  vector<Tf> v (st.mat.Rows(), false);
  for (int i=1,p=1; i<=n; i++)
    { v[i] = ts2(i,i);
      for (int j=i+1; j<=n; j++, p++)
	v[n+p] = ts2(i,j) + ts2(j,i);
    }

  // matrix vector product
  vector<Tf> vv (st.mat.Columns(), false);
  vv = v * st.mat;

  // vector to symtensor2 conversion
  int m = st.dim3();
  symtensor2<Tf> sts2 (m, false);
  for (int i=1,p=1; i<=m; i++)
    { sts2(i,i) = vv[i];
      for (int j=i+1; j<=m; j++, p++)
	sts2(j,i) = vv[m+p];
    }

  return sts2;
}


//=====Public methods for symtensor4 and symtensor2============================


/*!
  \brief Tensorial product for symmetric \f$ 2^\text{nd} \f$ order tensors

  Computes the tensorial product for the tensors \f$ \boldsymbol{a} = (a_{ij}) \f$ and \f$ \boldsymbol{b} = (b_{kl}) \f$ : \n
  \f$ c_{ijkl} = a_{ij} b_{kl} \f$, where \f$ \boldsymbol{c} = (c_{ijkl}) \f$.

  \param st1 symmetric \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param st2 symmetric \f$ 2^\text{nd} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} \times \boldsymbol{b} \f$
*/

//--------------------------------------------------------------------------------------------------
template <class Tf> symtensor4<Tf> operator ^ (const symtensor2<Tf>& st1, const symtensor2<Tf>& st2)
//--------------------------------------------------------------------------------------------------
{
  assert (st1.dim1() && st2.dim1());

  int n = st1.dim1();
  int m = st2.dim1();

  vector<Tf> v1 (0.5*n*(n+1), false), v2 (0.5*m*(m+1), false);

  for (int i=1,p=1; i<=n; i++)
    { v1[i] = st1(i,i);
      for (int j=i+1; j<=n; j++, p++)
	v1[n+p] = st1(i,j);
    }

  for (int i=1,p=1; i<=m; i++)
    { v2[i] = st2(i,i);
      for (int j=i+1; j<=m; j++, p++)
	v2[m+p] = st2(i,j);
    }

  return (v1 ^ v2);
}


//=====Public methods for symtensor4 and tensor3===============================


/*!
  \brief Contracted product for \f$ 3^\text{rd} \f$ order tensor with symmetric \f$ 4^\text{th} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{n,ij}) \f$ and \f$ \boldsymbol{b} = (b_{ijkl}) \f$ : \n
  \f$ c_{n,kl} = a_{n,ij} b_{ijkl} \f$, where \f$ \boldsymbol{c} = (c_{n,kl}) \f$.

  \param t3 \f$ 3^\text{rd} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param st symmetric\f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator || (const tensor3<Tf>& t3, const symtensor4<Tf>& st)
//-------------------------------------------------------------------------------------------
{
  assert (t3.dim1() && st.dim1());
  assert ( (t3.dim2() == st.dim1()) && (t3.dim3() == st.dim2()) );

  tensor3<Tf> ts;
  ts.create(t3.dim1());

  for (int n=1; n<=t3.dim1(); n++)
    ts[n] = (t3[n] || st);

  return ts;
}


/*!
  \brief Contracted product for symmetric \f$ 4^\text{th} \f$ order tensor with \f$ 3^\text{rd} \f$ order tensor

  Computes the contracted product of the tensors \f$ \boldsymbol{a} = (a_{ijkl}) \f$ and \f$ \boldsymbol{b} = (b_{n,kl}) \f$ : \n
  \f$ c_{n,ij} = a_{ijkl} b_{n,kl} \f$, where \f$ \boldsymbol{c} = (c_{n,ij}) \f$.

  \param st symmetric \f$ 4^\text{th} \f$ order tensor \f$ \boldsymbol{a} \f$
  \param t3 \f$ 3^\text{rd} \f$ order tensor \f$ \boldsymbol{b} \f$

  \return \f$ \boldsymbol{c} = \boldsymbol{a} :: \boldsymbol{b} \f$
*/

//-------------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator || (const symtensor4<Tf>& st, const tensor3<Tf>& t3)
//-------------------------------------------------------------------------------------------
{
  assert (t3.dim1() && st.dim1());
  assert ( (t3.dim2() == st.dim3()) && (t3.dim3() == st.dim4()) );

  tensor3<Tf> ts;
  ts.create(t3.dim1());

  for (int n=1; n<=t3.dim1(); n++)
    ts[n] = (st || t3[n]);

  return ts;
}


//=====Public methods for symtensor4 and tensor4===============================


/*!
  \brief Operator = for symmetric \f$ 4^\text{th} \f$ order tensor and \f$ 4^\text{th} \f$ order tensor with different size

  If the size of the right hand side tensor is zero, then we delete the table of coordinates of the left hand side symmetric tensor. \n
  Elsewhere, we transform the \f$ 4^\text{th} \f$ order tensor into a \ref abaqus::symtensor2matrix "symmetric tensor" so that we can use the same operator &= defined for matrices to copy its coordinates.

  \param tsr \f$ 4^\text{th} \f$ order tensor to copy

  \return reference of the left hand side symmetric \f$ 4^\text{th} \f$ order tensor (for multiple equalities)

  \sa mol::matrix::operator&=(const matrix&)
*/

//----------------------------------------------------------------------------------
template <class T> symtensor4<T>& symtensor4<T>::operator &= (const tensor4<T>& tsr)
//----------------------------------------------------------------------------------
{
  if (!tsr.dim1()) mat &= matrix<T> ();
  else mat &= symtensor4(tsr).mat;

  return *this;
}


/*!
  \brief Destructive operator = for symmetric \f$ 4^\text{th} \f$ order tensors and \f$ 4^\text{th} \f$ order tensor with different size

  If the size of the right hand side tensor is zero, then we delete the table of coordinates of the left hand side symmetric tensor. \n
  Elsewhere, we transform the \f$ 4^\text{th} \f$ order tensor into a \ref abaqus::symtensor2matrix "symmetric tensor" so that we can use the same operator = defined for matrices to copy its coordinates.

  \param tsr \f$ 4^\text{th} \f$ order tensor address

  \return reference of the left hand side symmetric \f$ 4^\text{th} \f$ order tensor (for multiple equalities)

  \sa mol::matrix::operator&=(const matrix*)
*/

//----------------------------------------------------------------------------
template <class T> symtensor4<T>& symtensor4<T>::operator &= (tensor4<T>* tsr)
//----------------------------------------------------------------------------
{
  if (!tsr->dim1()) mat &=& matrix<T> ();
  else mat &=& symtensor4(*tsr).mat;

  (*tsr).~tensor4<T>();

  return *this;
}


//=====Public methods for symtensor4===========================================


/*!
  \brief Computes symmetric \f$ 4^\text{th} \f$ order identity tensor

  \param n space dimension

  \return Identity tensor
*/

//-----------------------------------------------
template <class Tf> symtensor4<Tf> Id4sym (int n)
//-----------------------------------------------
{
  assert (n > 0);
  int k = 0.5 * n * (n+1);
  
  symtensor4<Tf> I;
  I.mat = Id2s<Tf> (k);
  for (int i=n+1; i<=k; i++)
    I.mat(i,i) *= 0.5;

  return I;
}


//-----------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const symtensor4<Tf>& st)
//-----------------------------------------------------------------------------
{
  assert (st.dim1());

  for (int i=1; i<=st.dim1(); i++)
    { for (int j=1; j<=st.dim3(); j++)
	s << st.mat(i,j) << " ";
      for (int j=st.dim3()+1; j<=st.mat.Columns(); j++)
	s << sqrt(2) * st.mat(i,j) << " ";
      s << endl;
    }

  for (int i=st.dim1()+1; i<=st.mat.Rows(); i++)
    { for (int j=1; j<=st.dim3(); j++)
	s << sqrt(2) * st.mat(i,j) << " ";
      for (int j=st.dim3()+1; j<=st.mat.Columns(); j++)
	s << 2 * st.mat(i,j) << " ";
      s << endl;
    }

  s << endl;

  return s;
}


//-----------------------------------------------------------------------------
template <class Tf> istream& operator >> (istream& s, const symtensor4<Tf>& st)
//-----------------------------------------------------------------------------
{
  assert (st.dim1());

  Tf value;

  for (int i=1; i<=st.dim1(); i++)
    { for (int j=1; j<=st.dim3(); j++)
	s >> st.mat(i,j);
      for (int j=st.dim3()+1; j<=st.mat.Columns(); j++)
	{ s >> value;
	  st.mat(i,j) = 0.5 * sqrt(2) * value;
	}
    }

  for (int i=st.dim1()+1; i<=st.mat.Rows(); i++)
    { for (int j=1; j<=st.dim3(); j++)
	{ s >> value;
	  st.mat(i,j) = 0.5 * sqrt(2) * value;
	}
      for (int j=st.dim3()+1; j<=st.mat.Columns(); j++)
	{ s >> value;
	  st.mat(i,j) = 0.5 * value;
	}
    }


  return s;
}


/*!
  \brief Writes symmetric \f$ 4^\text{th} \f$ order tensor to hard disk.

  \param filename path and filename
*/

//------------------------------------------------------------
template <class T> void symtensor4<T>::save (const char* path)
//------------------------------------------------------------
{
  assert (dim1());

  ofstream file(path, ios::out);
  assert (!file.fail());
  file << mat;
  file.close();
}


/*!
  \brief Writes symmetric \f$ 4^\text{th} \f$ order tensor to hard disk.

  \param filename path and filename
*/

//------------------------------------------------------------------
template <class T> void symtensor4<T>::save (const string& filename)
//------------------------------------------------------------------
{
  const char *path = filename.c_str();
  save (path);
}


/*!
  \brief Reads symmetric \f$ 4^\text{th} \f$ order tensor from hard disk.

  \param filename path and filename
*/

//------------------------------------------------------------
template <class T> void symtensor4<T>::read (const char* path)
//------------------------------------------------------------
{
  assert (dim1());

  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> mat;
  file.close();
}


/*!
  \brief Reads symmetric \f$ 4^\text{th} \f$ order tensor from hard disk.

  \param filename path and filename
*/

//------------------------------------------------------------------
template <class T> void symtensor4<T>::read (const string& filename)
//------------------------------------------------------------------
{
  const char *path = filename.c_str();
  read (path);
}


//---------------------------------------------------------------
template <class T> symtensor4<T>& symtensor4<T>::approximation ()
//---------------------------------------------------------------
{
  mat.approximation ();

  return *this;
}


}


#endif
