/***************************************************************
 *                 Mathematical Object Library                 *
 *  class integration : declaration for numerical integration  *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2002,2003,2004,2005,2006,2008,2009,2012 COLLARD Christophe
 * copyright © 2002,2003,2004,2005,2006,2008,2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2002,2003,2004,2005,2006,2008,2009 Arts et Métiers ParisTech
 * copyright © 2002,2003,2004,2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2002,2003,2004,2005,2006,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2002,2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::integration
    \brief numerical integration library \n

    \htmlonly 
    <FONT color="#838383">

    integration belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006, 2008, 2009, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008, 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2002-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type integration
#endif

#ifndef __integration_hpp
#define __integration_hpp


#ifndef __maths_hpp
#include "maths.hpp"
#endif

#ifndef __vectors_hpp
#include "vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "matrix.hpp"
#endif

#ifndef __linear_systems_hpp
#include "linear systems.hpp"
#endif

#ifndef __legendre_hpp
#include "legendre.hpp"
#endif

#ifndef __functions_hpp
#include "functions.hpp"
#endif

using namespace std;

namespace mol
{


//==================================
template <class T> class integration
//==================================
{
  private :
    int n;  // number of points for the numerical integration
    vector<T> pts, wgts; // integration points & weights on the interval [0;1]

  protected :
    vector<T> Points, Weights; // points and weights for the numerical integration

  public :
    integration (); // default constructor
    void Gauss (int, T, T, long double = precision); // computes a Gaussian integration with n points in [a;b]
    vector<T> points () const;   // return the integration points for the selected numerical integration method
    T point (int) const;         // return the integration point Xi for the selected numerical integration method
    vector<T> weights () const;  // return the integration weights for the selected numerical integration method
    T weight (int) const;        // return the integration weights Wi for the selected numerical integration method
    int NbIntPts() const {return n;}
};


//=====Private methods for integration===========================================


//=====Public methods for integration============================================


/*!
  \brief default constructor : initializes the number of integration points to zero
*/

//-----------------------------------------------
template <class T> integration<T>::integration ()
//-----------------------------------------------
{
  n = 0;
}


/*!
  \brief Computes Gauss-Legendre points and weights
  \param nb number of integration points
  \param a start point for the integration interval [a,b]
  \param b end point for the integration interval [a,b]
*/

//-------------------------------------------------------------------------------
template <class T> void integration<T>::Gauss (int nb, T a, T b, long double eps)
//-------------------------------------------------------------------------------
{
  assert (eps > 0);
  assert (!n); // if n>0 then the integration method is already defined
  Points = pts = Weights = wgts = n = nb;

  // computes Gauss-Legendre points
  Legendre<T> L(n);
  T x, x1;
  // [a;b] -> [-1;1]
  T alpha = 0.5*(b+a);
  T beta  = 0.5*(b-a);

  for (int i=1; i<=(n+1)/2; i++) // symmetric points & weights
    { x = cos( pi() * (i - 0.25 ) / (n + .5)); // starting value for the Newton Algorithm
      for (x1=x+1; abs(x-x1) > eps;)
	{ x1 = x;
	  x -= L(x) / L.d_dx(x);
	}
      pts[i] = - (pts[n+1-i] = x);
      Points[i] = alpha - beta * x;     // store and translate the solution from [-1;+1] to [a;b]
      Points[n+1-i] = alpha + beta * x; // symmetric solution
      wgts[i] = wgts[n+1-i] = 2 / ( (1-power(x,2)) * power(L.d_dx(x),2) );
      Weights[i] = Weights[n+1-i] = wgts[i] * beta;
    }
}


/*!
  \brief gives the Gauss-Legendre integration points
  \return points : the Gauss-Legendre integration points
*/

//----------------------------------------------------------
template <class T> vector<T> integration<T>::points () const
//----------------------------------------------------------
{
  assert (n); // if n=0 the is no integration method defined

  return Points;
}


/*!
  \brief gives the i-th Gauss-Legendre integration point
  \return points[i] : the i-th Gauss-Legendre integration point
*/

//------------------------------------------------------
template <class T> T integration<T>::point (int i) const
//------------------------------------------------------
{
  assert (n); // if n=0 the is no integration method defined
  assert (i>0 && i<=n);

  return Points[i];
}


/*!
  \brief gives the Gauss-Legendre integration weights
  \return weights : the Gauss-Legendre integration weights
*/

//-----------------------------------------------------------
template <class T> vector<T> integration<T>::weights () const
//-----------------------------------------------------------
{
  assert (n); // if n=0 the is no integration method defined
  return Weights;
}


/*!
  \brief gives the i-th Gauss-Legendre integration weight
  \return weights[i] : the i-th Gauss-Legendre integration weight
*/

//-------------------------------------------------------
template <class T> T integration<T>::weight (int i) const
//-------------------------------------------------------
{
  assert (n); // if n=0 the is no integration method defined
  assert (i>0 && i<=n);
  return Weights[i];
}


}


#endif
