/***************************************************************
 *                 Mathematical Object Library                 *
 *      class tensor3 : declarations for 3rd order tensors     *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2004,2005,2006,2007,2008,2009,2012 COLLARD Christophe
 * copyright © 2004,2005,2006,2007,2008,2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2004,2005,2006,2007,2008,2009 Arts et Métiers ParisTech
 * copyright © 2004,2005,2006,2007 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2004,2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::tensor3
    \brief 3rd order tensor library \n

    \htmlonly 
    <FONT color="#838383">

    tensor3 belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006, 2007, 2008, 2009, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2004-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tensor3
#endif

#ifndef __tensors3_hpp
#define __tensors3_hpp


#ifndef __vectors_hpp
#include "vectors.hpp"
#endif

#ifndef __tensors2_hpp
#include "tensors2.hpp"
#endif

using namespace std;

namespace mol
{


//==============================
template <class T> class tensor3
//==============================
{
  private :
    // tensor3 are stored using tensor2
    tensor2<T> *tsr;
    int size;

  public :
    tensor3 ();                   // default constructor
    tensor3 (int, bool=true, T=0);           // this constructor allocates memory
    tensor3 (int, int, int, bool=true, T=0); // this constructor allocates memory
    tensor3 (const tensor3<T>&);  // copy constructor
    tensor3 (tensor3<T>*);        // copy constructor for temporary objects
    ~tensor3 ();                  // destructor

    virtual int dim1 () const {return size;}             // returns the tensor 1st component size : k
    virtual int dim2 () const {return tsr[0].dim1();}    // returns the tensor 2nd component size : i
    virtual int dim3 () const {return tsr[0].dim2();}    // returns the tensor 3rd component size : j
    virtual void create (int);          // partial memory allocation (don't use this function except if you know what you're doing !!)
    virtual void assign (int,int,int);  // allocates memory without initilizing the tensor components to zero (so use it with great care)
    friend bool operator ! (const tensor3<T>& t) {return !t.size;}

    virtual T&  operator () (int,int,int) const;        // returns the element Pk,ij : P(tensor2,line,column)
    tensor2<T>& operator [] (int) const;                // returns a tensor2 Pk,ij -> Mij
    virtual tensor3<T>& operator = (const tensor3<T>&); // defines equal operator for 3rd order tensors
    virtual tensor3<T>& operator = (tensor3<T>*);  // allocates the data from the right hand temporary object to the left hand object - no copy (deletes temporary object)
    virtual tensor3<T>& operator &= (const tensor3<T>&); // copies tensor3 without size check (use this operator with great care)
    virtual tensor3<T>& operator &= (tensor3<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class Tf> friend tensor3<Tf>  operator + (const tensor3<Tf>&, const tensor3<Tf>&); // overloads operator + for 3rd order tensors
    template <class Tf> friend tensor3<Tf>  operator - (const tensor3<Tf>&, const tensor3<Tf>&); // overloads operator - for 3rd order tensors
    template <class Tf> friend Tf           operator | (const tensor3<Tf>&, const tensor3<Tf>&);  // overloads operator | for 3rd order tensors T|P = Tijk Pijk
    tensor3<T>& operator += (const tensor3<T>&);  // overloads += operator for tensors
    tensor3<T>& operator -= (const tensor3<T>&);  // overloads -= operator for tensors
    template <class Tf> friend bool operator == (const tensor3<Tf>&, const tensor3<Tf>&);  // compares 2 tensors
    template <class Tf> friend bool operator != (const tensor3<Tf>&, const tensor3<Tf>&);
    // operations with scalars
    tensor3<T>& operator *= (const T&);
    tensor3<T>& operator /= (const T&);
    template <class Tf> friend tensor3<Tf> operator * (const tensor3<Tf>&, const Tf&);
    template <class Tf> friend tensor3<Tf> operator * (const Tf&, const tensor3<Tf>&);
    template <class Tf> friend tensor3<Tf> operator / (const tensor3<Tf>&, const Tf&);
    // operations with vectors
    template <class Tf> friend tensor2<Tf> operator * (const tensor3<Tf>&, const vector<Tf>&);   // M_ki = P_kij * V_j
    template <class Tf> friend tensor2<Tf> operator * (const vector<Tf>&, const tensor3<Tf>&);   // M_ij = P_kij * V_k
    template <class Tf> friend tensor2<Tf> operator | (const tensor3<Tf>&, const vector<Tf>&);   // M_kj = P_kij * V_i
    template <class Tf> friend tensor2<Tf> operator | (const vector<Tf>&, const tensor3<Tf>&);   // M_kj = P_kij * V_i
    template <class Tf> friend tensor2<Tf> tensor_sum (const tensor3<Tf>&, const vector<Tf>&, int);  // M = P : v
    // operations with tensor2
    template <class Tf> friend vector<Tf>  operator || (const tensor3<Tf>&, const tensor2<Tf>&);  // v_p = T_p,ij M_ij
    template <class Tf> friend tensor3<Tf> operator * (const tensor2<Tf>&, const tensor3<Tf>&);   // T_pij = Msh ru_pq * T_qij
    template <class Tf> friend tensor3<Tf> operator ^ (const tensor2<Tf>&, const vector<Tf>&);    // T_ijp = M_ij * V_p
    template <class Tf> friend tensor3<Tf> operator ^ (const vector<Tf>&, const tensor2<Tf>&);    // T_pij = V_p * M_ij
    // overloads iostreams
    template <class Tf> friend ostream& operator << (ostream&, const tensor3<Tf>&);  // overloads output stream for tensors
    template <class Tf> friend istream& operator >> (istream&, const tensor3<Tf>&);  // overloads output stream for tensors
    virtual void save (const char*);   // writes to disk
    virtual void save (const string&); // writes to disk
    virtual void read (const char*);   // reads from disk
    virtual void read (const string&); // reads from disk
    tensor3<T>& approximation ();
};


//=====Private methods for tensor3=============================================


//=====Public methods for tensor3==============================================


/*!
  \brief Default constructor

*/

//---------------------------------------
template <class T> tensor3<T>::tensor3 ()
//---------------------------------------
{
  size = 0;
}


/*!
  \brief Constructor

  \param sz space dimensions (\f$\Bbb{R}^{sz} \times \Bbb{R}^{sz} \times \Bbb{R}^{sz} \f$) for the \f$ 3^\text{rd} \f$ order tensor
  \param init boolean that indicates if the constructor has to initialize the tensor coordinates (default value is true)
  \param value value for tensor coordinates (default value is 0)
*/

//-----------------------------------------------------------------
template <class T> tensor3<T>::tensor3 (int sz, bool init, T value)
//-----------------------------------------------------------------
{
  assert (sz>0);

  tsr = new tensor2<T> [size=sz];
  for (int i=0; i<size; i++)
    tsr[i] =& matrix<T> (sz,sz,init,value);
}


//------------------------------------------------------------------------------------------------
template <class T> tensor3<T>::tensor3 (int ntensor2, int nrows, int ncolumns, bool init, T value)
//------------------------------------------------------------------------------------------------
{
  assert (nrows>0 && ncolumns>0 && ntensor2>0);

  tsr = new tensor2<T> [size=ntensor2];
  for (int i=0; i<size; i++)
    tsr[i] =& matrix<T> (nrows,ncolumns,init,value);
}


//-----------------------------------------------------------
template <class T> tensor3<T>::tensor3 (const tensor3<T>& tn)
//-----------------------------------------------------------
{
  assert (tn.dim2() && tn.dim3() && tn.size);

  tsr = new tensor2<T> [size=tn.size];
  for (int i=0; i<tn.size; i++)
      tsr[i] = tn.tsr[i];
}


//-----------------------------------------------------
template <class T> tensor3<T>::tensor3 (tensor3<T>* tn)
//-----------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).size);

  size = (*tn).size;
  tsr = (*tn).tsr;
  (*tn).size = 0;
}


//----------------------------------------
template <class T> tensor3<T>::~tensor3 ()
//----------------------------------------
{
  // destruction of the memory allocated for the tensor

  if (size) delete [] tsr; // free memory only if it's been affected
  size = 0;
}


//------------------------------------------------------
template <class T> void tensor3<T>::create (int ntensor)
//------------------------------------------------------
{
  assert (ntensor>0);
  assert (!size);

  tsr = new tensor2<T> [size = ntensor];
}


//-------------------------------------------------------------------------------
template <class T> void tensor3<T>::assign (int ntensor, int nrows, int ncolumns)
//-------------------------------------------------------------------------------
{
  assert (ntensor>0 && !size);

  tsr = new tensor2<T> [size = ntensor];
  for (int i=0; i<size; i++)
    tsr[i] =& matrix<T> (nrows,ncolumns,false);
}


//-----------------------------------------------------------------------
template <class T> T& tensor3<T>::operator () (int k, int i, int j) const
//-----------------------------------------------------------------------
{
  assert (size);
  assert ((k>0)&&(k<=size));

  return tsr[k-1](i,j);
}


//------------------------------------------------------------------
template <class T> tensor2<T>& tensor3<T>::operator [] (int k) const
//------------------------------------------------------------------
{
  assert ((k>0)&&(k<=size));
  return tsr[k-1];
}


/*!
  \brief Standard operator = for \f$ 3^\text{rd} \f$ order tensor

  If both tensors are fully defined (size + table of coordinates), it checks that the space dimension is the same for the first coordinate and then uses the operator = of \f$ 2^\text{nd} \f$ order tensors. If the left hand side tensor is partially defined (space dimension of the first coordinate of the tensor is zero) then it allocates memory for this tensor and copies the coordinates using operator = of \f$ 2^\text{nd} \f$ order tensors. \n

  \param tn \f$ 3^\text{rd} \f$ order tensor to copy

  \return reference of the left hand side tensor (for multiple equalities)

  \sa mol::tensor2::operator=(const tensor2&)
*/

//--------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator = (const tensor3<T>& tn)
//--------------------------------------------------------------------------
{
  assert (tn.dim2() && tn.dim3() && tn.size);

  if (!size) tsr = new tensor2<T> [size=tn.size];
  else assert (size==tn.size && dim2()==tn.dim2() && dim3()==tn.dim3());
  for (int i=0; i<size; i++) 
      tsr[i] = tn.tsr[i];

  return *this;
}


//--------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator = (tensor3<T>* tn)
//--------------------------------------------------------------------
{
  assert ((*tn).dim2() && (*tn).dim3() && (*tn).size);

  if (!size) size = (*tn).size;
  else
    { assert (size==(*tn).size && dim2()==(*tn).dim2() && dim3()==(*tn).dim3());
      delete [] tsr;
    }
  tsr = (*tn).tsr;
  (*tn).size = 0;

  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator &= (const tensor3<T>& tn)
//---------------------------------------------------------------------------
{
  if (!tn)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size != tn.size)
	{ if (size) delete [] tsr;
	  tsr = new tensor2<T> [size = tn.size];  // allocates memory with the = operator for tensor2 (no initialization)
	}
      for (int i=1; i<=size; i++)
	(*this)[i] &= tn[i];
    }

  return *this;
}


//---------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator &= (tensor3<T>* tn)
//---------------------------------------------------------------------
{
  if (!(*tn).size)
    { if (size) delete [] tsr;
      size = 0;
    }

  else
    { if (size)
	delete [] tsr;
      size = (*tn).size;
      tsr = (*tn).tsr;
      (*tn).size = 0;
    }

  return *this;
}


//---------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator + (const tensor3<Tf>& t1, const tensor3<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  assert(t1.dim1());
  tensor3<Tf> t = t1;
  return t+=t2;
}


//---------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator - (const tensor3<Tf>& t1, const tensor3<Tf>& t2)
//---------------------------------------------------------------------------------------
{
  assert (t1.dim1()==t2.dim1()); // tensors must have the same size
  assert(t1.dim1());
  tensor3<Tf> t = t1;
  return t-=t2;
}


//------------------------------------------------------------------------------
template <class Tf> Tf operator | (const tensor3<Tf>& t1, const tensor3<Tf>& t2)
//------------------------------------------------------------------------------
{
  assert (t1.size==t2.size); // tensors must have the same size
  assert (t1.size);

  Tf sum=0;
  for (int i=1; i<=t1.size; i++)
    sum += (t1[i] | t2[i]);

  return sum;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator += (const tensor3<T>& ts)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor must be defined
  assert (size == ts.size);  // tensors must have the same size

  for (int i=0; i<size; i++)
    tsr[i] += ts.tsr[i];

  return *this;
}


//---------------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator -= (const tensor3<T>& ts)
//---------------------------------------------------------------------------
{
  assert (size);  // the tensor must be defined
  assert (size == ts.size);  // tensors must have the same size

  for (int i=0; i<size; i++)
    tsr[i] -= ts.tsr[i];

  return *this;
}


//---------------------------------------------------------------------------------
template <class Tf> bool operator == (const tensor3<Tf>& t1, const tensor3<Tf>& t2)
//---------------------------------------------------------------------------------
{
  assert (t1.size);

  int result=(t1.size==t2.size);
  for (int i=1; (i<=t1.size)&&(result); i++)
    result *= (t1[i]==t2[i]);

  return result;
}


//---------------------------------------------------------------------------------
template <class Tf> bool operator != (const tensor3<Tf>& t1, const tensor3<Tf>& t2)
//---------------------------------------------------------------------------------
{
  return !(t1 == t2);
}


//-------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator *= (const T& elt)
//-------------------------------------------------------------------
{
  assert (size);

  for (int i=0; i<size; i++)
    tsr [i] *= elt;

  return *this;
}


//-------------------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::operator /= (const T& elt)
//-------------------------------------------------------------------
{
  assert (size);
  assert (elt);

  for (int i=0; i<size; i++)
    tsr [i] /= elt;

  return *this;
}


//-------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const tensor3<Tf>& ts, const Tf& elt)
//-------------------------------------------------------------------------------
{
  assert (ts.size);
  tensor3<Tf> tr = ts;
  return tr *= elt;
}


//-------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const Tf& elt, const tensor3<Tf>& ts)
//-------------------------------------------------------------------------------
{
  assert (ts.size);
  return ts * elt;
}


//-------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator / (const tensor3<Tf>& ts, const Tf& elt)
//-------------------------------------------------------------------------------
{
  assert (ts.size);
  assert (elt);
  tensor3<Tf> tr = ts;
  return tr /= elt;
}


//-------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> operator * (const tensor3<Tf>& ts, const vector<Tf>& v)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim3());

  tensor2<Tf> m(ts.dim1(),ts.dim2());
  for (int k=1; k<=ts.size; k++)
    m[k] += ts[k] * v;

  return m;
}


//-------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> operator * (const vector<Tf>& v, const tensor3<Tf>& ts)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim1());

  tensor2<Tf> m(ts.dim2(),ts.dim3());
  for (int k=1; k<=ts.size; k++)
    m += ts[k] * v[k];

  return m;
}


//-------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> operator | (const tensor3<Tf>& ts, const vector<Tf>& v)
//-------------------------------------------------------------------------------------
{
  assert (v.dim());
  assert (v.dim() == ts.dim2());

  tensor2<Tf> m(ts.dim1(),ts.dim3());
  for (int k=1; k<=ts.size; k++)
    m[k] += v * ts[k];

  return m;
}


//-------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> operator | (const vector<Tf>& v, const tensor3<Tf>& ts)
//-------------------------------------------------------------------------------------
{
  return ts|v;
}


//--------------------------------------------------------------------------------------------
template <class Tf> tensor2<Tf> tensor_sum (const tensor3<Tf>& ts, const vector<Tf>& v, int i)
//--------------------------------------------------------------------------------------------
{
  assert (i>=1 && i<=3);

  tensor2<Tf> m;
  if (i==1) m =& (v*ts);
  else if (i==2) m =& (ts|v);
       else m =& (ts*v);

  return m;
}


//----------------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator || (const tensor3<Tf>& ts, const tensor2<Tf>& mat)
//----------------------------------------------------------------------------------------
{
  assert (mat.Rows());
  assert (ts.size);

  vector<Tf> v(ts.size,false);
  for (int i=1; i<=ts.size; i++)
    v[i] = (ts[i] | mat);

  return v;
}


//----------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator * (const tensor2<Tf>& mat, const tensor3<Tf>& ts)
//----------------------------------------------------------------------------------------
{
  assert (mat.Rows());
  assert (mat.Columns() == ts.dim1());

  tensor3<Tf> tsr (mat.Rows(), ts.dim2(), ts.dim3());

  for (int p=1; p<=mat.Rows(); p++)
    for (int q=1; q<=ts.dim1(); q++)
      for (int i=1; i<=ts.dim2(); i++)
	for (int j=1; j<=ts.dim3(); j++)
	  tsr (p,i,j) += mat(p,q) * ts(q,i,j);

  return tsr;
}


//-------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator ^ (const tensor2<Tf>& ts, const vector<Tf>& v)
//-------------------------------------------------------------------------------------
{
  assert (ts.dim1());
  assert (v.dim());

  tensor3<Tf> tsr (ts.dim1(), ts.dim2(), v.dim());

  for (int i=1; i<=ts.dim1(); i++)
    for (int j=1; j<=ts.dim2(); j++)
      for (int p=1; p<=v.dim(); p++)
	tsr (i,j,p) = ts(i,j) * v[p];

  return tsr;
}


//-------------------------------------------------------------------------------------
template <class Tf> tensor3<Tf> operator ^ (const vector<Tf>& v, const tensor2<Tf>& ts)
//-------------------------------------------------------------------------------------
{
  assert (ts.dim1());
  assert (v.dim());

  tensor3<Tf> tsr (v.dim(), ts.dim1(), ts.dim2());

  for (int p=1; p<=v.dim(); p++)
    for (int i=1; i<=ts.dim1(); i++)
      for (int j=1; j<=ts.dim2(); j++)
	tsr (p,i,j) = ts(i,j) * v[p];

  return tsr;
}


//------------------------------------------------------------------------
template <class T> ostream& operator << (ostream& s, const tensor3<T>& tn)
//------------------------------------------------------------------------
{
  assert (tn.size);

  for (int i=1; i<=tn.size; i++)
      s << tn[i];
  s << endl;

  return s;
}


//------------------------------------------------------------------------
template <class T> istream& operator >> (istream& s, const tensor3<T>& tn)
//------------------------------------------------------------------------
{
  assert (tn.size);

  for (int i=1; i<=tn.size; i++)
      s >> tn[i];

  return s;
}


/*!
  \brief Writes\f$ 3^\text{rd} \f$ order tensor to hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------
template <class T> void tensor3<T>::save (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ofstream file(path, ios::out);
  assert (!file.fail());
  file << *this;
  file.close();
}


/*!
  \brief Writes \f$ 3^\text{rd} \f$ order tensor to hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------------
template <class T> void tensor3<T>::save (const string& filename)
//---------------------------------------------------------------
{
  const char *path = filename.c_str();
  save (path);
}


/*!
  \brief Reads \f$ 3^\text{rd} \f$ order tensor from hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------
template <class T> void tensor3<T>::read (const char* path)
//---------------------------------------------------------
{
  assert (size);

  ifstream file(path, ios::in);
  assert (!file.fail());
  file >> *this;
  file.close();
}


/*!
  \brief Reads \f$ 3^\text{rd} \f$ order tensor from hard disk.

  \param filename path and filename
*/

//---------------------------------------------------------------
template <class T> void tensor3<T>::read (const string& filename)
//---------------------------------------------------------------
{
  const char *path = filename.c_str();
  read (path);
}


//---------------------------------------------------------
template <class T> tensor3<T>& tensor3<T>::approximation ()
//---------------------------------------------------------
{
  for (int i=0; i<size; i++)
    tsr[i].approximation();

  return *this;
}


}


#endif
