// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/builtins/commands.cc,v 1.11 2004/07/07 01:54:57 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/builtins/commands.hh>
#include <mpak/spec/fwd.hh>
#include <mpak/spec/command.hh>
#include <mpak/builtins/mpak_node.hh>
#include <mpak/builtins/config_node.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/package_node.hh>
#include <mpak/builtins/version_node.hh>
#include <mpak/builtins/dependency_node_data.hh>
#include <mpak/builtins/sources_node_data.hh>
#include <mpak/builtins/script_node_data.hh>
#include <mpak/builtins/installed_node_data.hh>
#include <mpak/builtins/option_node_data.hh>
#include <mpak/builtins/timestamp_node_data.hh>
#include <mpak/util/checksummer.hh>
#include <mpak/spec/node.hh>
#include <mpak/spec/context.hh>
#include <mpak/util/dependency.hh>
#include <mpak/util/file_stat.hh>
#include <mpak/util/node_path.hh>

#include <boost/shared_ptr.hpp>
#include <boost/variant.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/format.hpp>

#include <new>
#include <string>
#include <vector>
#include <list>
#include <iterator>
#include <cassert>
#include <cctype>
#include <cstdlib>

#include <sys/sysmacros.h>

#include <iostream>
#include <iomanip>

namespace mpak
{
    namespace builtins
    {
        namespace commands
        {
            namespace
            {
                inline bool
                test_condition (const boost::optional<std::string> &string)
                {
                    if (!string || (string->empty ()))
                        return false;
                    return true;
                }
            }
            
            boost::optional<std::string>
            mpak (const boost::shared_ptr<spec::node> &node,
                 const boost::shared_ptr<const spec::argument_vector> &arguments,
                 spec::context &context)
            {
                if (!arguments ||
                    arguments->size () == 0)
                    return boost::optional<std::string> (std::string ("t"));
                else if (arguments->size () > 2)
                    throw spec::bad_argument ("incorrect number of arguments passed to mpak call");
                
                bool require = true;
                boost::optional<std::string> argument;
                if (arguments->size () == 2) {
                    argument = context.reduce_argument (arguments->at (0), node);
                    if (argument && (*argument == "require")) {
                        require = true;
                    } else {
                        throw spec::bad_argument ("incorrect number of arguments passed to mpak call");
                    }
                    argument = context.reduce_argument (arguments->at (1), node);
                    if (!argument || argument->empty ()) {
                        throw spec::bad_argument ("empty string passed to mpak");
                    }
                }
                
                // TODO: check our version
                
                return boost::optional<std::string> ();
            }
            
            boost::optional<std::string>
            set (const boost::shared_ptr<spec::node> &node,
                 const boost::shared_ptr<const spec::argument_vector> &arguments,
                 spec::context &context)
            {
                if (!arguments ||
                    ((arguments->size () != 2)))
                    throw spec::bad_argument ("incorrect number of arguments passed to set call");
                
                boost::optional<std::string> key = context.reduce_argument (arguments->at (0), node);
                if (!key) {
                    throw spec::bad_argument ("empty argument passed for key in set call");
                }
                
                boost::optional<std::string> value (context.reduce_argument (arguments->at (1), node));
                if (!value) {
                    value = std::string ();
                }
                
                boost::shared_ptr<mpak_node> mpak_node (boost::dynamic_pointer_cast<mpak_node> (node));
                assert (mpak_node);
                
                mpak_node->set_env (*key, *value);
                return boost::optional<std::string> ();
            }

            namespace
            {
                boost::optional<std::string>
                get_helper (const std::string &key,
                            const boost::shared_ptr<const builtins::mpak_node> root,
                            const util::node_path &node_path,
                            const std::string &node_type,
                            bool must_exist)
                {
                    boost::optional<std::string> value;
                    
                    unsigned node_type_n; // category == 0, package == 1, version == 2
                    if (node_type == "mpak:category") {
                        node_type_n = 0;
                    } else if (node_type == "mpak:package") {
                        node_type_n = 1;
                    } else if (node_type == "mpak:version") {
                        node_type_n = 2;
                    } else {
                        throw spec::bad_command ("get must be called on one of mpak:category, mpak:package, mpak:version");
                    }
                    assert (node_type_n <= node_path.elements_size ());
                    util::node_path::elements_size_type num_category_elements (node_path.elements_size () - node_type_n);
                    
                    if (root->has_env (key)) {
                        value = root->get_env (key);
                    }
                    
                    boost::shared_ptr<const builtins::mpak_node> node (root);
                    for (util::node_path::elements_size_type n (0); n < num_category_elements; ++n) {
                        if (node->has_child ("mpak:category", node_path.element_at (n))) {
                            node = boost::dynamic_pointer_cast<const builtins::mpak_node> (node->get_child ("mpak:category",
                                                                                                            node_path.element_at (n)));
                            assert (node);
                        } else {
                            if (must_exist) {
                                assert (false);
                            } else {
                                return value;
                            }
                        }
                        
                        if (node->has_env (key)) {
                            value = node->get_env (key);
                        }
                    }
                    
                    if (node_type_n >= 1) { // package_node
                        if (node->has_child ("mpak:package", node_path.element_at (num_category_elements))) {
                            node = boost::dynamic_pointer_cast<const builtins::mpak_node> (node->get_child ("mpak:package",
                                                                                                            node_path.element_at (num_category_elements)));
                            assert (node);
                        } else {
                            if (must_exist) {
                                assert (false);
                            } else {
                                return value;
                            }
                        }
                        
                        if (node->has_env (key)) {
                            value = node->get_env (key);
                        }
                    }
                    
                    if (node_type_n >= 2) { // version_node
                        if (node->has_child ("mpak:version", node_path.element_at (num_category_elements + 1))) {
                            node = boost::dynamic_pointer_cast<const builtins::mpak_node> (node->get_child ("mpak:version",
                                                                                                            node_path.element_at (num_category_elements + 1)));
                            assert (node);
                        } else {
                            if (must_exist) {
                                assert (false);
                            } else {
                                return value;
                            }
                        }
                        
                        if (node->has_env (key)) {
                            value = node->get_env (key);
                        }
                    }
                    
                    return value;
                }
            }
            
            boost::optional<std::string>
            get::
            operator () (const boost::shared_ptr<spec::node> &node,
                         const boost::shared_ptr<const spec::argument_vector> &arguments,
                         spec::context &context)
                const
            {
                if (!arguments ||
                    ((arguments->size () != 1)))
                    throw spec::bad_argument ("incorrect number of arguments passed to get call");
                
                boost::optional<std::string> key = context.reduce_argument (arguments->at (0), node);
                if (!key) {
                    throw spec::bad_argument ("empty argument passed for key in get call");
                }
                
                util::node_path node_path (node);
                
                if (this->config_root_) {
                    boost::optional<std::string> value (get_helper (*key, this->config_root_, node_path, node->get_type (), false));
                    if (value)
                        return value;
                }
                
                boost::shared_ptr<const builtins::mpak_node> root (boost::dynamic_pointer_cast<const builtins::mpak_node> (node));
                assert (root);
                boost::optional<boost::weak_ptr<const spec::node> > opt_weak_parent;
                while (opt_weak_parent = root->get_parent ()) {
                    boost::shared_ptr<const spec::node> parent (*opt_weak_parent);
                    boost::shared_ptr<const builtins::mpak_node> parent_mpak_node (boost::dynamic_pointer_cast<const builtins::mpak_node> (parent));
                    assert (parent_mpak_node);
                    root = parent_mpak_node;
                }
                
                return get_helper (*key, root, node_path, node->get_type (), true);
            }
            
            boost::optional<std::string>
            if_else (const boost::shared_ptr<spec::node> &node,
                     const boost::shared_ptr<const spec::argument_vector> &arguments,
                     spec::context &context)
            {
                if (!arguments ||
                    ((arguments->size () != 2) &&
                     (arguments->size () != 3)))
                    throw spec::bad_argument ("incorrect number of arguments passed to if_else call");
                
                spec::argument_vector::const_iterator result_argument;
                boost::optional<std::string> condition = context.reduce_argument (arguments->at (0), node);
                if (test_condition (condition)) {
                    result_argument = arguments->begin () + 1;
                } else if (arguments->size () == 3) {
                    // if else part is given, use that
                    result_argument = arguments->begin () + 2;
                } else {
                    return boost::optional<std::string> ();
                }
                
                return context.reduce_argument (*result_argument, node);
            }
            
            boost::optional<std::string>
            return_ (const boost::shared_ptr<spec::node> &node,
                     const boost::shared_ptr<const spec::argument_vector> &arguments,
                     spec::context &context)
            {
                if (!arguments ||
                    (arguments->size () == 0)) {
                    return boost::optional<std::string> ();
                } else if (arguments->size () != 1) {
                    throw spec::bad_argument ("incorrect number of arguments passed to return call");
                }
                
                return *context.reduce_argument (arguments->at (0), node);
            }
            
            boost::optional<std::string>
            concat (const boost::shared_ptr<spec::node> &node,
                    const boost::shared_ptr<const spec::argument_vector> &arguments,
                    spec::context &context)
            {
                if (!arguments ||
                    (arguments->size () == 0)) {
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string> ret;
                for (spec::argument_vector::const_iterator i (arguments->begin ()); i != arguments->end (); ++i) {
                    boost::optional<std::string> str (context.reduce_argument (*i, node));
                    if (str) {
                        if (!ret) {
                            ret = str;
                        } else {
                            ret->append (*str);
                        }
                    }
                }
                return ret;
            }
            
            namespace tree
            {
                boost::optional<std::string>
                description (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () != 1)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to description call in node " +
                                                  util::node_path (node).get_string ());
                    }
                    boost::optional<std::string> description (context.reduce_argument (arguments->at (0), node));
                    
                    if (description) {
                        boost::shared_ptr<mpak_node> mpak_node (boost::dynamic_pointer_cast<mpak_node> (node));
                        assert (mpak_node);
                        
                        mpak_node->set_description (*description);
                    }
                    
                    return boost::optional<std::string> ();
                }

                boost::optional<std::string>
                slots (const boost::shared_ptr<spec::node> &node,
                       const boost::shared_ptr<const spec::argument_vector> &arguments,
                       spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () == 0))
                        throw spec::bad_argument ("incorrect number of arguments passed to slots call");
                    
                    if (node->get_type () != "mpak:package") {
                        throw spec::bad_command (std::string("slots called on node of type ") + node->get_type ());
                    }
                    
                    for (spec::argument_vector::const_iterator i = arguments->begin (); i != arguments->end (); ++i) {
                        const std::string slot (*context.reduce_argument (*i, node));
                        
                        boost::shared_ptr<package_node> package_node (boost::dynamic_pointer_cast<package_node> (node));
                        assert (package_node);
                        package_node->add_slot (slot);
                    }
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                slot (const boost::shared_ptr<spec::node> &node,
                      const boost::shared_ptr<const spec::argument_vector> &arguments,
                      spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () > 1))
                        throw spec::bad_argument ("incorrect number of arguments passed to slot call");
                    
                    if (node->get_type () != "mpak:version") {
                        throw spec::bad_command (std::string("slot called on node of type ") + node->get_type ());
                    }
                    
                    boost::shared_ptr<version_node> version_node (boost::dynamic_pointer_cast<version_node> (node));
                    assert (version_node);
                    
                    boost::optional<std::string> slot;
                    if (arguments->size () != 0) {
                        slot = context.reduce_argument (arguments->at (0), node);
                    }
                    version_node->set_slot (slot);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                dependency (const boost::shared_ptr<spec::node> &node,
                            const boost::shared_ptr<const spec::argument_vector> &arguments,
                            spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () < 1))
                        throw spec::bad_argument ("incorrect number of arguments passed to dependencies call");
                    
                    boost::optional<std::string> type = context.reduce_argument (arguments->front (), node);
                    if (!type) {
                        throw spec::bad_argument ("first argument of dependencies is empty");
                    }
                    
                    boost::shared_ptr<dependency_node_data> data;
                    if (!node->has_data ("mpak:dependency")) {
                        data.reset (new dependency_node_data);
                        node->add_data ("mpak:dependency", data);
                    } else {
                        data = boost::dynamic_pointer_cast<dependency_node_data> (node->get_data ("mpak:dependency"));
                        assert (data);
                    }
                    
                    for (spec::argument_vector::const_iterator i = ++arguments->begin (); i != arguments->end (); ++i) {
                        const boost::optional<std::string> dependency_string (*context.reduce_argument (*i, node));
                        if (!dependency_string) {
                            throw spec::bad_argument ("empty argument passed to dependencies");
                        }
                        
                        data->add_dependency (*type, util::dependency (*dependency_string));
                    }
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                source (const boost::shared_ptr<spec::node> &node,
                        const boost::shared_ptr<const spec::argument_vector> &arguments,
                        spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () < 1) ||
                        (arguments->size () > 2))
                        throw spec::bad_argument ("incorrect number of arguments passed to source call");
                    
                    boost::optional<std::string> opt_location = context.reduce_argument (arguments->front (), node);
                    if (!opt_location) {
                        throw spec::bad_argument ("first argument of source is empty");
                    }
                    std::string location (*opt_location);
                    if (location.empty ())
                        throw spec::bad_argument ("first argument of source is empty");
                    
                    std::string filename;
                    if (arguments->size () == 2) {
                        boost::optional<std::string> opt_filename = context.reduce_argument (arguments->at (1), node);
                        if (!opt_filename) {
                            throw spec::bad_argument ("second argument of source is empty");
                        }
                        filename = *opt_filename;
                        if (filename.find ('/') != std::string::npos) {
                            throw spec::bad_argument ("invalid filename argument");
                        }
                    } else {
                        if (location[0] == '@') {
                            throw spec::bad_argument ("a filename must be provided for a source file in another package");
                        } else {
                            std::string::size_type slashpos (location.rfind ('/'));
                            if (slashpos == std::string::npos) {
                                throw spec::bad_argument ("location \"" + location + "\" is not a url?");
                            } else {
                                filename.assign (location.begin() + slashpos + 1, location.end ());
                                if (filename.empty ()) {
                                    throw spec::bad_argument ("a filename must be provided for this url: " + location);
                                }
                            }
                        }
                    }
                    
                    boost::shared_ptr<sources_node_data> data;
                    if (!node->has_data ("mpak:source")) {
                        data.reset (new sources_node_data);
                        node->add_data ("mpak:source", data);
                    } else {
                        data = boost::dynamic_pointer_cast<sources_node_data> (node->get_data ("mpak:source"));
                        assert (data);
                    }
                    
                    data->add_source (filename, location);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                checksum (const boost::shared_ptr<spec::node> &node,
                          const boost::shared_ptr<const spec::argument_vector> &arguments,
                          spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () != 3))
                        throw spec::bad_argument ("incorrect number of arguments passed to checksum call");
                    
                    boost::optional<std::string> opt_filename = context.reduce_argument (arguments->front (), node);
                    if (!opt_filename) {
                        throw spec::bad_argument ("first argument of checksum is empty");
                    }
                    std::string filename (*opt_filename);
                    if (filename.empty ())
                        throw spec::bad_argument ("first argument of checksum is empty");
                    
                    boost::optional<std::string> opt_checksum_algo_string = context.reduce_argument (arguments->at (1), node);
                    if (!opt_checksum_algo_string) {
                        throw spec::bad_argument ("second argument of checksum is empty");
                    }
                    std::string checksum_algo_string (*opt_checksum_algo_string);
                    if (checksum_algo_string.empty ())
                        throw spec::bad_argument ("first argument of checksum is empty");
                    util::checksummer::algorithm checksum_algo (util::checksummer::string_to_algorithm (checksum_algo_string));
                    
                    std::string checksum_string;
                    boost::optional<std::string> opt_checksum_string = context.reduce_argument (arguments->at (2), node);
                    if (!opt_checksum_string) {
                        throw spec::bad_argument ("third argument of checksum is empty");
                    }
                    checksum_string = *opt_checksum_string;
                    if (checksum_string.empty ()) {
                        throw spec::bad_argument ("third argument of checksum is empty");
                    }
                    
                    util::checksummer::checksum_type checksum (util::checksummer::string_to_checksum (checksum_string));
                    
                    boost::shared_ptr<sources_node_data> data;
                    if (!node->has_data ("mpak:source")) {
                        data.reset (new sources_node_data);
                        node->add_data ("mpak:source", data);
                    } else {
                        data = boost::dynamic_pointer_cast<sources_node_data> (node->get_data ("mpak:source"));
                        assert (data);
                    }
                    
                    data->add_checksum (filename, std::make_pair (checksum_algo, checksum));
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                script (const boost::shared_ptr<spec::node> &node,
                        const boost::shared_ptr<const spec::argument_vector> &arguments,
                        spec::context &context)
                {
                    if (!arguments ||
                        (arguments->size () < 1) ||
                        (arguments->size () > 2))
                        throw spec::bad_argument ("incorrect number of arguments passed to script call");
                    
                    const boost::optional<std::string> name (context.reduce_argument (arguments->at (0), node));
                    if (!name || name->empty ())
                        throw spec::bad_argument ("script name is empty");
                    const boost::optional<std::string> script (context.reduce_argument (arguments->at (1), node));
                    if (!name || script->empty ())
                        throw spec::bad_argument ("script is empty");
                    
                    boost::shared_ptr<script_node_data> script_data;
                    if (node->has_data ("mpak:script")) {
                        script_data = boost::dynamic_pointer_cast<script_node_data> (node->get_data ("mpak:script"));
                        assert (script_data);
                    } else {
                        script_data.reset (new script_node_data);
                        node->add_data ("mpak:script", script_data);
                    }
                    
                    script_data->append_script (*name, *script);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                interpreter (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                {
                    if (arguments &&
                        (arguments->size () > 1)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to interpreter call");
                    }
                    
                    boost::optional<std::string> interpreter;
                    if (arguments) {
                        interpreter = *context.reduce_argument (arguments->at (0), node);
                        if (interpreter && interpreter->empty ()) {
                            throw spec::bad_argument ("empty string passed to interpreter");
                        }
                    }
                    
                    boost::shared_ptr<script_node_data> script_data;
                    if (node->has_data ("mpak:script")) {
                        script_data = boost::dynamic_pointer_cast<script_node_data> (node->get_data ("mpak:script"));
                        assert (script_data);
                    } else {
                        script_data.reset (new script_node_data);
                        node->add_data ("mpak:script", script_data);
                    }
                    
                    script_data->set_interpreter (interpreter);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                file_stat (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                {
                    const unsigned base_argument_count (5); // path type uid gid mode
                    
                    if (!arguments ||
                        (arguments->size () < base_argument_count)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                    }
                    
                    if (node->get_type () != "mpak:version") {
                        throw spec::bad_command (std::string ("file_stat called on node of type ") + node->get_type ());
                    }
                    
                    util::file_stat file_stat;
                    
                    spec::argument_vector::const_iterator i (arguments->begin ());
                    
                    boost::optional<std::string> opt_path_string (context.reduce_argument (*i, node));
                    if (!opt_path_string || opt_path_string->empty ()) {
                            throw spec::bad_argument ("empty string passed to file_stat for file type");
                    }
                    const std::string &path_string (*opt_path_string);
                    boost::filesystem::path path (path_string);
                    ++i;
                    
                    boost::optional<std::string> opt_file_type (context.reduce_argument (*i, node));
                    if (!opt_file_type || opt_file_type->empty ()) {
                            throw spec::bad_argument ("empty string passed to file_stat for file type");
                    }
                    const std::string &file_type (*opt_file_type);
                    ++i;
                    
                    boost::optional<std::string> opt_uid_string (context.reduce_argument (*i, node));
                    ++i;
                    if (!opt_uid_string || opt_uid_string->empty ()) {
                        throw spec::bad_argument ("empty string passed for file uid");
                    }
                    file_stat.uid = strtoul (opt_uid_string->c_str (), 0, 10);
                    
                    boost::optional<std::string> opt_gid_string (context.reduce_argument (*i, node));
                    ++i;
                    if (!opt_gid_string || opt_gid_string->empty ()) {
                        throw spec::bad_argument ("empty string passed for file gid");
                    }
                    file_stat.gid = strtoul (opt_gid_string->c_str (), 0, 10);
                    
                    boost::optional<std::string> opt_mode_string (context.reduce_argument (*i, node));
                    ++i;
                    if (!opt_mode_string || opt_mode_string->empty ()) {
                        throw spec::bad_argument ("empty string passed for file mode");
                    }
                    file_stat.mode = strtoul (opt_mode_string->c_str (), 0, 8);
                    
                    if (file_type == "directory") {
                        if (arguments->size () != (base_argument_count)) {
                            throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                        }
                        file_stat.type = util::file_stat::type_directory;
                    } else if ((file_type == "chardev") ||
                               (file_type == "blkdev")) {
                        if (arguments->size () != (base_argument_count + 2)) { // major minor
                            throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                        }
                        
                        if (file_type == "chardev")
                            file_stat.type = util::file_stat::type_chardev;
                        else
                            file_stat.type = util::file_stat::type_blkdev;
                        
                        boost::optional<std::string> opt_major_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_major_string || opt_major_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for device major number");
                        }
                        
                        boost::optional<std::string> opt_minor_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_minor_string || opt_minor_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for device minor number");
                        }
                        
                        unsigned major = strtoul (opt_major_string->c_str (), 0, 10);
                        unsigned minor = strtoul (opt_minor_string->c_str (), 0, 10);
                        file_stat.devnum = makedev (major, minor);
                        
                    } else if (file_type == "regular") {
                        if (arguments->size () != (base_argument_count + 4)) { // size mtime checksum_algo checksum
                            throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                        }
                        file_stat.type = util::file_stat::type_regular;
                        
                        boost::optional<std::string> opt_size_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_size_string || opt_size_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for file size");
                        }
                        file_stat.size = strtoul (opt_size_string->c_str (), 0, 10);
                        
                        boost::optional<std::string> opt_mtime_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_mtime_string || opt_mtime_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for file mtime");
                        }
                        file_stat.mtime = strtoul (opt_mtime_string->c_str (), 0, 10);
                        
                        boost::optional<std::string> opt_algo_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_algo_string || opt_algo_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for checksum algorithm");
                        }
                        file_stat.checksum_algo = util::checksummer::string_to_algorithm (*opt_algo_string);
                        
                        boost::optional<std::string> opt_checksum_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_checksum_string || opt_checksum_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for checksum");
                        }
                        file_stat.checksum = util::checksummer::string_to_checksum (*opt_checksum_string);
                        
                    } else if (file_type == "fifo") {
                        if (arguments->size () != (base_argument_count)) {
                            throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                        }
                        file_stat.type = util::file_stat::type_fifo;
                    } else if (file_type == "symlink") {
                        if (arguments->size () != (base_argument_count + 1)) { // symlink_target
                            throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                        }
                        file_stat.type = util::file_stat::type_symlink;
                        
                        boost::optional<std::string> opt_target_string (context.reduce_argument (*i, node));
                        ++i;
                        if (!opt_target_string || opt_target_string->empty ()) {
                            throw spec::bad_argument ("empty string passed for symlink target");
                        }
                        file_stat.symlink_target = *opt_target_string;
                        
                    } else if (file_type == "socket") {
                        if (arguments->size () != (base_argument_count)) {
                            throw spec::bad_argument ("incorrect number of arguments passed to file_stat call");
                        }
                        file_stat.type = util::file_stat::type_socket;
                    } else {
                        throw spec::bad_argument ("unknown file type passed to file_stat call");
                    }
                    
                    assert (i == arguments->end ());
                    
                    boost::shared_ptr<installed_node_data> installed_data;
                    if (node->has_data ("mpak:installed")) {
                        installed_data = boost::dynamic_pointer_cast<installed_node_data> (node->get_data ("mpak:installed"));
                        assert (installed_data);
                    } else {
                        installed_data.reset (new installed_node_data);
                        node->add_data ("mpak:installed", installed_data);
                    }
                    
                    installed_data->add_file_stat (path, file_stat);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                timestamp (const boost::shared_ptr<spec::node> &node,
                           const boost::shared_ptr<const spec::argument_vector> &arguments,
                           spec::context &context)
                {
                    if (node->get_type () != "mpak:version") {
                        throw spec::bad_command ("timestamp may only be called on mpak:version nodes");
                    }
                    
                    if (!arguments ||
                        (arguments->size () < 2)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to timestamp call");
                    }
                    
                    struct timeval tv;
                    boost::optional<std::string> opt_tv_sec_string (context.reduce_argument (arguments->at (0), node));
                    if (!opt_tv_sec_string || opt_tv_sec_string->empty ()) {
                        throw spec::bad_argument ("seconds argument of timestamp call is empty");
                    }
                    
                    boost::optional<std::string> opt_tv_usec_string (context.reduce_argument (arguments->at (1), node));
                    if (!opt_tv_usec_string || opt_tv_usec_string->empty ()) {
                        throw spec::bad_argument ("microseconds argument of timestamp call is empty");
                    }
                    
                    tv.tv_sec = strtoul (opt_tv_sec_string->c_str (), 0, 10);
                    tv.tv_usec = strtoul (opt_tv_usec_string->c_str (), 0, 10);
                    
                    boost::shared_ptr<builtins::timestamp_node_data> timestamp_node_data (new timestamp_node_data);
                    timestamp_node_data->set_timestamp (tv);
                    node->add_data ("mpak:timestamp", timestamp_node_data);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                option::
                operator () (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                    const
                {
                    if (arguments) {
                        boost::shared_ptr<option_node_data> option_data;
                        if (node->has_data (this->option_type_)) {
                            option_data = boost::dynamic_pointer_cast<option_node_data> (node->get_data (this->option_type_));
                            assert (option_data);
                        } else {
                            option_data.reset (new option_node_data);
                            node->add_data (this->option_type_, option_data);
                        }
                        
                        for (spec::argument_vector::const_iterator i (arguments->begin ()); i != arguments->end (); ++i) {
                            boost::optional<std::string> opt_flag (context.reduce_argument (*i, node));
                            if (!opt_flag || opt_flag->empty ()) {
                                throw spec::bad_argument ("empty string passed to " + this->option_type_ + "command");
                            }
                            option_data->add_flag (*opt_flag);
                        }
                    }
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                if_option::
                operator () (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                    const
                {
                    if (!arguments || arguments->size () < 2 || arguments->size () > 3) {
                        throw spec::bad_argument ("incorrect number of arguments passed to " + this->option_type_ + " call");
                    }
                    
                    boost::optional<std::string> opt_flag (context.reduce_argument (arguments->at (0), node));
                    if (!opt_flag) {
                        throw spec::bad_argument ("enable flag is empty");
                    }
                    
                    boost::shared_ptr<const spec::node> root (node);
                    boost::optional<boost::weak_ptr<const spec::node> > parent_node;
                    while (parent_node = root->get_parent ()) {
                        root = boost::shared_ptr<const spec::node> (*parent_node);
                    }
                    
                    util::node_path node_path (node);
                    boost::shared_ptr<builtins::option_node_data> option_node_data (collect_options (this->config_root_,
                                                                                                     node_path,
                                                                                                     node->get_type (),
                                                                                                     this->option_type_,
                                                                                                     false));
                    boost::shared_ptr<builtins::option_node_data> option_flags_node_data (collect_options (root,
                                                                                                           node_path,
                                                                                                           node->get_type (),
                                                                                                           this->option_flags_type_,
                                                                                                           true));
                    
                    if (option_flags_node_data->has_flag (*opt_flag)) {
                        spec::argument_vector::const_iterator result_argument;
                        if (option_node_data->has_flag (*opt_flag)) {
                            result_argument = arguments->begin () + 1;
                        } else if (arguments->size () == 3) {
                            result_argument = arguments->begin () + 2;
                        } else {
                            return boost::optional<std::string> ();
                        }
                        
                        return context.reduce_argument (*result_argument, node);
                    } else {
                        throw spec::bad_argument ("invalid enable option " + *opt_flag + " in node " + util::node_path (node).get_string ());
                    }
                }
                 
            }
            
            namespace config
            {
                boost::optional<std::string>
                package_tree_dir (const boost::shared_ptr<spec::node> &node,
                                  const boost::shared_ptr<const spec::argument_vector> &arguments,
                                  spec::context &context)
                {
                    if (node->get_type () != "mpak:config") {
                        throw spec::bad_command ("package_tree_dir can only be called in config");
                    }
                    if (!arguments ||
                        (arguments->size () != 1)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to package_tree_dir call");
                    }
                    boost::optional<std::string> path_string (*context.reduce_argument (arguments->at (0), node));
                    if (!path_string || path_string->empty ()) {
                        throw spec::bad_argument ("string passed to package_tree_dir has no value");
                    }
                    
                    boost::filesystem::path path (*path_string);
                    if (!path.is_complete ())
                        throw spec::bad_argument ("path passed to package_tree_dir is not absolute");
                    
                    boost::shared_ptr<config_node> config_node (boost::dynamic_pointer_cast<config_node> (node));
                    assert (config_node);
                    
                    config_node->set_package_tree_dir (path);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                statedb_dir (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                {
                    if (node->get_type () != "mpak:config") {
                        throw spec::bad_command ("statedb_dir can only be called in config");
                    }
                    if (!arguments ||
                        (arguments->size () != 1)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to statedb_dir call");
                    }
                    boost::optional<std::string> path_string (*context.reduce_argument (arguments->at (0), node));
                    if (!path_string || path_string->empty ()) {
                        throw spec::bad_argument ("string passed to statedb_dir has no value");
                    }
                    
                    boost::filesystem::path path (*path_string);
                    if (!path.is_complete ())
                        throw spec::bad_argument ("path passed to statedb_dir is not absolute");
                    
                    boost::shared_ptr<config_node> config_node (boost::dynamic_pointer_cast<config_node> (node));
                    assert (config_node);
                    
                    config_node->set_statedb_dir (path);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                mpaktmp_dir (const boost::shared_ptr<spec::node> &node,
                             const boost::shared_ptr<const spec::argument_vector> &arguments,
                             spec::context &context)
                {
                    if (node->get_type () != "mpak:config") {
                        throw spec::bad_command ("mpaktmp_dir can only be called in config");
                    }
                    if (!arguments ||
                        (arguments->size () != 1)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to mpaktmp_dir call");
                    }
                    boost::optional<std::string> path_string (*context.reduce_argument (arguments->at (0), node));
                    if (!path_string || path_string->empty ()) {
                        throw spec::bad_argument ("string passed to mpaktmp_dir has no value");
                    }
                    
                    boost::filesystem::path path (*path_string);
                    if (!path.is_complete ())
                        throw spec::bad_argument ("path passed to mpaktmp_dir is not absolute: " + path.native_directory_string ());
                    
                    boost::shared_ptr<config_node> config_node (boost::dynamic_pointer_cast<config_node> (node));
                    assert (config_node);
                    
                    config_node->set_mpaktmp_dir (path);
                    
                    return boost::optional<std::string> ();
                }
                
                boost::optional<std::string>
                source_dir (const boost::shared_ptr<spec::node> &node,
                            const boost::shared_ptr<const spec::argument_vector> &arguments,
                            spec::context &context)
                {
                    if (node->get_type () != "mpak:config") {
                        throw spec::bad_command ("source_dir can only be called in config");
                    }
                    if (!arguments ||
                        (arguments->size () != 1)) {
                        throw spec::bad_argument ("incorrect number of arguments passed to source_dir call");
                    }
                    boost::optional<std::string> path_string (*context.reduce_argument (arguments->at (0), node));
                    if (!path_string || path_string->empty ()) {
                        throw spec::bad_argument ("string passed to source_dir has no value");
                    }
                    
                    boost::filesystem::path path (*path_string);
                    if (!path.is_complete ())
                        throw spec::bad_argument ("path passed to source_dir is not absolute");
                    
                    boost::shared_ptr<config_node> config_node (boost::dynamic_pointer_cast<config_node> (node));
                    assert (config_node);
                    
                    config_node->set_source_dir (path);
                    
                    return boost::optional<std::string> ();
                }
            }
        }
        
        namespace command_gens
        {
            boost::shared_ptr<spec::command_info_list>
            mpak_node_command_gen (const boost::shared_ptr<const spec::node> &node)
            {
                boost::shared_ptr<const builtins::mpak_node> mpak_node (boost::dynamic_pointer_cast<const builtins::mpak_node> (node));
                assert (mpak_node);
                
                boost::shared_ptr<spec::command_info_list> command_infos (spec::default_node_command_gen (node));
                
                boost::shared_ptr<spec::argument_vector> description_arguments (new spec::argument_vector);
                description_arguments->push_back (mpak_node->get_description ());
                command_infos->push_back (spec::command_info ("mpak:description", description_arguments));
                
                for (builtins::mpak_node::env_iterator i (mpak_node->begin_env ()); i != mpak_node->end_env (); ++i) {
                    boost::shared_ptr<spec::argument_vector> set_arguments (new spec::argument_vector);
                    set_arguments->push_back (i->first);
                    set_arguments->push_back (i->second);
                    command_infos->push_back (spec::command_info ("mpak:set", set_arguments));
                }
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            category_node_command_gen (const boost::shared_ptr<const spec::node> &node)
            {
                return spec::default_node_command_gen (node);
            }
            
            boost::shared_ptr<spec::command_info_list>
            package_node_command_gen (const boost::shared_ptr<const spec::node> &node)
            {
                return spec::default_node_command_gen (node);
            }
            
            boost::shared_ptr<spec::command_info_list>
            version_node_command_gen (const boost::shared_ptr<const spec::node> &node)
            {
                boost::shared_ptr<const builtins::version_node> version_node (boost::dynamic_pointer_cast<const builtins::version_node> (node));
                assert (version_node);
                
                boost::shared_ptr<spec::command_info_list> command_infos (mpak_node_command_gen (node));
                
                if (version_node->get_slot ()) {
                    boost::shared_ptr<spec::argument_vector> slot_arguments (new spec::argument_vector);
                    slot_arguments->push_back (*version_node->get_slot ());
                    command_infos->push_back (spec::command_info ("mpak:slot", slot_arguments));
                }
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            dependency_node_data_command_gen (const boost::shared_ptr<const spec::node_data> &node_data)
            {
                boost::shared_ptr<const builtins::dependency_node_data> dependency_node_data (boost::dynamic_pointer_cast<const builtins::dependency_node_data> (node_data));
                assert (dependency_node_data);
                
                boost::shared_ptr<spec::command_info_list> command_infos (new spec::command_info_list);
                
                for (builtins::dependency_node_data::all_dependency_iterator i (dependency_node_data->begin_all_dependencies ());
                     i != dependency_node_data->end_all_dependencies (); ++i) {
                    boost::shared_ptr<spec::argument_vector> dependency_arguments (new spec::argument_vector);
                    dependency_arguments->push_back (i->first);
                    dependency_arguments->push_back (i->second.get_string ());
                    command_infos->push_back (spec::command_info ("mpak:dependency", dependency_arguments));
                }
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            installed_node_data_command_gen (const boost::shared_ptr<const spec::node_data> &node_data)
            {
                boost::shared_ptr<const builtins::installed_node_data> installed_node_data (boost::dynamic_pointer_cast<const builtins::installed_node_data> (node_data));
                assert (installed_node_data);
                
                boost::shared_ptr<spec::command_info_list> command_infos (new spec::command_info_list);
                
                for (builtins::installed_node_data::file_stat_iterator i (installed_node_data->begin_file_stats ());
                     i != installed_node_data->end_file_stats (); ++i) {
                    boost::shared_ptr<spec::argument_vector> file_stat_arguments (new spec::argument_vector);
                    
                    // path type uid gid mode
                    file_stat_arguments->push_back (i->first.string ());
                    
                    const util::file_stat &file_stat (i->second);
                    switch (file_stat.type) {
                    case util::file_stat::type_directory:
                        file_stat_arguments->push_back ("directory");
                        break;
                    case util::file_stat::type_chardev:
                        file_stat_arguments->push_back ("chardev");
                        break;
                    case util::file_stat::type_blkdev:
                        file_stat_arguments->push_back ("blkdev");
                        break;
                    case util::file_stat::type_regular:
                        file_stat_arguments->push_back ("regular");
                        break;
                    case util::file_stat::type_fifo:
                        file_stat_arguments->push_back ("fifo");
                        break;
                    case util::file_stat::type_symlink:
                        file_stat_arguments->push_back ("symlink");
                        break;
                    case util::file_stat::type_socket:
                        file_stat_arguments->push_back ("socket");
                        break;
                    default:
                        assert (false);
                    }
                    
                    file_stat_arguments->push_back ((boost::format ("%d") % file_stat.uid).str ());
                    file_stat_arguments->push_back ((boost::format ("%d") % file_stat.gid).str ());
                    file_stat_arguments->push_back ((boost::format ("%04o") % file_stat.mode).str ());
                    
                    switch (file_stat.type) {
                    case util::file_stat::type_chardev: // major minor
                    case util::file_stat::type_blkdev:
                        file_stat_arguments->push_back ((boost::format ("%d") % major (file_stat.devnum)).str ());
                        file_stat_arguments->push_back ((boost::format ("%d") % minor (file_stat.devnum)).str ());
                        break;
                    case util::file_stat::type_regular: // size mtime checksum_algo checksum
                        file_stat_arguments->push_back ((boost::format ("%d") % file_stat.size).str ());
                        file_stat_arguments->push_back ((boost::format ("%d") % file_stat.mtime).str ());
                        file_stat_arguments->push_back (util::checksummer::algorithm_to_string (file_stat.checksum_algo));
                        file_stat_arguments->push_back (util::checksummer::checksum_to_string (file_stat.checksum));
                        break;
                    case util::file_stat::type_symlink: // symlink_target
                        assert (file_stat.symlink_target);
                        file_stat_arguments->push_back (*file_stat.symlink_target);
                        break;
                    case util::file_stat::type_directory:
                    case util::file_stat::type_socket:
                    case util::file_stat::type_fifo:
                        break;
                    }
                    
                    command_infos->push_back (spec::command_info ("mpak:file_stat", file_stat_arguments));
                }
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            timestamp_node_data_command_gen (const boost::shared_ptr<const spec::node_data> &node_data)
            {
                boost::shared_ptr<const builtins::timestamp_node_data> timestamp_node_data (boost::dynamic_pointer_cast<const builtins::timestamp_node_data> (node_data));
                assert (timestamp_node_data);
                
                boost::shared_ptr<spec::command_info_list> command_infos (new spec::command_info_list);
                
                struct timeval tv (timestamp_node_data->get_timestamp ());
                boost::shared_ptr<spec::argument_vector> timestamp_arguments (new spec::argument_vector);
                timestamp_arguments->push_back ((boost::format ("%d") % tv.tv_sec).str ());
                timestamp_arguments->push_back ((boost::format ("%d") % tv.tv_usec).str ());
                command_infos->push_back (spec::command_info ("mpak:timestamp", timestamp_arguments));
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            script_node_data_command_gen (const boost::shared_ptr<const spec::node_data> &node_data)
            {
                boost::shared_ptr<const builtins::script_node_data> script_node_data (boost::dynamic_pointer_cast<const builtins::script_node_data> (node_data));
                assert (script_node_data);
                
                boost::shared_ptr<spec::command_info_list> command_infos (new spec::command_info_list);
                
                for (builtins::script_node_data::script_iterator i (script_node_data->begin_scripts ());
                     i != script_node_data->end_scripts (); ++i) {
                    boost::shared_ptr<spec::argument_vector> script_arguments (new spec::argument_vector);
                    script_arguments->push_back (i->first);
                    script_arguments->push_back (i->second);
                    command_infos->push_back (spec::command_info ("mpak:script", script_arguments));
                }
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            option_node_data_command_gen::
            operator () (const boost::shared_ptr<const spec::node_data> &node_data)
                const
            {
                boost::shared_ptr<const builtins::option_node_data> option_node_data (boost::dynamic_pointer_cast<const builtins::option_node_data> (node_data));
                assert (option_node_data);
                
                boost::shared_ptr<spec::command_info_list> command_infos (new spec::command_info_list);
                
                boost::shared_ptr<spec::argument_vector> flag_arguments (new spec::argument_vector);
                for (builtins::option_node_data::flag_iterator i (option_node_data->begin_flags ());
                     i != option_node_data->end_flags (); ++i) {
                    flag_arguments->push_back (*i);
                }
                command_infos->push_back (spec::command_info (this->command_name_, flag_arguments));
                
                return command_infos;
            }
            
            boost::shared_ptr<spec::command_info_list>
            sources_node_data_command_gen (const boost::shared_ptr<const spec::node_data> &node_data)
            {
                boost::shared_ptr<const builtins::sources_node_data> sources_node_data (boost::dynamic_pointer_cast<const builtins::sources_node_data> (node_data));
                assert (sources_node_data);
                
                boost::shared_ptr<spec::command_info_list> command_infos (new spec::command_info_list);
                
                for (builtins::sources_node_data::sources_iterator i (sources_node_data->begin_sources ());
                     i != sources_node_data->end_sources (); ++i) {
                    boost::shared_ptr<spec::argument_vector> sources_arguments (new spec::argument_vector);
                    sources_arguments->push_back (i->second);
                    sources_arguments->push_back (i->first);
                    command_infos->push_back (spec::command_info ("mpak:source", sources_arguments));
                }
                
                for (builtins::sources_node_data::checksum_iterator i (sources_node_data->begin_checksums ());
                     i != sources_node_data->end_checksums (); ++i) {
                    boost::shared_ptr<spec::argument_vector> checksum_arguments (new spec::argument_vector);
                    checksum_arguments->push_back (i->first);
                    checksum_arguments->push_back (util::checksummer::algorithm_to_string (i->second.first));
                    checksum_arguments->push_back (util::checksummer::checksum_to_string (i->second.second));
                    command_infos->push_back (spec::command_info ("mpak:checksum", checksum_arguments));
                }
                
                return command_infos;
            }
        }
    }
}
