// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/spec/context.cc,v 1.5 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/spec/context.hh>
#include <mpak/spec/pickler.hh>
#include <mpak/spec/command_info.hh>
#include <mpak/spec/node.hh>
#include <mpak/spec/command.hh>
#include <mpak/util/node_path.hh>

#include <boost/shared_ptr.hpp>
#include <boost/optional.hpp>
#include <boost/variant.hpp>

#include <new>
#include <string>
#include <algorithm>

namespace mpak
{
    namespace spec
    {
        context::
        context (const boost::shared_ptr<const pickler> pickler)
            : command_map_ (),
              pickler_ (pickler)
        {
        }
        
        boost::optional<std::string>
        context::
        execute_command (const command_info &info, const boost::shared_ptr<node> &node)
        {
            return this->get_command (info.identifier) (node, info.arguments, *this);
        }
        
        boost::optional<std::string>
        context::
        execute_command_list (const boost::shared_ptr<const command_info_list> &command_infos, boost::shared_ptr<node> node)
        {
            if (!command_infos)
                return boost::optional<std::string> ();
            
            boost::optional<std::string> ret;
            std::for_each (command_infos->begin (), command_infos->end (),
                           phoenix::var(ret) = phoenix::bind (&context::execute_command) (*phoenix::val(this), phoenix::arg1, node));
            return ret;
        }
        
        namespace {
            class reduce_argument_visitor_
                : public boost::static_visitor<boost::optional<std::string> >
            {
            private:
                context * const context_;
                boost::shared_ptr<node> node_;
                
            public:
                reduce_argument_visitor_ (const boost::shared_ptr<node> &node,
                                         context * const context)
                    : context_ (context),
                      node_ (node)
                {
                }
                
                boost::optional<std::string>
                operator () (const std::string &string)
                    const
                {
                    return boost::optional<std::string> (string);
                }
                
                boost::optional<std::string>
                operator () (const boost::shared_ptr<const command_info_list> &command_infos)
                    const
                {
                    return this->context_->execute_command_list (command_infos, node_);
                }
            };
        }
        
        boost::optional<std::string>
        context::
        reduce_argument (const argument_type &argument, const boost::shared_ptr<node> &node)
        {
            return boost::apply_visitor (reduce_argument_visitor_ (node, this), argument);
        }
        
        const command
        context::
        get_command (const std::string &command_name)
            const
        {
            command_map_type_::const_iterator i = this->command_map_.find (command_name);
            if (i != this->command_map_.end ()) {
                return i->second;
            } else {
                throw failure ((boost::format ("could not find command \"%s\"") % command_name).str ());
            }
        }
        
        void
        context::
        add_command (const std::string &command_name, const command &command)
        {
            bool success (this->command_map_.insert (command_map_type_::value_type (command_name, command)).second);
            assert (success);
        }
        
        void
        context::
        add_node_command_gen (const std::string &node_type, const node_command_gen &node_command_gen)
        {
            bool success (this->node_command_gen_map_.insert (node_command_gen_map_type_::value_type (node_type, node_command_gen)).second);
            assert (success);
        }
        
        void
        context::
        add_node_data_command_gen (const std::string &node_data_type, const node_data_command_gen &node_data_command_gen)
        {
            bool success (this->node_data_command_gen_map_.insert (node_data_command_gen_map_type_::value_type (node_data_type,
                                                                                                                node_data_command_gen)).second);
            assert (success);
        }
        
        namespace
        {
            inline std::string
            make_key (const std::string &node_data_type, const std::string &node_type)
            {
                std::string key (node_data_type);
                key.push_back ('\0');
                key.append (node_type);
                return key;
            }
        }
        
        void
        context::
        add_node_data_command_gen (const std::string &node_data_type, const std::string &node_type,
                                   const node_data_command_gen &node_data_command_gen)
        {
            bool success (this->node_data_command_gen_map_.insert (node_data_command_gen_map_type_::value_type (make_key (node_data_type, node_type),
                                                                                                                node_data_command_gen)).second);
            assert (success);
        }
        
        const node_command_gen
        context::
        get_node_command_gen (const std::string &node_type)
            const
        {
            node_command_gen_map_type_::const_iterator i = this->node_command_gen_map_.find (node_type);
            assert (i != this->node_command_gen_map_.end ());
            return i->second;
        }
        
        const node_data_command_gen
        context::
        get_node_data_command_gen (const std::string &node_data_type)
            const
        {
            node_data_command_gen_map_type_::const_iterator i = this->node_data_command_gen_map_.find (node_data_type);
            assert (i != this->node_data_command_gen_map_.end ());
            return i->second;
        }
        
        const node_data_command_gen
        context::
        get_node_data_command_gen (const std::string &node_data_type, const std::string &node_type)
            const
        {
            node_data_command_gen_map_type_::const_iterator i = this->node_data_command_gen_map_.find (make_key (node_data_type, node_type));
            if (i == this->node_data_command_gen_map_.end ())
                return this->get_node_data_command_gen (node_data_type);
            return i->second;
        }
        
        boost::shared_ptr<command_info_list>
        context::
        generate_command_infos (const boost::shared_ptr<const node> &node)
            const
        {
            return this->get_node_command_gen (node->get_type ()) (node);
        }
        
        boost::shared_ptr<command_info_list>
        context::
        generate_command_infos (const std::string &node_data_type,
                                const boost::shared_ptr<const node_data> &node_data)
            const
        {
            return this->get_node_data_command_gen (node_data_type) (node_data);
        }
        
        boost::shared_ptr<command_info_list>
        context::
        generate_command_infos (const std::string &node_data_type,
                                const std::string &node_type,
                                const boost::shared_ptr<const node_data> &node_data)
            const
        {
            return this->get_node_data_command_gen (node_data_type, node_type) (node_data);
        }
        
        boost::shared_ptr<command_info_list>
        context::
        generate_command_infos_with_data (const boost::shared_ptr<const node> &node)
            const
        {
            boost::shared_ptr<command_info_list> command_infos (this->generate_command_infos (node));
            
            for (node::data_const_iterator i (node->begin_data ()); i != node->end_data (); ++i) {
                boost::shared_ptr<command_info_list> data_command_infos (this->generate_command_infos (i->first, node->get_type (), i->second));
                command_infos->splice (command_infos->end (), *data_command_infos, data_command_infos->begin (), data_command_infos->end ());
            }
            
            return command_infos;
        }
        
        namespace
        {
            void
            write_nodes_helper (const boost::shared_ptr<const node> &node,
                                const boost::shared_ptr<const pickler> &pickler,
                                const context &context,
                                const util::node_path &node_path)
            {
                boost::shared_ptr<command_info_list> command_infos (context.generate_command_infos_with_data (node));
                if (node_path.empty ()) {
                    pickler->write (command_infos);
                } else {
                    pickler->write (node->get_type (), node_path, command_infos);
                }
                
                for (node::child_const_iterator i (node->begin_children ()); i != node->end_children (); ++i) {
                    if ((*i)->is_dirty ()) {
                        util::node_path child_node_path (node_path);
                        child_node_path.push_back_element ((*i)->get_name ());
                        write_nodes_helper (*i, pickler, context, child_node_path);
                    }
                }
            }
        }
        
        void
        context::
        write_nodes (const boost::shared_ptr<const node> &node)
            const
        {
            util::node_path node_path;
            write_nodes_helper (node, this->pickler_, *this, node_path);
        }
   }
}
