// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/checksummer.cc,v 1.3 2004/06/19 06:27:58 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/util/checksummer.hh>

#include <boost/filesystem/path.hpp>
#include <boost/format.hpp>

#include <new>
#include <vector>
#include <string>
#include <cassert>
#include <cstdlib>
#include <streambuf>

#include <openssl/evp.h>

namespace mpak
{
    namespace util
    {
        namespace detail
        {
            class checksummer_impl_
            {
            private:
                const EVP_MD *type_;
            public:
                
                checksummer_impl_ (const checksummer::algorithm a);
                
                checksummer::checksum_type
                checksum (std::streambuf *const sb)
                    const;
            };
            
            checksummer_impl_::
            checksummer_impl_ (const checksummer::algorithm a)
            {
                switch (a) {
                case checksummer::algorithm_md2:
                    this->type_ = EVP_md2 ();
                    break;
                case checksummer::algorithm_md5:
                    this->type_ = EVP_md5 ();
                    break;
                case checksummer::algorithm_sha:
                    this->type_ = EVP_sha ();
                    break;
                case checksummer::algorithm_sha1:
                    this->type_ = EVP_sha1 ();
                    break;
                case checksummer::algorithm_mdc2:
                    this->type_ = EVP_mdc2 ();
                    break;
                case checksummer::algorithm_ripemd160:
                    this->type_ = EVP_ripemd160 ();
                    break;
                default:
                    assert (false);
                }
            }
            
            checksummer::checksum_type
            checksummer_impl_::
            checksum (std::streambuf *const sb)
                const
            {
                const std::streamsize buffer_size (EVP_MD_block_size (this->type_));
                const unsigned chksum_size (EVP_MD_size (this->type_));
                
                char buffer[buffer_size];
                unsigned char chksum[chksum_size];
                
                EVP_MD_CTX ctx;
                EVP_DigestInit (&ctx, this->type_);
                
                while (sb->sgetc () != std::char_traits<char>::eof ()) {
                    int n (sb->sgetn (buffer, buffer_size));
                    EVP_DigestUpdate (&ctx, buffer, n);
                }
                
                EVP_DigestFinal (&ctx, chksum, NULL);
                
                return checksummer::checksum_type (chksum, chksum + chksum_size);
            }
            
        }
        
        checksummer::
        checksummer (const checksummer::algorithm a)
            : impl_ (new detail::checksummer_impl_ (a))
        {
        }
        
        checksummer::
        ~checksummer (void)
        {
        }
        
        checksummer::checksum_type
        checksummer::
        checksum (std::streambuf *const sb)
            const
        {
            return this->impl_->checksum (sb);
        }
        
        void
        checksummer::
        set_algorithm (const checksummer::algorithm a)
        {
            this->impl_.reset (new detail::checksummer_impl_ (a));
            this->algorithm_ = a;
        }
        
        const std::string
        checksummer::
        algorithm_to_string (const checksummer::algorithm a)
        {
            switch (a) {
            case algorithm_md2:
                return "md2";
                break;
            case algorithm_md5:
                return "md5";
                break;
            case algorithm_sha:
                return "sha";
                break;
            case algorithm_sha1:
                return "sha1";
                break;
            case algorithm_mdc2:
                return "mdc2";
                break;
            case algorithm_ripemd160:
                return "ripemd160";
                break;
            default:
                throw failure ("unknown checksum algorithm");
            }
        }
        
        const checksummer::algorithm
        checksummer::
        string_to_algorithm (const std::string &string)
        {
            if (string == "default")
                return algorithm_default;
            else if (string == "md2")
                return algorithm_md2;
            else if (string == "md5")
                return algorithm_md5;
            else if (string == "sha")
                return algorithm_sha;
            else if (string == "sha1")
                return algorithm_sha1;
            else if (string == "mdc2")
                return algorithm_mdc2;
            else if (string == "ripemd160")
                return algorithm_ripemd160;
            else
                throw failure ("unknown checksum algorithm: " + string);
        }
        
        const checksummer::checksum_type
        checksummer::
        string_to_checksum (const std::string &string)
        {
            if (string.find_first_not_of ("0123456789ABCDEFabcdef") != std::string::npos)
                throw failure ("invalid character found in checksum string");
            if (string.size () & 1)
                throw failure ("checksum string has odd number of digits");
            checksum_type checksum;
            checksum.reserve (string.size () / 2);
            for (std::string::size_type pos (0); pos < string.size (); pos += 2) {
                checksum.push_back (strtoul (string.substr (pos, 2).c_str (), 0, 16));
            }
            return checksum;
        }
        
        const std::string
        checksummer::
        checksum_to_string (const checksum_type &checksum)
        {
            std::string ret;
            ret.reserve (checksum.size () * 2);
            for (checksum_type::const_iterator i (checksum.begin ()); i != checksum.end (); ++i) {
                ret.append ((boost::format ("%02x") % (unsigned) *i).str ());
            }
            return ret;
        }
    }
}
