// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/dependency.cc,v 1.7 2004/07/07 02:40:42 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/util/dependency.hh>
#include <mpak/util/dependency_grammar.hh>
#include <mpak/util/version_spec.hh>
#include <mpak/util/node_path.hh>
#include <mpak/util/node_path_list.hh>
#include <mpak/spec/node.hh>
#include <mpak/spec/fwd.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/version_node.hh>
#include <mpak/builtins/option_node_data.hh>
#include <mpak/builtins/config_node.hh>

#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/iterator/filter_iterator.hpp>

#include <algorithm>
#include <sstream>

#include <iostream>

namespace mpak
{
    namespace util
    {
        void
        dependency::
        parse_string (const std::string &dependency_string)
        {
            if (dependency_string.empty ()) {
                this->clear ();
                return;
            }
            
            dependency dep;
            detail::dependency_grammar dependency_g;
            
            BOOST_SPIRIT_DEBUG_NODE(dependency_g);
            
            boost::spirit::parse_info<std::string::const_iterator> info =
                boost::spirit::parse (dependency_string.begin (),
                                      dependency_string.end (),
                                      dependency_g[phoenix::var (dep) = phoenix::arg1]);
            
            if (!(info.full && dep.valid ())) {
                throw failure ("invalid dependency string: " + dependency_string);
            }
            
            this->swap (dep);
        }
        
        const std::string
        dependency::
        get_string (void)
            const
        {
            if (!this->valid ())
                throw failure ("invalid dependency");
            
            std::ostringstream oss;
            
            oss << this->path_.get_string ();
            
            if (this->slot_)
                oss << "::" << *this->slot_;
            
            if (!this->options_.empty ()) {
                oss << '(';
                std::vector<std::string>::const_iterator i (this->options_.begin ());
                goto skip_first;
                for (; i != this->options_.end (); ++i) {
                    oss << ' ';
                skip_first:
                    oss << *i;
                }
                oss << ')';
            }
            
            if (this->compare_type_ != compare_none) {
                oss << ' ';
                switch (this->compare_type_) {
                case compare_none:
                    break;
                case compare_less:
                    oss << '<';
                    break;
                case compare_less_equal:
                    oss << "<=";
                    break;
                case compare_equal:
                    oss << '=';
                    break;
                case compare_greater_equal:
                    oss << ">=";
                    break;
                case compare_greater:
                    oss << '>';
                    break;
                case compare_not_equal:
                    oss << "!=";
                    break;
                default:
                    // should never get here
                    assert (false);
                }
                oss << ' ' << this->version_spec_->get_string ();
            }
            
            return oss.str ();
        }
        
        bool
        dependency::
        check_options_ (const boost::shared_ptr<const builtins::config_node> &config_root,
                        const boost::shared_ptr<const builtins::category_node> &root_node,
                        const node_path &node_path,
                        match_options flags)
            const
        {
            boost::shared_ptr<const builtins::option_node_data> with_node_data;
            boost::shared_ptr<const builtins::option_node_data> enable_node_data;
            boost::shared_ptr<const builtins::option_node_data> with_flags_node_data;
            boost::shared_ptr<const builtins::option_node_data> enable_flags_node_data;
            
            boost::shared_ptr<const spec::node> node (node_path.match (root_node, "mpak:version"));
            
            if (flags & match_collect_options) {
                if (!this->options_.empty ()) {
                    with_node_data = builtins::collect_options (config_root, node_path, "mpak:version", "mpak:with", false);
                    enable_node_data = builtins::collect_options (config_root, node_path, "mpak:version", "mpak:enable", false);
                    with_flags_node_data = builtins::collect_options (root_node, node_path, "mpak:version", "mpak:with_flags", true);
                    enable_flags_node_data = builtins::collect_options (root_node, node_path, "mpak:version", "mpak:enable_flags", true);
                }
            } else {
                if (node->has_data ("mpak:with") || node->has_data ("mpak:enable")) {
                    
                    if (node->has_data ("mpak:with")) {
                        with_node_data = boost::dynamic_pointer_cast<const builtins::option_node_data> (node->get_data ("mpak:with"));
                        assert (with_node_data);
                    }
                    if (node->has_data ("mpak:with_flags")) {
                        with_flags_node_data = boost::dynamic_pointer_cast<const builtins::option_node_data> (node->get_data ("mpak:with_flags"));
                        assert (with_flags_node_data);
                    }
                    
                    if (node->has_data ("mpak:enable")) {
                        enable_node_data = boost::dynamic_pointer_cast<const builtins::option_node_data> (node->get_data ("mpak:enable"));
                        assert (enable_node_data);
                    }
                    if (node->has_data ("mpak:enable_flags")) {
                        enable_flags_node_data = boost::dynamic_pointer_cast<const builtins::option_node_data> (node->get_data ("mpak:enable_flags"));
                        assert (enable_flags_node_data);
                    }
                } else {
                    // node wasn't built with any options set
                    return false;
                }
            }
            
            for (std::vector<std::string>::const_iterator i (this->options_.begin ()); i != this->options_.end (); ++i) {
                if (*i->begin () == '@') {
                    // with option
                    const std::string with_flag (i->substr (1));
                    if (with_flags_node_data && with_flags_node_data->has_flag (with_flag)) {
                        if (!with_node_data || !with_node_data->has_flag (with_flag)) {
                            return false;
                        }
                    } else {
                        throw failure ("node " + node->get_name () + " has no with flag " + with_flag);
                    }
                } else {
                    // enable option
                    const std::string enable_flag (*i);
                    if (enable_flags_node_data && enable_flags_node_data->has_flag (enable_flag)) {
                        if (!enable_node_data || !enable_node_data->has_flag (enable_flag)) {
                            return false;
                        }
                    } else {
                        throw failure ("node " + node->get_name () + " has no enable flag " + enable_flag);
                    }
                }
            }
            
            return true;
        }
        
        const node_path_list
        dependency::
        match (const boost::shared_ptr<const builtins::config_node> &config_root,
               const boost::shared_ptr<const builtins::category_node> &root_node,
               match_options flags)
            const
        {
            assert (this->valid ());
            node_path_list matches;
            boost::shared_ptr<const spec::node> node (this->path_.match (root_node, "mpak:package"));
            if (node) {
                for (spec::node::child_const_iterator i (node->begin_children ()); i != node->end_children (); ++i) {
                    if ((*i)->get_type () == "mpak:version") {
                        util::node_path version_node_path (this->path_);
                        version_node_path.push_back_element ((*i)->get_name ());
                        boost::shared_ptr<const builtins::version_node> version_node (boost::dynamic_pointer_cast<const builtins::version_node> (*i));
                        if (this->check_helper_ (config_root, root_node, version_node_path, version_node, flags)) {
                            matches.push_back (version_node_path);
                        }
                    }
                }
            }
            return matches;
        }
        
        bool
        dependency::
        check (const boost::shared_ptr<const builtins::config_node> &config_root,
               const boost::shared_ptr<const builtins::category_node> &root_node,
               const node_path &node_path,
               match_options flags)
            const
        {
            assert (this->valid ());
            
            if (this->path_ != node_path.parent ())
                return false;
            
            boost::shared_ptr<const spec::node> node (node_path.match (root_node, "mpak:version"));
            assert (node);
            boost::shared_ptr<const builtins::version_node> version_node (boost::dynamic_pointer_cast<const builtins::version_node> (node));
            assert (version_node);
            
            return this->check_helper_ (config_root, root_node, node_path, version_node, flags);
        }
        
        bool
        dependency::
        check_helper_ (const boost::shared_ptr<const builtins::config_node> &config_root,
                       const boost::shared_ptr<const builtins::category_node> &root_node,
                       const node_path &node_path,
                       const boost::shared_ptr<const builtins::version_node> &version_node,
                       match_options flags)
            const
        {
            if (this->slot_) {
                if (version_node->get_slot ()) {
                    if (*this->slot_ != *version_node->get_slot ()) {
                        return false;
                    }
                } else {
                    throw failure (node_path.get_string () + " has no slot " + *this->slot_);
                }
            }
            
            if (this->compare_type_ != compare_none) {
                switch (this->compare_type_) {
                case compare_less:
                    if (!(version_node->get_spec () < *this->get_version_spec ()))
                        return false;
                    break;
                case compare_less_equal:
                    if (!(version_node->get_spec () <= *this->get_version_spec ()))
                        return false;
                    break;
                case compare_equal:
                    if (!(version_node->get_spec () == *this->get_version_spec ()))
                        return false;
                    break;
                case compare_greater_equal:
                    if (!(version_node->get_spec () >= *this->get_version_spec ()))
                        return false;
                    break;
                case compare_greater:
                    if (!(version_node->get_spec () > *this->get_version_spec ()))
                        return false;
                    break;
                case compare_not_equal:
                    if (!(version_node->get_spec () != *this->get_version_spec ()))
                        return false;
                    break;
                default:
                    // should never get here
                    assert (false);
                }
            }
            
            if (flags && match_check_options) {
                if (!this->check_options_ (config_root, root_node, node_path, flags))
                    return false;
            }
            
            return true;
        }
    }
}
