// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/file_operations.hh,v 1.5 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__UTIL__FILE_OPERATIONS_HH__
#define __MPAK__UTIL__FILE_OPERATIONS_HH__

#include <boost/filesystem/path.hpp>
#include <boost/function.hpp>

#include <stdexcept>

namespace mpak
{
    namespace util
    {
        class file_operation_error
            : public std::runtime_error
        {
        public:
            inline file_operation_error (const std::string &what)
                : runtime_error (what)
            {
            }
        };
        
        typedef boost::function<void (const boost::filesystem::path &)> traverse_tree_function;
        enum traverse_tree_flags {
            traverse_tree_preorder    = 0x1 << 0,
            traverse_tree_postorder   = 0x2 << 0,
            traverse_tree_type_       = 0x3 << 0,
        };
        
        void traverse_tree (const boost::filesystem::path &path,
                            const traverse_tree_function &function,
                            traverse_tree_flags flags = traverse_tree_preorder);
        const std::string read_symbolic_link (const boost::filesystem::path &path);
        void make_symbolic_link (const std::string &to, const boost::filesystem::path &symlink);
        bool remove (const boost::filesystem::path &path);
        
        // creates an exact copy, including file attributes. this
        // function works even if the file is not a regular file
        // (symlink, device node, etc.) and will creates a new one of
        // those. copies directories recursively, preserving hard
        // links to files within the source tree in the destination
        // tree. source can be any file, directory, special file, etc.
        // dest_dir must be a directory.
        void duplicate (const boost::filesystem::path &source, const boost::filesystem::path &dest_dir);
        // like duplicate, but copies the contents of source_dir to
        // dest_dir, preserving hard links. both arguments must be
        // directories.
        void duplicate_contents (const boost::filesystem::path &source_dir, const boost::filesystem::path &dest_dir);
        
        // creates a new path rel_path from path and base_path such that
        // base_path / rel_path == path.
        // base_path is assumed to be a directory, and both paths should be complete (throws otherwise)
        // if the returned path must not have any '..' components, pass false as the third argument.
        boost::filesystem::path
        make_relative_path (const boost::filesystem::path &path, const boost::filesystem::path &base_path, bool allow_ascending = true);
    }
}

#endif // ifndef __MPAK__UTIL__FILE_OPERATIONS_HH__
