// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/util/version_spec.hh,v 1.2 2004/05/29 15:52:12 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__UTIL__VERSION_SPEC_HH__
#define __MPAK__UTIL__VERSION_SPEC_HH__

#include <mpak/defs.hh>
#include <boost/optional.hpp>

#include <vector>
#include <string>
#include <stdexcept>

namespace mpak
{
    namespace util
    {
        class version_spec
        {
        public:
            class failure
                : public std::runtime_error
            {
            public:
                inline failure (const std::string &what)
                    : runtime_error (what)
                {
                }
            };
            
            typedef std::vector<std::string> version_number_vector;
            
            enum release_types {
                release_pl     =  1,
                release_normal =  0,
                release_rc     = -1,
                release_pre    = -2,
                release_test   = -3,
                release_beta   = -4,
                release_alpha  = -5,
            };
            
        private:
            std::string version_string_;
            version_number_vector version_numbers_;
            char version_letter_;
            boost::optional<release_types> release_type_;
            boost::optional<unsigned> release_number_;
            boost::optional<unsigned> revision_number_;
            
        public:
            version_spec (void)
                : version_numbers_ (),
                  version_letter_ (),
                  release_type_ (),
                  release_number_ (),
                  revision_number_ ()
            {
            }
            
            explicit
            version_spec (const std::string &version_string)
                : version_numbers_ (),
                  version_letter_ (),
                  release_type_ (),
                  release_number_ (),
                  revision_number_ ()
            {
                this->parse_string (version_string);
            }
            
            version_spec (const version_spec &version_spec_)
                : version_numbers_ (version_spec_.version_numbers_),
                  version_letter_ (version_spec_.version_letter_),
                  release_type_ (version_spec_.release_type_),
                  release_number_ (version_spec_.release_number_),
                  revision_number_ (version_spec_.revision_number_)
            {
            }
            
            const version_number_vector &
            get_version_numbers (void)
                const
            {
                return this->version_numbers_;
            }
            
            void
            set_version_numbers (const version_number_vector &version_numbers)
            {
                this->version_numbers_.assign (version_numbers.begin (), version_numbers.end ());
            }
            
            char
            get_version_letter (void)
            {
                return this->version_letter_;
            }
            
            void
            set_version_letter (char version_letter)
            {
                this->version_letter_ = version_letter;
            }
            
            const boost::optional<std::string>
            get_release_type_as_string (void)
                const;
            const boost::optional<release_types> &
            get_release_type_as_int (void)
                const
            {
                return this->release_type_;
            }
            
            void
            set_release_type (const boost::optional<std::string> &release_type);
            void
            set_release_type (const boost::optional<release_types> &release_type);
            
            const boost::optional<unsigned>
            get_release_number (void)
                const
            {
                return this->release_number_;
            }
            
            void
            set_release_number (const unsigned release_number)
            {
                this->release_number_ = release_number;
            }
            
            const boost::optional<unsigned>
            get_revision_number (void)
                const
            {
                return this->revision_number_;
            }
            
            void
            set_revision_number (const unsigned revision_number)
            {
                this->revision_number_ = revision_number;
            }
            
            void parse_string (const std::string &version_string);
            const std::string get_string (void) const;
            const std::string get_string_no_revision (void) const;
            
            bool
            valid (void)
                const;
            
            void swap (version_spec &that)
            {
                this->version_numbers_.swap (that.version_numbers_);
                std::swap (this->version_letter_, that.version_letter_);
                boost::swap (this->release_type_, that.release_type_);
                boost::swap (this->release_number_, that.release_number_);
                boost::swap (this->revision_number_, that.revision_number_);
            }
            
            version_spec &
            operator= (const version_spec &that)
            {
                version_spec new_version_spec (that);
                this->swap (new_version_spec);
                return *this;
            }
            
            void
            clear (void)
            {
                version_spec spec;
                this->swap (spec);
            }
            
            bool
            empty (void)
                const
            {
                assert (this->valid ());
                return (this->version_numbers_.empty () &&
                        !this->version_letter_ &&
                        !this->release_type_ &&
                        !this->release_number_ &&
                        !this->revision_number_);
            }
            
            bool
            operator < (const version_spec &that)
                const;
            bool
            operator == (const version_spec &that)
                const;
            
            bool
            operator != (const version_spec &that)
                const
            {
                return !this->operator == (that);
            }
            
            bool
            operator <= (const version_spec &that)
                const
            {
                return !this->operator> (that);
            }
            
            bool
            operator > (const version_spec &that)
                const
            {
                return that.operator < (*this);
            }
            
            bool
            operator >= (const version_spec &that)
                const
            {
                return !this->operator < (that);
            }
            
        private:
        };
    }
}

namespace std
{
    inline void
    swap (mpak::util::version_spec &version_spec1, mpak::util::version_spec &version_spec2)
    {
        version_spec1.swap (version_spec2);
    }
}

#endif // ifndef __MPAK__UTIL__VERSION_SPEC_HH__
