// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/build/fetcher.cc,v 1.6 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/build/fetcher.hh>
#include <mpak/util/node_path_grammar.hh>

#include <boost/scoped_ptr.hpp>
#include <boost/spirit.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>

#include <curl/curl.h>

#include <new>
#include <string>
#include <fstream>
#include <sys/stat.h>

namespace mpak
{
    namespace build
    {
        namespace detail
        {
            int curl_progress_callback (void *clientp,
                                        double dltotal,
                                        double dlnow,
                                        double ultotal,
                                        double ulnow);
            std::size_t curl_write_callback (char *buffer,
                                             size_t size,
                                             size_t nitems,
                                             void *outstream);
            bool default_fetch_callback (double, double);
            
            class fetcher_impl_
            {
            private:
                CURL *curl_;
                
            public:
                fetcher_impl_ (void)
                    : curl_ (curl_easy_init ())
                {
                }
                
                fetcher_impl_ (const fetcher_impl_ &that)
                    : curl_ (curl_easy_duphandle (that.curl_))
                {
                }
                
                ~fetcher_impl_ (void)
                {
                    curl_easy_cleanup (this->curl_);
                }
                
                void swap (fetcher_impl_ &that)
                {
                    std::swap (this->curl_, that.curl_);
                }
                
                void fetch (const fetcher::fetch_callback &callback,
                            const std::string &url,
                            const boost::filesystem::path &local_file_path);
            };
            
            int curl_progress_callback (void *clientp,
                                        double dltotal,
                                        double dlnow,
                                        double ultotal,
                                        double ulnow)
            {
                fetcher::fetch_callback *callback (static_cast<fetcher::fetch_callback *> (clientp));
                assert (callback);
                return (*callback) (dltotal, dlnow) ? 0 : 1;
            }
            
            std::size_t curl_write_callback (char *buffer,
                                             size_t size,
                                             size_t nitems,
                                             void *outstream)
            {
                std::filebuf *fb (static_cast<std::filebuf *> (outstream));
                return fb->sputn (buffer, size * nitems);
            }
            
            bool default_fetch_callback (double, double)
            {
                return true;
            }
            
            void
            fetcher_impl_::
            fetch (const fetcher::fetch_callback &callback,
                   const std::string &url,
                   const boost::filesystem::path &local_file_path)
            {
                std::filebuf fb;
                
                std::string errstring;
                bool failed (false);
                if (curl_easy_setopt (this->curl_, CURLOPT_WRITEFUNCTION, &curl_write_callback) != CURLE_OK) {
                    failed = true;
                    errstring = "could not set write callback";
                    goto onfail;
                }
                if (curl_easy_setopt (this->curl_, CURLOPT_WRITEDATA, static_cast<void *> (&fb)) != CURLE_OK) {
                    failed = true;
                    errstring = "could not set write stream";
                    goto onfail;
                }
                
                if (curl_easy_setopt (this->curl_, CURLOPT_NOPROGRESS, 0)) {
                    failed = true;
                    errstring = "could not turn on progress callback";
                    goto onfail;
                }
                if (curl_easy_setopt (this->curl_, CURLOPT_PROGRESSFUNCTION, &curl_progress_callback) != CURLE_OK) {
                    failed = true;
                    errstring = "could not set progress callback";
                    goto onfail;
                }
                if (curl_easy_setopt (this->curl_, CURLOPT_PROGRESSDATA, &callback) != CURLE_OK) {
                    failed = true;
                    errstring = "could not set progress callback data";
                    goto onfail;
                }
                
                if (curl_easy_setopt (this->curl_, CURLOPT_URL, url.c_str ()) != CURLE_OK) {
                    failed = true;
                    errstring = "could not set URL";
                    goto onfail;
                }
                
                if (exists (local_file_path)) {
                    struct stat statbuf;
                    if (lstat (local_file_path.native_file_string ().c_str (), &statbuf)) {
                        throw fetcher::failure ("could not open output file: " + local_file_path.native_file_string ());
                    }
                    if (curl_easy_setopt (this->curl_, CURLOPT_RESUME_FROM, (long) statbuf.st_size) != CURLE_OK) {
                        failed = true;
                        errstring = "could not set resume from point";
                        goto onfail;
                    }
                    if (!fb.open (local_file_path.native_file_string ().c_str (), std::ios_base::out | std::ios_base::app)) {
                        throw fetcher::failure ("could not open output file: " + local_file_path.native_file_string ());
                    }
                } else {
                    if (!fb.open (local_file_path.native_file_string ().c_str (), std::ios_base::out)) {
                        throw fetcher::failure ("could not open output file: " + local_file_path.native_file_string ());
                    }
                }
                
                if (curl_easy_perform (this->curl_) != CURLE_OK) {
                    failed = true;
                    errstring = "fetch failed";
                    goto onfail;
                }
                
                // set back defaults
            onfail:
                curl_easy_setopt (this->curl_, CURLOPT_URL, 0);
                
                curl_easy_setopt (this->curl_, CURLOPT_RESUME_FROM, 0);
                
                curl_easy_setopt (this->curl_, CURLOPT_PROGRESSFUNCTION, 0);
                curl_easy_setopt (this->curl_, CURLOPT_PROGRESSDATA, 0);
                
                curl_easy_setopt (this->curl_, CURLOPT_WRITEFUNCTION, 0);
                curl_easy_setopt (this->curl_, CURLOPT_WRITEDATA, 0);
                
                if (failed)
                    throw fetcher::failure (errstring);
            }
        }
        
        fetcher::
        fetcher (void)
            : impl_ (new detail::fetcher_impl_),
              url_ (),
              local_file_path_ ()
        {
        }
        
        fetcher::
        fetcher (const std::string &url, const boost::filesystem::path &local_file_path)
            : impl_ (new detail::fetcher_impl_),
              url_ (url),
              local_file_path_ (local_file_path)
        {
        }
        
        fetcher::
        fetcher (const fetcher &that)
            : impl_ (new detail::fetcher_impl_ (*that.impl_)),
              url_ (that.url_),
              local_file_path_ (that.local_file_path_)
        {
        }
        
        fetcher::
        ~fetcher (void)
        {
        }
        
        void
        fetcher::
        fetch (void)
            const
        {
            this->impl_->fetch (&detail::default_fetch_callback, this->url_, this->local_file_path_);
        }
        
        void
        fetcher::
        fetch (const fetcher::fetch_callback &callback)
            const
        {
            this->impl_->fetch (callback, this->url_, this->local_file_path_);
        }
    }
}
