// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/builtins/init.cc,v 1.9 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/builtins/init.hh>
#include <mpak/spec/context.hh>
#include <mpak/spec/command.hh>
#include <mpak/builtins/commands.hh>
#include <mpak/build/manager.hh>
#include <mpak/builtins/stages.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/package_node.hh>
#include <mpak/builtins/version_node.hh>

#include <boost/shared_ptr.hpp>

#include <new>

namespace mpak
{
    namespace builtins
    {
        void
        init_package_tree_context (spec::context &context, const boost::shared_ptr<const builtins::config_node> &config_root)
        {
            context.add_command ("mpak:get", commands::get (config_root));
            context.add_command ("mpak:set", &commands::set);
            context.add_command ("mpak:if_else", &commands::if_else);
            context.add_command ("mpak:return", &commands::return_);
            context.add_command ("mpak:concat", &commands::concat);
            
            context.add_command ("mpak:description", &commands::tree::description);
            
            spec::create_node_command<builtins::category_node> category ("mpak:category");
            spec::create_node_command<builtins::package_node> package ("mpak:package");
            spec::create_node_command<builtins::version_node> version ("mpak:version");
            category.add_parent_type ("mpak:category");
            package.add_parent_type ("mpak:category");
            version.add_parent_type ("mpak:package");
            
            context.add_command ("mpak:category", category);
            context.add_command ("mpak:package", package);
            context.add_command ("mpak:version", version);
            
            context.add_command ("mpak:slots", &commands::tree::slots);
            context.add_command ("mpak:slot", &commands::tree::slot);
            
            context.add_command ("mpak:dependency", &commands::tree::dependency);
            context.add_command ("mpak:source", &commands::tree::source);
            context.add_command ("mpak:checksum", &commands::tree::checksum);
            
            context.add_command ("mpak:script", &commands::tree::script);
            context.add_command ("mpak:interpreter", &commands::tree::interpreter);
            
            context.add_command ("mpak:with_flags", commands::tree::option ("mpak:with_flags"));
            context.add_command ("mpak:enable_flags", commands::tree::option ("mpak:enable_flags"));
            
            context.add_command ("mpak:if_with", commands::tree::if_option ("mpak:with", "mpak:with_flags", config_root));
            context.add_command ("mpak:if_enable", commands::tree::if_option ("mpak:enable", "mpak:enable_flags", config_root));
            
            context.add_node_command_gen ("mpak:category", &command_gens::category_node_command_gen);
            context.add_node_command_gen ("mpak:package", &command_gens::package_node_command_gen);
            context.add_node_command_gen ("mpak:version", &command_gens::version_node_command_gen);
            
            context.add_node_data_command_gen ("mpak:dependency", &command_gens::dependency_node_data_command_gen);
            context.add_node_data_command_gen ("mpak:script", &command_gens::script_node_data_command_gen);
            context.add_node_data_command_gen ("mpak:source", &command_gens::sources_node_data_command_gen);
        }
        
        void
        init_installed_tree_context (spec::context &context)
        {
            context.add_command ("mpak:set", &commands::set);
            context.add_command ("mpak:if_else", &commands::if_else);
            context.add_command ("mpak:return", &commands::return_);
            context.add_command ("mpak:concat", &commands::concat);
            
            context.add_command ("mpak:description", &commands::tree::description);
            
            spec::create_node_command<builtins::category_node> category ("mpak:category");
            spec::create_node_command<builtins::package_node> package ("mpak:package");
            spec::create_node_command<builtins::version_node> version ("mpak:version");
            category.add_parent_type ("mpak:category");
            package.add_parent_type ("mpak:category");
            version.add_parent_type ("mpak:package");
            
            context.add_command ("mpak:category", category);
            context.add_command ("mpak:package", package);
            context.add_command ("mpak:version", version);
            
            context.add_command ("mpak:slots", &commands::tree::slots);
            context.add_command ("mpak:slot", &commands::tree::slot);
            
            context.add_command ("mpak:dependency", &commands::tree::dependency);
            context.add_command ("mpak:source", &commands::tree::source);
            context.add_command ("mpak:checksum", &commands::tree::checksum);
            
            context.add_command ("mpak:script", &commands::tree::script);
            context.add_command ("mpak:interpreter", &commands::tree::interpreter);
            
            context.add_command ("mpak:file_stat", &commands::tree::file_stat);
            
            context.add_command ("mpak:timestamp", &commands::tree::timestamp);
            
            context.add_command ("mpak:with", commands::tree::option ("mpak:with"));
            context.add_command ("mpak:enable", commands::tree::option ("mpak:enable"));
            context.add_command ("mpak:with_flags", commands::tree::option ("mpak:with_flags"));
            context.add_command ("mpak:enable_flags", commands::tree::option ("mpak:enable_flags"));
            
            context.add_node_command_gen ("mpak:category", &command_gens::category_node_command_gen);
            context.add_node_command_gen ("mpak:package", &command_gens::package_node_command_gen);
            context.add_node_command_gen ("mpak:version", &command_gens::version_node_command_gen);
            
            context.add_node_data_command_gen ("mpak:dependency", &command_gens::dependency_node_data_command_gen);
            context.add_node_data_command_gen ("mpak:timestamp", &command_gens::timestamp_node_data_command_gen);
            context.add_node_data_command_gen ("mpak:installed", &command_gens::installed_node_data_command_gen);
            context.add_node_data_command_gen ("mpak:with",
                                               command_gens::option_node_data_command_gen ("mpak:with"));
            context.add_node_data_command_gen ("mpak:with_flags",
                                               command_gens::option_node_data_command_gen ("mpak:with_flags"));
            context.add_node_data_command_gen ("mpak:enable",
                                               command_gens::option_node_data_command_gen ("mpak:enable"));
            context.add_node_data_command_gen ("mpak:enable_flags",
                                               command_gens::option_node_data_command_gen ("mpak:enable_flags"));
            context.add_node_data_command_gen ("mpak:script", &command_gens::script_node_data_command_gen);
            context.add_node_data_command_gen ("mpak:source", &command_gens::sources_node_data_command_gen);
        }
        
        void
        init_config_context (spec::context &context)
        {
            context.add_command ("mpak:get", commands::get ());
            context.add_command ("mpak:set", &commands::set);
            context.add_command ("mpak:if_else", &commands::if_else);
            context.add_command ("mpak:return", &commands::return_);
            context.add_command ("mpak:concat", &commands::concat);
            
            context.add_command ("mpak:package_tree_dir", &commands::config::package_tree_dir);
            context.add_command ("mpak:statedb_dir", &commands::config::statedb_dir);
            context.add_command ("mpak:mpaktmp_dir", &commands::config::mpaktmp_dir);
            context.add_command ("mpak:source_dir", &commands::config::source_dir);
            
            spec::create_node_command<builtins::category_node> category ("mpak:category", false);
            spec::create_node_command<builtins::package_node> package ("mpak:package", false);
            spec::create_node_command<builtins::version_node> version ("mpak:version", false);
            category.add_parent_type ("mpak:config");
            category.add_parent_type ("mpak:category");
            package.add_parent_type ("mpak:category");
            version.add_parent_type ("mpak:package");
            
            context.add_command ("mpak:category", category);
            context.add_command ("mpak:package", package);
            context.add_command ("mpak:version", version);
            
            context.add_command ("mpak:with", commands::tree::option ("mpak:with"));
            context.add_command ("mpak:enable", commands::tree::option ("mpak:enable"));
        }
        
        void
        init_build_manager (build::manager &manager)
        {
            const boost::shared_ptr<const build::stage> clean (new stages::clean);
            const boost::shared_ptr<const build::stage> check (new stages::check);
            const boost::shared_ptr<const build::stage> setup (new stages::setup);
            const boost::shared_ptr<const build::stage> dependencies (new stages::dependencies);
            const boost::shared_ptr<const build::stage> fetch (new stages::fetch);
            const boost::shared_ptr<const build::stage> unpack (new stages::script ("unpack"));
            const boost::shared_ptr<const build::stage> configure (new stages::script ("configure"));
            const boost::shared_ptr<const build::stage> build (new stages::script ("build"));
            const boost::shared_ptr<const build::stage> install (new stages::script ("install", &stages::install_script_pre_execute));
            const boost::shared_ptr<const build::stage> merge (new stages::merge);
            const boost::shared_ptr<const build::stage> unmerge (new stages::unmerge);
            
            manager.add_stage ("clean", clean);
            manager.add_stage ("check", check);
            manager.add_stage ("setup", setup);
            manager.add_stage ("fetch", fetch);
            manager.add_stage ("dependencies", dependencies);
            manager.add_stage ("unpack", unpack);
            manager.add_stage ("configure", configure);
            manager.add_stage ("build", build);
            manager.add_stage ("install", install);
            manager.add_stage ("merge", merge);
            manager.add_stage ("unmerge", unmerge, build::manager::stage_use_installed_tree);
            
            manager.add_dependency ("setup", "check");
            manager.add_dependency ("fetch", "setup");
            manager.add_dependency ("dependencies", "fetch");
            manager.add_dependency ("unpack", "dependencies");
            manager.add_dependency ("configure", "unpack");
            manager.add_dependency ("build", "configure");
            manager.add_dependency ("install", "build");
            manager.add_dependency ("merge", "install");
        }
    }
}
