// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/spec/database_pickler.cc,v 1.5 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/spec/database_pickler.hh>
#include <mpak/spec/node.hh>
#include <mpak/spec/command_info.hh>
#include <mpak/util/node_path.hh>

#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/scoped_array.hpp>
#include <boost/variant.hpp>
#include <boost/format.hpp>

#include <new>
#include <sstream>
#include <iomanip>
#include <cstdlib>
#include <cctype>

#include <gdbm.h>

#include <valgrind/valgrind.h>

namespace mpak
{
    namespace spec
    {
        namespace {
            std::string
            make_key (const std::string &node_type, const util::node_path &node_path)
            {
                std::string key ("node");
                key.push_back ('\0');
                key.append (node_type);
                key.push_back ('\0');
                key.append (node_path.get_string ());
                return key;
            }
            
            std::string
            make_key (void)
            {
                std::string key ("node");
                key.push_back ('\0');
                key.append ("root");
                return key;
            }
            
            std::string
            escape_string (const std::string &string)
            {
                std::string ret;
                for (std::string::const_iterator i (string.begin ()); i != string.end (); ++i) {
                    if (!std::isprint (*i)) {
                        ret.append ((boost::format ("\\%02x") % (int) *i).str ());
                    } else {
                        ret.push_back (*i);
                    }
                }
                return ret;
            }
        }
        
        namespace detail
        {
            class database_pickler_impl_
            {
            private:
                GDBM_FILE db_;
                
            public:
                database_pickler_impl_ (const boost::filesystem::path &database_path);
                ~database_pickler_impl_ (void);
                std::string read (const std::string &key);
                void write (const std::string &key, const std::string &value);
                void remove (const std::string &key);
            };
            
            void gdbm_fatal (void)
            {
                throw pickler::failure ("fatal gdbm error");
            }
            
            database_pickler_impl_::
            database_pickler_impl_ (const boost::filesystem::path &database_path)
            {
                bool create (false);
                if (!exists (database_path)) {
                    create = true;
                }
                std::string path_string (database_path.native_file_string ());
                
                boost::scoped_array<char> path_buffer (new char[path_string.size () + 1]);
                std::copy (path_string.begin (), path_string.end (), path_buffer.get ());
                path_buffer[path_string.size ()] = '\0';
                
                if (create) {
                    this->db_ = gdbm_open (path_buffer.get (), 0, GDBM_WRCREAT, 0644, &gdbm_fatal);
                    if (!this->db_) {
                        throw pickler::failure ("could not open database: " + path_string);
                    }
                    
                    const std::string key (make_key ());
                    
                    boost::scoped_array<char> key_buffer (new char[key.size ()]);
                    std::copy (key.begin (), key.end (), key_buffer.get ());
                    datum key_datum = { key_buffer.get (), key.size () };
                    
                    boost::scoped_array<char> value_buffer (new char[0]);
                    datum value_datum  = { value_buffer.get (), 0 };
                    
                    if (gdbm_store (this->db_, key_datum, value_datum, GDBM_REPLACE)) {
                        throw pickler::failure ("could not store root entry");
                    }
                } else {
                    this->db_ = gdbm_open (path_buffer.get (), 0, GDBM_WRITER, 0644, &gdbm_fatal);
                    if (!this->db_) {
                        throw pickler::failure ("could not open database: " + path_string);
                    }
                }
            }
            
            database_pickler_impl_::
            ~database_pickler_impl_ (void)
            {
                gdbm_close (this->db_);
            }
            
            std::string
            database_pickler_impl_::
            read (const std::string &key)
            {
                boost::scoped_array<char> key_buffer (new char[key.size ()]);
                std::copy (key.begin (), key.end (), key_buffer.get ());
                datum key_datum = { key_buffer.get (), key.size () };
                datum value_datum = gdbm_fetch (this->db_, key_datum);
                
                if (!value_datum.dptr) {
                    VALGRIND_PRINTF_BACKTRACE (("unable to retrieve record" + escape_string (key)).c_str ());
                    assert (false);
                    throw pickler::failure ("unable to retrieve record " + escape_string (key));
                }
                
                std::string ret (value_datum.dptr, value_datum.dsize);
                std::free (value_datum.dptr);
                
                return ret;
            }
            
            void
            database_pickler_impl_::
            write (const std::string &key, const std::string &value)
            {
                boost::scoped_array<char> key_buffer (new char[key.size ()]);
                std::copy (key.begin (), key.end (), key_buffer.get ());
                datum key_datum = { key_buffer.get (), key.size () };
                
                boost::scoped_array<char> value_buffer (new char[value.size ()]);
                std::copy (value.begin (), value.end (), value_buffer.get ());
                datum value_datum = { value_buffer.get (), value.size () };
                
                if (gdbm_store (this->db_, key_datum, value_datum, GDBM_REPLACE)) {
                    throw pickler::failure ("could not store key " + key);
                }
            }
            
            void
            database_pickler_impl_::
            remove (const std::string &key)
            {
                boost::scoped_array<char> key_buffer (new char[key.size ()]);
                std::copy (key.begin (), key.end (), key_buffer.get ());
                datum key_datum = { key_buffer.get (), key.size () };
                
                gdbm_delete (this->db_, key_datum);
            }
        }
        
        database_pickler::
        database_pickler (const boost::filesystem::path &database_path)
            : impl_ (new detail::database_pickler_impl_ (database_path)),
              database_path_ (database_path)
        {
        }
        
        database_pickler::
        ~database_pickler (void)
        {
        }
        
        namespace {
            struct database_string_grammar;
            struct database_argument_list_grammar;
            struct database_command_grammar;
            struct database_command_list_grammar;
            struct database_grammar;
            
            struct database_string_closure
                : public boost::spirit::closure<database_string_closure, std::string>
            {
                member1 string_;
            };
            
            struct database_string_grammar
                : public boost::spirit::grammar<database_string_grammar, database_string_closure::context_t>
            {
                template<class scanner_type_>
                struct definition
                {
                    typedef database_string_closure::context_t context_type;
                    typedef boost::spirit::rule<scanner_type_, context_type> rule_type;
                    
                    rule_type start_;
                    
                    definition (const database_string_grammar &self)
                    {
                        using boost::spirit::anychar_p;
                        using boost::spirit::ch_p;
                        using phoenix::bind;
                        using phoenix::arg1;
                        using phoenix::arg2;
                        using phoenix::construct_;
                        
                        typedef typename scanner_type_::iterator_t iterator_type;
                        std::string &(std::string::*string_assign) (iterator_type, iterator_type) = &std::string::assign;
                        start_ = (*(anychar_p - ch_p ('\e')))[bind (string_assign) (self.string_, arg1, arg2)];
                        
                        BOOST_SPIRIT_DEBUG_NODE (start_);
                    }
                    
                    const boost::spirit::rule<scanner_type_, context_type> &
                    start (void) const
                    {
                        return this->start_;
                    }
                };
            };
            
            struct database_argument_list_closure
                : public boost::spirit::closure<database_argument_list_closure, boost::shared_ptr<argument_vector> >
            {
                member1 arguments;
            };
            
            struct database_argument_list_grammar
                : public boost::spirit::grammar<database_argument_list_grammar, database_argument_list_closure::context_t>
            {
                template<class scanner_type_>
                struct definition
                {
                    typedef database_argument_list_closure::context_t context_type;
                    typedef boost::spirit::rule<scanner_type_, context_type> rule_type;
                    
                    rule_type start_;
                    database_command_list_grammar command_list;
                    database_string_grammar string;
                    
                    definition (const database_argument_list_grammar &self)
                    {
                        using boost::spirit::anychar_p;
                        using boost::spirit::ch_p;
                        using boost::spirit::epsilon_p;
                        using phoenix::arg1;
                        using phoenix::arg2;
                        using phoenix::bind;
                        using phoenix::construct_;
                        using phoenix::new_;
                        
                        start_ = *(ch_p ('\e') >> ch_p ('A') >> ( (epsilon_p (ch_p ('\e')) >>
                                                                   command_list(construct_<boost::shared_ptr<command_info_list> >
                                                                                (new_<command_info_list> ()))
                                                                   [bind (&argument_vector::push_back) (*self.arguments, arg1)])
                                                                | string[bind (&argument_vector::push_back) (*self.arguments, arg1)])
                                                                );
                        
                        BOOST_SPIRIT_DEBUG_NODE (start_);
                        BOOST_SPIRIT_DEBUG_NODE (command_list);
                        BOOST_SPIRIT_DEBUG_NODE (string);
                    }
                    
                    const boost::spirit::rule<scanner_type_, context_type> &
                    start (void) const
                    {
                        return this->start_;
                    }
                };
            };
            
            struct database_command_closure
                : public boost::spirit::closure<database_command_closure, command_info>
            {
                member1 command_info_;
            };
            
            struct database_command_grammar
                : public boost::spirit::grammar<database_command_grammar, database_command_closure::context_t>
            {
                template<class scanner_type_>
                struct definition
                {
                    typedef database_command_closure::context_t context_type;
                    typedef boost::spirit::rule<scanner_type_, context_type> rule_type;
                    
                    rule_type start_;
                    database_string_grammar string;
                    database_argument_list_grammar argument_list;
                    
                    definition (const database_command_grammar &self)
                    {
                        using boost::spirit::anychar_p;
                        using boost::spirit::ch_p;
                        
                        using phoenix::arg1;
                        using phoenix::arg2;
                        using phoenix::bind;
                        using phoenix::construct_;
                        using phoenix::new_;
                        
                        start_ =
                            ch_p ('\e') >> ch_p ('C') >>
                            string[bind (&command_info::identifier) (self.command_info_) = arg1] >>
                            argument_list(construct_<boost::shared_ptr<argument_vector> > (new_<argument_vector> ()))
                                [bind (&command_info::arguments) (self.command_info_) = arg1] >>
                            ch_p ('\e') >> ch_p ('E');
                        
                        BOOST_SPIRIT_DEBUG_NODE (start_);
                        BOOST_SPIRIT_DEBUG_NODE (string);
                        BOOST_SPIRIT_DEBUG_NODE (argument_list);
                    }
                    
                    const boost::spirit::rule<scanner_type_, context_type> &
                    start (void) const
                    {
                        return this->start_;
                    }
                };
            };
            
            struct database_command_list_closure
                : public boost::spirit::closure<database_command_list_closure, boost::shared_ptr<command_info_list> >
            {
                member1 command_infos;
            };
            
            struct database_command_list_grammar
                : public boost::spirit::grammar<database_command_list_grammar, database_command_list_closure::context_t>
            {
                template<class scanner_type_>
                struct definition
                {
                    typedef database_command_list_closure::context_t context_type;
                    typedef boost::spirit::rule<scanner_type_, context_type> rule_type;
                    
                    rule_type start_;
                    database_command_grammar command;
                    
                    definition (const database_command_list_grammar &self)
                    {
                        using boost::spirit::anychar_p;
                        using boost::spirit::ch_p;
                        using phoenix::arg1;
                        using phoenix::arg2;
                        using phoenix::construct_;
                        using phoenix::bind;
                        
                        start_ = *(command[bind (&command_info_list::push_back) (*self.command_infos, arg1)]);
                        
                        BOOST_SPIRIT_DEBUG_NODE (start_);
                        BOOST_SPIRIT_DEBUG_NODE (command);
                    }
                    
                    const boost::spirit::rule<scanner_type_, context_type> &
                    start (void) const
                    {
                        return this->start_;
                    }
                };
            };
            
            struct database_closure
                : public boost::spirit::closure<database_closure, boost::shared_ptr<command_info_list> >
            {
                member1 command_infos;
            };
            
            struct database_grammar
                : public boost::spirit::grammar<database_grammar, database_closure::context_t>
            {
                template<class scanner_type_>
                struct definition
                {
                    typedef database_closure::context_t context_type;
                    typedef boost::spirit::rule<scanner_type_, context_type> rule_type;
                    
                    rule_type start_;
                    database_command_list_grammar command_list;
                    
                    definition (const database_grammar &self)
                    {
                        using phoenix::arg1;
                        using phoenix::construct_;
                        using phoenix::new_;
                        
                        start_ = command_list(construct_<boost::shared_ptr<command_info_list> > (new_<command_info_list> ()))[self.command_infos = arg1];
                        
                        BOOST_SPIRIT_DEBUG_NODE (start_);
                        BOOST_SPIRIT_DEBUG_NODE (command_list);
                    }
                    
                    const boost::spirit::rule<scanner_type_, context_type> &
                    start (void) const
                    {
                        return this->start_;
                    }
                };
            };
            
        }
        
        const boost::shared_ptr<command_info_list>
        database_pickler::
        read_helper_ (const std::string &key)
            const
        {
            std::string value (this->impl_->read (key));
            
            boost::shared_ptr<mpak::spec::command_info_list> command_infos;
            typedef boost::spirit::position_iterator<std::string::const_iterator> iterator_type;
            iterator_type begin (value.begin (), value.end (), this->database_path_.native_file_string () + ':' + key);
            iterator_type end;
            
            database_grammar database_g;
            
            BOOST_SPIRIT_DEBUG_NODE(database_g);
            
            boost::spirit::parse_info<iterator_type>
                info (boost::spirit::parse (begin, end,
                                            database_g[phoenix::var (command_infos) = phoenix::arg1]));
            
            if (!info.full) {
                std::ostringstream oss;
                for (iterator_type i (info.stop); i != end; ++i) {
                    if (std::isprint (*i))
                        oss << *i;
                    else
                        oss << "\\" << std::setw (3) << std::setfill ('0') << int (*i);
                }
                throw failure ("parse failed, remaining input: " + oss.str ());
            }
            
            return command_infos;
        }
        
        const boost::shared_ptr<command_info_list>
        database_pickler::
        read (const std::string &node_type,
              const util::node_path &node_path)
            const
        {
            const std::string key (make_key (node_type, node_path));
            return this->read_helper_ (key);
        }
        
        const boost::shared_ptr<command_info_list>
        database_pickler::
        read (void)
            const
        {
            const std::string key (make_key ());
            return this->read_helper_ (key);
        }
        
        namespace {
            const std::string
            stringify_command_info_list_ (const command_info_list &command_infos);
            
            const std::string
            stringify_command_info_ (const command_info &command_info);
            
            const std::string
            stringify_argument_vector_ (const boost::shared_ptr<argument_vector> &arguments);
            
            const std::string
            stringify_argument_ (const argument_type &argument);
            
            class stringify_argument_visitor_
                : public boost::static_visitor<std::string>
            {
            public:
                inline const std::string
                operator () (const std::string &string)
                    const
                {
                    std::string ret (string);
                    return ret;
                }
                
                inline const std::string
                operator () (const boost::shared_ptr<const command_info_list> &command_infos)
                    const
                {
                    if (command_infos) {
                        std::string ret (stringify_command_info_list_ (*command_infos));
                        return ret;
                    } else {
                        return std::string ();
                    }
                }
            };
            
            const std::string
            stringify_command_info_list_ (const command_info_list &command_infos)
            {
                std::string ret;
                for (command_info_list::const_iterator ci (command_infos.begin ()); ci != command_infos.end (); ++ci) {
                    ret.append (stringify_command_info_ (*ci)); 
                }
                return ret;
            }
            
            const std::string
            stringify_command_info_ (const command_info &command_info)
            {
                std::string ret;
                ret.push_back ('\e');
                ret.push_back ('C');                          // begin command
                ret.append (command_info.identifier);
                if (command_info.arguments)
                    ret.append (stringify_argument_vector_ (command_info.arguments));
                ret.push_back ('\e');
                ret.push_back ('E');                          // end command
                return ret;
            }
            
            const std::string
            stringify_argument_vector_ (const boost::shared_ptr<argument_vector> &arguments)
            {
                std::string ret;
                if (!arguments)
                    return ret;
                for (argument_vector::const_iterator ai (arguments->begin ()); ai != arguments->end (); ++ai) {
                    ret.push_back ('\e');
                    ret.push_back ('A');                          // arguments
                    ret.append (stringify_argument_ (*ai));
                }
                return ret;
            }
            
            inline const std::string
            stringify_argument_ (const argument_type &argument)
            {
                return boost::apply_visitor (stringify_argument_visitor_ (), argument);
            }
        }
        
        void
        database_pickler::
        write_helper_ (const std::string &key,
                       const boost::shared_ptr<const command_info_list> &command_infos)
            const
        {
            std::string value (stringify_command_info_list_ (*command_infos));
            this->impl_->write (key, value);
        }
        
        void
        database_pickler::
        write (const boost::shared_ptr<const command_info_list> &command_infos)
            const
        {
            const std::string key (make_key ());
            this->write_helper_ (key, command_infos);
        }
        
        void
        database_pickler::
        write (const std::string &node_type,
               const util::node_path &node_path,
               const boost::shared_ptr<const command_info_list> &command_infos)
            const
        {
            const std::string key (make_key (node_type, node_path));
            this->write_helper_ (key, command_infos);
        }
        
        void
        database_pickler::
        remove (const std::string &node_type,
               const util::node_path &node_path)
            const
        {
            const std::string key (make_key (node_type, node_path));
            this->impl_->remove (key);
        }
        
    }
}
