// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/util/file_stat.cc,v 1.2 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <mpak/util/file_stat.hh>
#include <mpak/util/checksummer.hh>
#include <mpak/util/file_operations.hh>

#include <boost/filesystem/path.hpp>

#include <string>

#include <sys/stat.h>
#include <time.h>
#include <sys/time.h>
#include <sys/sysmacros.h>

#include <iostream>
#include <boost/format.hpp>

namespace mpak
{
    namespace util
    {
        void
        file_stat::
        get_stats_from_path (const boost::filesystem::path &path, checksummer::algorithm checksum_algo)
        {
            struct stat stat;
            if (lstat (path.native_file_string ().c_str (), &stat)) {
                throw failure ("could not get stats for file " + path.native_file_string ());
            }
            
            file_stat new_file_stat;
            
            new_file_stat.mode = stat.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID | S_ISVTX);
            
            new_file_stat.uid = stat.st_uid;
            new_file_stat.gid = stat.st_gid;
            
            new_file_stat.size = 0;
            
            new_file_stat.devnum = 0;
            
            new_file_stat.mtime = stat.st_mtim.tv_sec;
            new_file_stat.symlink_target = boost::optional<std::string> ();
            
            if (S_ISDIR (stat.st_mode)) {
                new_file_stat.type = type_directory;
            } else if (S_ISCHR (stat.st_mode)) {
                new_file_stat.type = type_chardev;
                new_file_stat.devnum = stat.st_rdev;
            } else if (S_ISBLK (stat.st_mode)) {
                new_file_stat.type = type_blkdev;
                new_file_stat.devnum = stat.st_rdev;
            } else if (S_ISREG (stat.st_mode)) {
                new_file_stat.type = type_regular;
                new_file_stat.size = stat.st_size;
                new_file_stat.checksum_algo = checksum_algo;
                checksummer cs (checksum_algo);
                new_file_stat.checksum = cs.checksum (path);
            } else if (S_ISFIFO (stat.st_mode)) {
                new_file_stat.type = type_fifo;
            } else if (S_ISLNK (stat.st_mode)) {
                new_file_stat.type = type_symlink;
                new_file_stat.symlink_target = read_symbolic_link (path);
            } else if (S_ISSOCK (stat.st_mode)) {
                new_file_stat.type = type_socket;
            } else {
                throw failure ("file is of unknown type: " + path.native_file_string ());
            }
            
            this->swap (new_file_stat);
        }
    }
}
