/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * Coordinator Node
 *
 * TODO: a fixed timeout isn't a
 *       good thing, you have to consider the rtt from the requester node to
 *       the coordinator node. Any better idea?
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    // Used to calculate the range of authoritative sources for a record
    // associated to a given NIP
    public const int COORD_DUPLICATION = 10;

    public class RmtCoordPeer : RmtPeer, ICoord
    {
        public RmtCoordPeer(PeerToPeer peer_to_peer_service, Object? key=null, NIP? hIP=null, AggregatedNeighbour? aggregated_neighbour=null)
        {
            base(peer_to_peer_service, key, hIP, aggregated_neighbour);
        }

        public void duplicate_all_knowledge(PartialNIP gnode, CoordinatorKnowledge coordinator_knowledge) throws HookingError, RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "duplicate_all_knowledge";
            rc.add_parameter(gnode);
            rc.add_parameter(coordinator_knowledge);
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                rmt(rc);
            }
            catch (RPCError e) {throw e;}
            catch (HookingError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public HookReservation? reserve(PartialNIP gnode) throws HookingError, RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "reserve";
            rc.add_parameter(gnode);
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                ISerializable ret = rmt(rc);
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
                else return (HookReservation)ret;
            }
            catch (RPCError e) {throw e;}
            catch (HookingError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public void free(PartialNIP gnode, int pos) throws HookingError, RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "free";
            rc.add_parameter(gnode);
            rc.add_parameter(new SerializableInt(pos));
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                rmt(rc);
            }
            catch (RPCError e) {throw e;}
            catch (HookingError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public void duplicate_free(PartialNIP gnode, int pos) throws HookingError, RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "duplicate_free";
            rc.add_parameter(gnode);
            rc.add_parameter(new SerializableInt(pos));
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                rmt(rc);
            }
            catch (RPCError e) {throw e;}
            catch (HookingError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public bool register_bnode(PartialNIP gnode, NIP nip, bool is_border, bool has_tunnel, bool is_willing) throws HookingError, RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "register_bnode";
            rc.add_parameter(gnode);
            rc.add_parameter(nip);
            rc.add_parameter(new SerializableBool(is_border));
            rc.add_parameter(new SerializableBool(has_tunnel));
            rc.add_parameter(new SerializableBool(is_willing));
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                SerializableBool ret = (SerializableBool)rmt(rc);
                return ret.b;
            }
            catch (RPCError e) {throw e;}
            catch (HookingError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }

        public Gee.List<InfoCoord> report_status() throws RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "report_status";
            // NOTE: RmtPeer.rmt can throw any Error. It does already filters RemotableException classes.
            try {
                ListISerializable ret = (ListISerializable)rmt(rc);
                return (Gee.List<InfoCoord>)ret.backed;
            }
            catch (RPCError e) {throw e;}
            catch (Error e) {throw new RPCError.GENERIC(@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");}
        }
    }

    public class Coord_hkey : Object
    {
        public int level_of_gnode;
        public NIP nip;
    }

    struct struct_helper_Coord_communicate_to_new_coordinator
    {
        public Coord self;
        public int level_of_gnode;
    }

    public class Coord : PeerToPeer, ICoord
    {
        public static const int mypid = 1;
        private ArrayList<Coordinator> _coordinators;

        public Coord(AggregatedNeighbourManager aggregated_neighbour_manager,
                     MapRoute maproute, PeerToPeerAll peer_to_peer_all,
                     CoordinatorKnowledgeSet coordinator_knowledge_set)
        {
            base(aggregated_neighbour_manager, maproute, mypid);

            // let's register ourself in peer_to_peer_all
            try
            {
                peer_to_peer_all.peer_to_peer_register(this);
            }
            catch (PeerToPeerError e)
            {
                error(@"Coord service: registering failed: $(e.message)");
            }

            // An instance of Coordinator for each level
            initialize_coordinators();
            // Memorize received coordinator_knowledge_set.
            for (int level_of_gnode = 1; level_of_gnode < maproute.levels+1; level_of_gnode++)
            {
                Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
                if (coordinator_knowledge_set.dict.has_key(coordinator.gnode))
                {
                    CoordinatorKnowledge coordinator_knowledge = coordinator_knowledge_set.dict[coordinator.gnode];
                    log_debug(@"Coord: memorizing knowledge records for gnode $(coordinator.gnode)");
                    coordinator.add_bookings(coordinator_knowledge.bookings);
                    coordinator.last_assigned_elderliness = coordinator_knowledge.last_assigned_elderliness;
                    coordinator.add_bnode_list(coordinator_knowledge.bnode_list);
                }
            }
            log_debug("Coord: finished memorizing coordinator_knowledge_set.");

            maproute.node_new.connect(node_joins_maproute);
            maproute.node_deleted.connect(node_exits_maproute);
            this.map_peer_to_peer_validated.connect(coord_has_valid_map);
        }

        public RmtCoordPeer peer(NIP? hIP=null, Object? key=null, AggregatedNeighbour? aggregated_neighbour=null)
        {
            assert(hIP != null || key != null);
            return new RmtCoordPeer(this, key, hIP, aggregated_neighbour);
        }

        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            log_debug(@"$(this.get_type().name()): dispatching $data");
            string[] pieces = data.method_name.split(".");
            if (pieces[0] == "duplicate_all_knowledge")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("duplicate_all_knowledge is a function.");
                if (data.parameters.size != 2) throw new RPCError.MALFORMED_PACKET("duplicate_all_knowledge wants 2 parameters.");
                ISerializable iser0 = data.parameters.get(0);
                ISerializable iser1 = data.parameters.get(1);
                if (! iser0.get_type().is_a(typeof(PartialNIP)))
                    throw new RPCError.MALFORMED_PACKET("duplicate_all_knowledge parameter 1 is not a PartialNIP.");
                if (! iser1.get_type().is_a(typeof(CoordinatorKnowledge)))
                    throw new RPCError.MALFORMED_PACKET("duplicate_all_knowledge parameter 2 is not a CoordinatorKnowledge.");
                PartialNIP gnode = (PartialNIP)iser0;
                CoordinatorKnowledge coordinator_knowledge = (CoordinatorKnowledge)iser1;
                duplicate_all_knowledge(gnode, coordinator_knowledge);
                return new SerializableNone();
            }
            if (pieces[0] == "reserve")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("reserve is a function.");
                if (data.parameters.size != 1) throw new RPCError.MALFORMED_PACKET("reserve wants 1 parameters.");
                ISerializable iser0 = data.parameters.get(0);
                if (! iser0.get_type().is_a(typeof(PartialNIP)))
                    throw new RPCError.MALFORMED_PACKET("reserve parameter 1 is not a PartialNIP.");
                PartialNIP gnode = (PartialNIP)iser0;
                HookReservation? ret = reserve(gnode);
                if (ret == null) return new SerializableNone();
                else return (HookReservation)ret;
            }
            if (pieces[0] == "free")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("free is a function.");
                if (data.parameters.size != 2) throw new RPCError.MALFORMED_PACKET("free wants 2 parameters.");
                ISerializable iser0 = data.parameters.get(0);
                ISerializable iser1 = data.parameters.get(1);
                if (! iser0.get_type().is_a(typeof(PartialNIP)))
                    throw new RPCError.MALFORMED_PACKET("free parameter 1 is not a PartialNIP.");
                if (! iser1.get_type().is_a(typeof(SerializableInt)))
                    throw new RPCError.MALFORMED_PACKET("free parameter 2 is not a int.");
                PartialNIP gnode = (PartialNIP)iser0;
                SerializableInt pos = (SerializableInt)iser1;
                free(gnode, pos.i);
                return new SerializableNone();
            }
            if (pieces[0] == "duplicate_free")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("duplicate_free is a function.");
                if (data.parameters.size != 2) throw new RPCError.MALFORMED_PACKET("duplicate_free wants 2 parameters.");
                ISerializable iser0 = data.parameters.get(0);
                ISerializable iser1 = data.parameters.get(1);
                if (! iser0.get_type().is_a(typeof(PartialNIP)))
                    throw new RPCError.MALFORMED_PACKET("duplicate_free parameter 1 is not a PartialNIP.");
                if (! iser1.get_type().is_a(typeof(SerializableInt)))
                    throw new RPCError.MALFORMED_PACKET("duplicate_free parameter 2 is not a int.");
                PartialNIP gnode = (PartialNIP)iser0;
                SerializableInt pos = (SerializableInt)iser1;
                duplicate_free(gnode, pos.i);
                return new SerializableNone();
            }
            if (pieces[0] == "register_bnode")
            {
                if (pieces.length != 1) throw new RPCError.MALFORMED_PACKET("register_bnode is a function.");
                if (data.parameters.size != 5) throw new RPCError.MALFORMED_PACKET("register_bnode wants 5 parameters.");
                ISerializable iser0 = data.parameters.get(0);
                ISerializable iser1 = data.parameters.get(1);
                ISerializable iser2 = data.parameters.get(2);
                ISerializable iser3 = data.parameters.get(3);
                ISerializable iser4 = data.parameters.get(4);
                if (! iser0.get_type().is_a(typeof(PartialNIP)))
                    throw new RPCError.MALFORMED_PACKET("register_bnode parameter 1 is not a PartialNIP.");
                if (! iser1.get_type().is_a(typeof(NIP)))
                    throw new RPCError.MALFORMED_PACKET("register_bnode parameter 2 is not a NIP.");
                if (! iser2.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET("register_bnode parameter 3 is not a SerializableBool.");
                if (! iser3.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET("register_bnode parameter 4 is not a SerializableBool.");
                if (! iser4.get_type().is_a(typeof(SerializableBool)))
                    throw new RPCError.MALFORMED_PACKET("register_bnode parameter 5 is not a SerializableBool.");
                PartialNIP gnode = (PartialNIP)iser0;
                NIP nip = (NIP)iser1;
                SerializableBool is_border = (SerializableBool)iser2;
                SerializableBool has_tunnel = (SerializableBool)iser3;
                SerializableBool is_willing = (SerializableBool)iser4;
                bool ret = register_bnode(gnode, nip, is_border.b, has_tunnel.b, is_willing.b);
                return new SerializableBool(ret);
            }
            if (pieces[0] == "report_status")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "report_status is a function.");
                if (data.parameters.size != 0)
                    throw new RPCError.MALFORMED_PACKET(
                        "report_status wants no parameters.");
                return new ListISerializable.with_backer(report_status());
            }
            return base._dispatch(caller, data);
        }

        public Gee.List<InfoCoord> report_status()
        {
            log_debug("Coord: report_status started");
            var ret = new ArrayList<InfoCoord>();
            for (int _level_of_nodes = maproute.levels-1; _level_of_nodes >= 0; _level_of_nodes--)
            {
                int _level_of_gnode = _level_of_nodes + 1;
                log_debug(@"Coord: report_status level $(_level_of_gnode)");
                PartialNIP actual_coord = new PartialNIP({});

                try
                {
                    Coord_hkey key = new Coord_hkey();
                    key.level_of_gnode = _level_of_gnode;
                    key.nip = maproute.me;
                    actual_coord = search_participant_as_nip(h(key));
                }
                catch (PeerToPeerError e)
                {
                    log_error("Coord.report_status: no participants in Coord service.");
                }

                var coordinator = get_coordinator_for_gnode_at_level(_level_of_gnode);
                var knowledge = coordinator.report_status();
                var info_coord = new InfoCoord();
                info_coord.level = _level_of_gnode;
                info_coord.knowledge = knowledge;
                info_coord.is_main = coordinator.is_main;
                info_coord.actual_coord = actual_coord;
                ret.add(info_coord);
            }
            return ret;
        }

        public override NIP h(Object k)
        {
            Coord_hkey key = (Coord_hkey)k;
            int[] positions = key.nip.get_positions();
            for (int l = 0; l < key.level_of_gnode; l++)
                positions[l] = 0;
            return new NIP(positions);
        }

        public Coordinator get_coordinator_for_gnode_at_level(int l)
        {
            return _coordinators.get(l-1);
        }

        private void set_coordinator_for_gnode_at_level(int l)
        {
            assert(_coordinators.size == l-1);
            _coordinators.add(new Coordinator(maproute.me.get_gnode_at_level(l), this));
        }

        public void initialize_coordinators()
        {
            _coordinators = new ArrayList<Coordinator>();
            for (int l = 1; l <= maproute.levels; l++)
            {
                set_coordinator_for_gnode_at_level(l);
            }
        }

        public override void stop_operations()
        {
            string ipstr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            log_debug(@"Coord: stopping operations for $(ipstr)");
            log_debug(@"Coord: calling base");
            base.stop_operations();
            log_debug(@"Coord: base done");
            for (int l = 1; l < maproute.levels+1; l++)
            {
                log_debug(@"Coord: calling coordinator for level $(l) for $(ipstr)");
                Coordinator coordinator = get_coordinator_for_gnode_at_level(l);
                coordinator.stop_operations();
            }
        }

        public PartialNIP get_my_gnode_at_level(int l)
        {
            return maproute.me.get_gnode_at_level(l);
        }

        public int get_level_of_gnode(PartialNIP gnode)
        {
            return gnode.level_of_gnode();
        }

        public Coordinator get_coordinator_for_gnode(PartialNIP gnode) throws HookingError
        {
            int level_of_gnode = get_level_of_gnode(gnode);
            // check that this is right gnode
            if (! get_my_gnode_at_level(level_of_gnode).is_equal(gnode))
                throw new HookingError.INEXISTENT_GNODE(@"$gnode");
            return get_coordinator_for_gnode_at_level(level_of_gnode);
        }

        /** Client-side helpers **/

        public HookReservation? enter_into(int level_of_gnode, NIP nip, AggregatedNeighbour? neighbour_to_contact)
        {
            /** I have a neighbour whose nip is 'nip' (could be
              *  of a different network with different levels
              *  and gsize for example) and it said that its gnode of
              *  level 'level_of_gnode' has some free space.
              * I try to get to the coordinator passing through
              *  neighbour_to_contact (could be another neighbour)
              */
            Coord_hkey key = new Coord_hkey();
            key.level_of_gnode = level_of_gnode;
            key.nip = nip;
            RmtCoordPeer _peer = peer(null, key, neighbour_to_contact);
            HookReservation? ret = null;
            try
            {
                ret = _peer.reserve(nip.get_gnode_at_level(level_of_gnode));
            }
            catch (RPCError e)
            {
                log_info(@"Coord: got RPCError code $(e.code): $(e.message)");
            }
            catch (HookingError e)
            {
                log_info(@"Coord: got HookingError code $(e.code): $(e.message)");
            }
            return ret;
        }

        public Bookings get_bookings(int level_of_gnode)
        {
            Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
            return coordinator.get_bookings();
        }

        public int get_last_assigned_elderliness(int level_of_gnode)
        {
            Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
            return coordinator.last_assigned_elderliness;
        }

        public BnodeList get_bnode_list(int level_of_gnode)
        {
            Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
            return coordinator.get_bnode_list();
        }

        /** Remotables **/

        public HookReservation? reserve(PartialNIP gnode) throws HookingError
        {
            Coordinator coordinator = get_coordinator_for_gnode(gnode);
            return coordinator.reserve();
        }

        public void free(PartialNIP gnode, int pos) throws HookingError
        {
            Coordinator coordinator = get_coordinator_for_gnode(gnode);
            coordinator.free(pos);
        }

        public void duplicate_all_knowledge(PartialNIP gnode, CoordinatorKnowledge coordinator_knowledge) throws HookingError
        {
            Coordinator coordinator = get_coordinator_for_gnode(gnode);
            coordinator.add_bookings(coordinator_knowledge.bookings);
            if (coordinator.last_assigned_elderliness < coordinator_knowledge.last_assigned_elderliness)
                coordinator.last_assigned_elderliness = coordinator_knowledge.last_assigned_elderliness;
            coordinator.add_bnode_list(coordinator_knowledge.bnode_list);
        }

        public void duplicate_free(PartialNIP gnode, int pos) throws HookingError
        {
            Coordinator coordinator = get_coordinator_for_gnode(gnode);
            coordinator.force_remove(pos);
        }

        public bool register_bnode(PartialNIP gnode, NIP nip, bool is_border, bool has_tunnel, bool is_willing) throws HookingError
        {
            Coordinator coordinator = get_coordinator_for_gnode(gnode);
            return coordinator.register_bnode(nip, is_border, has_tunnel, is_willing);
        }

        /** Handling events **/

        /** A new node in the service
          */
        public void node_joins_maproute(int level_of_nodes, int pos)
        {
            int level_of_gnode = level_of_nodes + 1;
            Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
            coordinator.map_has_changed();

            // There is no more need for a booking, we have the node in the maproute.
            coordinator.force_remove(pos);
            // If I was Coordinator and now no more, then the new Coordinator has to be informed.
            if (coordinator.was_main && ! coordinator.is_main)
                communicate_to_new_coordinator(level_of_gnode);
        }

        private void impl_communicate_to_new_coordinator(int level_of_gnode) throws Error
        {
            Tasklet.declare_self("Coord.communicate_to_new_coordinator");
            Coord_hkey key = new Coord_hkey();
            key.level_of_gnode = level_of_gnode;
            key.nip = maproute.me;
            RmtCoordPeer _peer = peer(null, key);
            log_debug(@"Coord: communicating my previous records to new Coord at level $level_of_gnode");
            CoordinatorKnowledge coordinator_knowledge = new CoordinatorKnowledge(get_my_gnode_at_level(level_of_gnode),
                            get_bookings(level_of_gnode),
                            get_last_assigned_elderliness(level_of_gnode),
                            get_bnode_list(level_of_gnode));
            _peer.duplicate_all_knowledge(get_my_gnode_at_level(level_of_gnode), coordinator_knowledge);
        }

        private static void * helper_communicate_to_new_coordinator(void *v) throws Error
        {
            struct_helper_Coord_communicate_to_new_coordinator *tuple_p = (struct_helper_Coord_communicate_to_new_coordinator *)v;
            // The caller function has to add a reference to the ref-counted instances
            Coord self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            int level_of_gnode_save = tuple_p->level_of_gnode;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_communicate_to_new_coordinator(level_of_gnode_save);
            // void method, return null
            return null;
        }

        public void communicate_to_new_coordinator(int level_of_gnode)
        {
            struct_helper_Coord_communicate_to_new_coordinator arg = struct_helper_Coord_communicate_to_new_coordinator();
            arg.self = this;
            arg.level_of_gnode = level_of_gnode;
            Tasklet.spawn((FunctionDelegate)helper_communicate_to_new_coordinator, &arg);
        }

        /** A deleted node in the service
          */
        public void node_exits_maproute(int level_of_nodes, int pos)
        {
            int level_of_gnode = level_of_nodes + 1;
            Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
            coordinator.map_has_changed();
        }

        /** We now have a valid map, I could have become a coordinator
          */
        public void coord_has_valid_map()
        {
            for (int level_of_gnode = 1; level_of_gnode < maproute.levels+1; level_of_gnode++)
            {
                Coordinator coordinator = get_coordinator_for_gnode_at_level(level_of_gnode);
                coordinator.map_has_changed();
            }
        }
    }

    struct struct_helper_Coordinator_manage_bnode_tunnels
    {
        public Coordinator self;
    }

    struct struct_helper_Coordinator_forward_knowledge
    {
        public Coordinator self;
    }

    struct struct_helper_Coordinator_forward_knowledge_to_nip
    {
        public Coordinator self;
        public NIP nip;
    }

    struct struct_helper_Coordinator_forward_free
    {
        public Coordinator self;
        public int pos;
    }

    struct struct_helper_Coordinator_forward_free_to_nip
    {
        public Coordinator self;
        public NIP to_nip;
        public int pos;
    }

    public class Coordinator : Object
    {
        public int last_assigned_elderliness;
        private Bookings bookings;
        private BnodeList bnode_list;
        public PartialNIP gnode {get; private set;}
        private weak Coord parent;
        private weak MapRoute maproute;
        private int level_of_gnode;
        private int level_of_nodes;
        private bool _is_main_previous_state;
        private bool _is_main_latest_evaluation;
        private Tasklet? manage_bnode_tunnels_handle;

        public Coordinator(PartialNIP gnode, Coord parent)
        {
            this.gnode = gnode;
            this.parent = parent;
            maproute = parent.maproute;
            level_of_gnode = gnode.level_of_gnode();
            level_of_nodes = level_of_gnode - 1;
            bookings = new Bookings(gnode);
            bnode_list = new BnodeList(gnode);
            last_assigned_elderliness = 0;
            _is_main_previous_state = false;
            _is_main_latest_evaluation = false;
            manage_bnode_tunnels_handle = null;
            // Border nodes of the entire network aren't managed.
            if (level_of_gnode != maproute.levels)
                manage_bnode_tunnels();
        }

        public void stop_operations()
        {
            string ipstr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            log_debug(@"Coordinator: stopping operations for $(ipstr)");
            if (manage_bnode_tunnels_handle != null)
            {
                log_debug(@"Coordinator: aborting manage_bnode_tunnels for $(ipstr)");
                manage_bnode_tunnels_handle.abort();
                manage_bnode_tunnels_handle = null;
            }
        }

        public CoordinatorKnowledge report_status()
        {
            return new CoordinatorKnowledge(gnode, 
                                            bookings,
                                            last_assigned_elderliness,
                                            bnode_list);
        }

        public void map_has_changed()
        {
            try
            {
                _is_main_previous_state = _is_main_latest_evaluation;
                Coord_hkey key = new Coord_hkey();
                key.level_of_gnode = level_of_gnode;
                key.nip = maproute.me;
                _is_main_latest_evaluation = parent.search_participant(parent.h(key)) == null && parent.has_valid_map;
            }
            catch (PeerToPeerError e)
            {
                log_error("Coordinator.map_has_changed: no participants in Coord service.");
            }
        }

        public bool is_main {
            get {
                return _is_main_latest_evaluation;
            }
        }

        public bool was_main {
            get {
                return _is_main_previous_state;
            }
        }

        /** A node wants to become a member of our gnode.
          * If possible, we add the reservation and return the HookReservation,
          * else, we return None.
          */
        public HookReservation? reserve()
        {
            log_debug(@"Coord.reserve: gnode = $gnode");

            int[] _fnl = maproute.free_nodes_list(level_of_nodes);
            ArrayList<int> fnl = new ArrayList<int>();
            foreach (int i in _fnl)
                if (! bookings.is_booked(i))
                    fnl.add(i);
            if (fnl.is_empty) return null;

            int pos = fnl.get(Random.int_range(0, fnl.size));
            bookings.new_book(pos);

            log_debug(@"Coord.reserve: returns pos $pos");
            // new_elderliness = max(ID(G) for G in H) | H is my gnode of level lvl+1
            ArrayList<int> pool = new ArrayList<int>();
            for (int pos2 = 0; pos2 < maproute.gsize; pos2++)
            {
                RouteNode node = maproute.node_get(level_of_nodes, pos2);
                if (! node.is_empty()) pool.add(node.get_eldest_gid().elderliness);
            }
            pool.add(maproute.get_gid_list()[level_of_nodes].elderliness);
            pool.add(last_assigned_elderliness);

            int? max_pool = null;
            foreach (int maybe_max in pool)
            {
                if (max_pool == null) max_pool = maybe_max;
                else if (max_pool < maybe_max) max_pool = maybe_max;
            }
            int new_elderliness = max_pool + 1;
            log_debug(@"Coord.reserve: returns new_elderliness $new_elderliness");
            last_assigned_elderliness = new_elderliness;

            GNodeID[] gids = maproute.get_gid_uppermost_list(level_of_gnode);
            string output_gids = "";
            foreach (GNodeID gid in gids) output_gids += @"$gid, ";
            log_debug(@"Coord.reserve: returns gids [$output_gids]");

            forward_knowledge();
            CoordinatorKnowledgeSet coordinator_knowledge_set = new CoordinatorKnowledgeSet();
            for (int _level_of_nodes = level_of_nodes; _level_of_nodes < maproute.levels; _level_of_nodes++)
            {
                int _level_of_gnode = _level_of_nodes + 1;
                PartialNIP _gnode = parent.get_my_gnode_at_level(_level_of_gnode);
                // bookings...
                Bookings _bookings = new Bookings(_gnode);
                for (int pos2 = 0; pos2 < maproute.gsize; pos2++)
                {
                    if ((! maproute.node_get(_level_of_nodes, pos2).is_free()) ||
                                 parent.get_coordinator_for_gnode_at_level(_level_of_gnode)
                                 .bookings.is_booked(pos2))
                    {
                        _bookings.new_book(pos2);
                    }
                }
                // elderliness...
                int _last_assigned_elderliness = parent.get_coordinator_for_gnode_at_level(_level_of_gnode)
                                                .last_assigned_elderliness;
                // bnode_list...
                BnodeList _bnode_list = parent.get_coordinator_for_gnode_at_level(_level_of_gnode)
                                                .get_bnode_list();
                coordinator_knowledge_set.dict_set(_gnode, new CoordinatorKnowledge(_gnode, 
                                    _bookings, _last_assigned_elderliness, _bnode_list));
            }
            log_debug("Coord.reserve: returns a new coordinator_knowledge_set");

            HookReservation ret = new HookReservation();
            ret.gnode = gnode;
            ret.pos = pos;
            ret.elderliness = new_elderliness;
            ret.gids = gids;
            ret.coordinator_knowledge_set = coordinator_knowledge_set;
            return ret;
        }

        public void free(int pos)
        {
            log_debug(@"Coord.free: gnode,pos = ($gnode, $pos)");
            force_remove(pos);
            forward_free(pos);
        }

        private void impl_forward_free(int pos) throws Error
        {
            Tasklet.declare_self("Coordinator.forward_free");
            Coord_hkey key = new Coord_hkey();
            key.level_of_gnode = level_of_gnode;
            key.nip = maproute.me;
            NIP hash_nip = parent.h(key);
            log_debug(@"Coord: starting find_nearest COORD_DUPLICATION to gnode $gnode");
            Gee.List<NIP> bunch = parent.find_nearest_to_register(hash_nip, COORD_DUPLICATION, level_of_gnode);
            NIP[] bunch_not_me = {};
            foreach (NIP n in bunch)
                if (! n.is_equal(maproute.me))
                    bunch_not_me += n;
            foreach (NIP to_nip in bunch_not_me)
                forward_free_to_nip(to_nip, pos);
        }

        private static void * helper_forward_free(void *v) throws Error
        {
            struct_helper_Coordinator_forward_free *tuple_p = (struct_helper_Coordinator_forward_free *)v;
            // The caller function has to add a reference to the ref-counted instances
            Coordinator self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            int pos_save = tuple_p->pos;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_forward_free(pos_save);
            // void method, return null
            return null;
        }

        /** Communicate to secondary coordinators to free this 'pos'.
          */
        public void forward_free(int pos)
        {
            struct_helper_Coordinator_forward_free arg = struct_helper_Coordinator_forward_free();
            arg.self = this;
            arg.pos = pos;
            Tasklet.spawn((FunctionDelegate)helper_forward_free, &arg);
        }

        private void impl_forward_free_to_nip(NIP to_nip, int pos) throws Error
        {
            Tasklet.declare_self("Coordinator.forward_free_to_nip");
            string dest_addr = nip_to_str(maproute.levels, maproute.gsize, to_nip);
            string my_addr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            log_debug(@"Coord: forwarding free $pos to $dest_addr");
            AddressManagerTCPClient remote = new AddressManagerTCPClient(dest_addr, null, my_addr, false);
            try
            {
                remote.coordnode.duplicate_free(gnode, pos);
                log_debug(@"Coord: forwarded free $pos to $dest_addr");
            }
            catch (Error e)
            {
                log_warn(@"'Coord: forwarded free $pos to $dest_addr got exception $(e.message)");
            }
        }

        private static void * helper_forward_free_to_nip(void *v) throws Error
        {
            struct_helper_Coordinator_forward_free_to_nip *tuple_p = (struct_helper_Coordinator_forward_free_to_nip *)v;
            // The caller function has to add a reference to the ref-counted instances
            Coordinator self_save = tuple_p->self;
            NIP to_nip_save = tuple_p->to_nip;
            // The caller function has to copy the value of byvalue parameters
            int pos_save = tuple_p->pos;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_forward_free_to_nip(to_nip_save, pos_save);
            // void method, return null
            return null;
        }

        /** Forwards a freed position to another hash node in the bunch.
          */
        public void forward_free_to_nip(NIP to_nip, int pos)
        {
            struct_helper_Coordinator_forward_free_to_nip arg = struct_helper_Coordinator_forward_free_to_nip();
            arg.self = this;
            arg.pos = pos;
            arg.to_nip = to_nip;
            Tasklet.spawn((FunctionDelegate)helper_forward_free_to_nip, &arg);
        }

        public Bookings get_bookings()
        {
            return bookings.clone();
        }

        public void add_bookings(Bookings bookings)
        {
            foreach (BookingRecord booking_record in bookings)
                add_record(booking_record);
        }

        /** Add a record to my bookings, do not forward.
          */
        public void add_record(BookingRecord booking_record)
        {
            RouteNode node = maproute.node_get(level_of_nodes, booking_record.pos);
            if (! node.is_free())
                return; // There is no need for a booking when we have the node in the maproute.
            bookings.force_add(booking_record);
        }

        /** Remove a record from my bookings, do not forward.
          */
        public void force_remove(int pos)
        {
            bookings.force_remove(pos);
        }

        public BnodeList get_bnode_list()
        {
            return bnode_list.clone();
        }

        public bool register_bnode(NIP nip, bool is_border, bool has_tunnel, bool is_willing)
        {
            if (is_border)
            {
                // Add a record (or refresh) that nip is a border node
                bool ret = bnode_list.update_bnode(nip, has_tunnel, is_willing);
                forward_knowledge();
                return ret;
            }
            else
            {
                // Delete a record (if there is) that nip is a border node
                bnode_list.delete_bnode(nip);
                forward_knowledge(); // TODO should also forward deletion. not a must.
                // Always interested in removing
                return true;
            }
        }

        public void add_bnode_list(BnodeList bnode_list)
        {
            foreach (BnodeRecord bnode_record in bnode_list)
                bnode_list.copy_bnode(bnode_record);
        }

        private void impl_manage_bnode_tunnels() throws Error
        {
            Tasklet.declare_self("Coordinator.manage_bnode_tunnels");
            manage_bnode_tunnels_handle = Tasklet.self();
            log_debug("Coord: manage_bnode_tunnels: started");
            while (true)
            {
                try
                {
                    ms_wait(10000);
                    // If I am the actual Coordinator
                    log_debug(@"Coord: manage_bnode_tunnels: is_main = $is_main");
                    if (is_main)
                    {
                        // See how many nodes approximatively exist in the self.gnode
                        // TODO
                        int number_of_nodes = 50;
                        // See how many border nodes with a tunnel
                        int number_of_tunnels = 0;
                        foreach (BnodeRecord bn in bnode_list)
                            if (bn.has_tunnel)
                                number_of_tunnels++;
                        log_debug(@"Coord: manage_bnode_tunnels: number_of_tunnels = $number_of_tunnels");
                        // See how many border nodes without a tunnel
                        ArrayList<BnodeRecord> candidates = new ArrayList<BnodeRecord>(BnodeRecord.equal_func);
                        foreach (BnodeRecord bn in bnode_list)
                            if (bn.is_willing)
                                candidates.add(bn);
                        string str_candidates = "[";
                        foreach (BnodeRecord record in candidates) str_candidates += @"$record,";
                        str_candidates += "]";
                        log_debug(@"Coord: manage_bnode_tunnels: candidates = $str_candidates");
                        // Decision is based on the ratio below. Better suggestions?
                        if (candidates.size >= 2)
                        {
                            if (number_of_tunnels == 0 || number_of_nodes/number_of_tunnels > 10)
                            {
                                // A new tunnel is wanted
                                log_debug("Coord: manage_bnode_tunnels: a new tunnel is wanted");
                                arrange_tunnel(candidates);
                            }
                        }
                    }
                }
                catch (Error e)
                {
                    // some trouble, but we must carry on to manage border nodes.
                    log_error(@"Uncaught exception $(e.message) while manage_bnode_tunnels of $gnode");
                }
            }
        }

        private static void * helper_manage_bnode_tunnels(void *v) throws Error
        {
            struct_helper_Coordinator_manage_bnode_tunnels *tuple_p = (struct_helper_Coordinator_manage_bnode_tunnels *)v;
            // The caller function has to add a reference to the ref-counted instances
            Coordinator self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_manage_bnode_tunnels();
            // void method, return null
            return null;
        }

        public void manage_bnode_tunnels()
        {
            struct_helper_Coordinator_manage_bnode_tunnels arg = struct_helper_Coordinator_manage_bnode_tunnels();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_manage_bnode_tunnels, &arg);
        }

        class Pair<A, B> : Object
        {
            public A a {get; private set;}
            public B b {get; private set;}
            public Pair(A a, B b)
            {
                this.a = a;
                this.b = b;
            }
        }
        public void arrange_tunnel(ArrayList<BnodeRecord> candidates)
        {
            // A new tunnel is wanted
            string my_addr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            AddressManagerTCPClient remote_x, remote_y;
            // Searching for a pair of bordernodes with max distance
            ArrayList<Pair<NIP, PairNipDistance>> pool = new ArrayList<Pair<NIP, PairNipDistance>>();
            for (int i = 0; i < candidates.size; i++)
            {
                NIP x = candidates[i].nip;
                ArrayList<NIP> list_of_y = new ArrayList<NIP>(PartialNIP.equal_func);
                for (int j = i+1; j < candidates.size; j++)
                {
                    list_of_y.add(candidates[j].nip);
                }
                remote_x = new AddressManagerTCPClient(nip_to_str(maproute.levels, maproute.gsize, x), null, my_addr);
                Gee.List<PairNipDistance> dists = remote_x.border_nodes_manager.get_distances(gnode, list_of_y);
                string str_dists = "{";
                foreach (PairNipDistance _dist in dists) str_dists += @"$(_dist), ";
                str_dists = str_dists.substring(0, str_dists.length-2) + "}";
                log_debug(@"Coord: manage_bnode_tunnels: $(x) dists = $(str_dists)");
                foreach (PairNipDistance y_dist in dists)
                    if (y_dist.distance > 1)
                        pool.add(new Pair<NIP, PairNipDistance>(x, y_dist));
            }
            if (pool.is_empty)
                return;
            // reverse sort distance
            CompareDataFunc<Pair<NIP, PairNipDistance>> cmp = (a,b) => {return a.b.distance - b.b.distance;};
            pool.sort(cmp);
            NIP nip_x = pool[0].a;
            NIP nip_y = pool[0].b.nip;
            log_debug(@"Coord: arrange_tunnel: between x $(nip_x) and y $(nip_y)");
            // Let x and y have a secondary address. Let x' and y' make a tunnel.
            remote_x = new AddressManagerTCPClient(nip_to_str(maproute.levels, maproute.gsize, nip_x), null, my_addr);
            NIP nip_x_secondary = remote_x.border_nodes_manager.get_new_address(gnode);
            log_debug(@"Coord: arrange_tunnel: x has secondary $(nip_x_secondary)");
            remote_y = new AddressManagerTCPClient(nip_to_str(maproute.levels, maproute.gsize, nip_y), null, my_addr);
            NIP nip_y_secondary = remote_y.border_nodes_manager.get_new_address(gnode, nip_x_secondary);
            log_debug(@"Coord: arrange_tunnel: y has secondary $(nip_y_secondary)");
            log_debug(@"Coord: arrange_tunnel: passed $(nip_x_secondary) to $(nip_y_secondary)");
            remote_x.border_nodes_manager.assign_peer_nip(nip_x_secondary, nip_y_secondary);
            log_debug(@"Coord: arrange_tunnel: passed $(nip_y_secondary) to $(nip_x_secondary)");
        }

        private void impl_forward_knowledge() throws Error
        {
            Tasklet.declare_self("Coordinator.forward_knowledge");
            Coord_hkey key = new Coord_hkey();
            key.level_of_gnode = level_of_gnode;
            key.nip = maproute.me;
            NIP hash_nip = parent.h(key);
            Gee.List<NIP> bunch = parent.find_nearest_to_register(hash_nip, COORD_DUPLICATION, level_of_gnode);
            NIP[] bunch_not_me = {};
            foreach (NIP n in bunch)
                if (! n.is_equal(maproute.me))
                    bunch_not_me += n;
            foreach (NIP to_nip in bunch_not_me)
                forward_knowledge_to_nip(to_nip);
        }

        private static void * helper_forward_knowledge(void *v) throws Error
        {
            struct_helper_Coordinator_forward_knowledge *tuple_p = (struct_helper_Coordinator_forward_knowledge *)v;
            // The caller function has to add a reference to the ref-counted instances
            Coordinator self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_forward_knowledge();
            // void method, return null
            return null;
        }

        /** Communicate to secondary coordinators all our current knowledge.
          */
        public void forward_knowledge()
        {
            struct_helper_Coordinator_forward_knowledge arg = struct_helper_Coordinator_forward_knowledge();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_forward_knowledge, &arg);
        }

        private void impl_forward_knowledge_to_nip(NIP to_nip) throws Error
        {
            Tasklet.declare_self("Coordinator.forward_knowledge_to_nip");
            string dest_addr = nip_to_str(maproute.levels, maproute.gsize, to_nip);
            string my_addr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            log_debug(@"Coord: forwarding knowledge to $dest_addr");
            AddressManagerTCPClient remote = new AddressManagerTCPClient(dest_addr, null, my_addr, false);
            try
            {
                remote.coordnode.duplicate_all_knowledge(gnode,
                            new CoordinatorKnowledge(gnode,
                                        get_bookings(),
                                        last_assigned_elderliness,
                                        get_bnode_list()));
                log_debug(@"Coord: forwarded knowledge to $dest_addr");
            }
            catch (Error e)
            {
                log_warn(@"'Coord: forwarded knowledge to $dest_addr got exception $(e.message)");
            }
        }

        private static void * helper_forward_knowledge_to_nip(void *v) throws Error
        {
            struct_helper_Coordinator_forward_knowledge_to_nip *tuple_p = (struct_helper_Coordinator_forward_knowledge_to_nip *)v;
            // The caller function has to add a reference to the ref-counted instances
            Coordinator self_save = tuple_p->self;
            NIP nip_save = tuple_p->nip;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_forward_knowledge_to_nip(nip_save);
            // void method, return null
            return null;
        }

        /** Forwards all current knowledge to a secondary coordinator.
          */
        public void forward_knowledge_to_nip(NIP nip)
        {
            struct_helper_Coordinator_forward_knowledge_to_nip arg = struct_helper_Coordinator_forward_knowledge_to_nip();
            arg.self = this;
            arg.nip = nip;
            Tasklet.spawn((FunctionDelegate)helper_forward_knowledge_to_nip, &arg);
        }
    }
}

