/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    public errordomain InetError {
        NO_NIP,
        GENERIC
    }
    string family;
    NipIpConverter? nip_ip_converter = null;
    bool initialized = false;
    void initialize()
    {
        if (!initialized)
        {
            family = "UNK";
            if (Settings.IP_VERSION == 4) family = "AF_INET";
            if (Settings.IP_VERSION == 6) family = "AF_INET6";
            nip_ip_converter = new class10_nip_to_ip();
            // nip_ip_converter = WholeNetworkIPv4NipToIp()
            initialized = true;
        }
    }

    public int lvl_to_bits(int levels, int gsize, int lvl)
    {
        initialize();
        return nip_ip_converter.lvl_to_bits(levels, gsize, lvl);
    }

    public int bits_to_lvl(int levels, int gsize, int bits)
    {
        initialize();
        return nip_ip_converter.bits_to_lvl(levels, gsize, bits);
    }

    public string nip_to_str(int levels, int gsize, NIP nip)
    {
        initialize();
        uchar[] pip = nip_ip_converter.nip_to_pip(levels, gsize, nip);
        return pip_to_dotted(family, pip);
    }

    public NIP str_to_nip(int levels, int gsize, string ipstr) throws InetError
    {
        initialize();
        uchar[] pip = dotted_to_pip(family, ipstr);
        return nip_ip_converter.pip_to_nip(levels, gsize, pip);
    }

    public uint8[] nip_to_octets(int levels, int gsize, NIP nip)
    {
        initialize();
        uchar[] pip = nip_ip_converter.nip_to_pip(levels, gsize, nip);
        uint8[] ret = new uint8[pip.length];
        for (int i = 0; i < pip.length; i++) ret[i] = (uint8)pip[i];
        return ret;
    }

    public ArrayList<int> valid_ids(int levels, int gsize, int lvl, PartialNIP partial_nip)
    {
        initialize();
        return nip_ip_converter.valid_ids(levels, gsize, lvl, partial_nip);
    }

    /** Given this gnode, returns its CIDR. E.g. 10.11.22.0/24
      * returns 2 strings
      */
    public void gnode_to_ip_cidr(PartialNIP gnode, int gsize, out string ipstr, out string bits)
    {
        NIP lowestnip = gnode.lowest_nip();
        int levels = gnode.levels;
        int given_up_to_level = gnode.level_of_gnode();
        ipstr = nip_to_str(levels, gsize, lowestnip);
        bits = lvl_to_bits(levels, gsize, given_up_to_level).to_string();
    }

    /** Inverse of gnode_to_ip_cidr.
      */
    public PartialNIP ip_cidr_to_gnode(int levels, int gsize, string ipstr, string bits)
    {
        int lvl = bits_to_lvl(levels, gsize, int.parse(bits));
        NIP nip = str_to_nip(levels, gsize, ipstr);
        return nip.get_gnode_at_level(lvl);
    }

    public interface NipIpConverter : Object
    {
        /** Returns bits corresponding to 'lvl'
          */
        public abstract int lvl_to_bits(int levels, int gsize, int lvl);

        /** Inverse of lvl_to_bits
          */
        public abstract int bits_to_lvl(int levels, int gsize, int bits);

        /** Converts the given pip to a nip
          */
        public abstract NIP pip_to_nip(int levels, int gsize, uchar[] pip) throws InetError;

        /** The reverse of pip_to_nip
          */
        public abstract uchar[] nip_to_pip(int levels, int gsize, NIP nip);

        /** Returns the list of valid IDs for level lvl, given that IDs for upper
          * levels are already choosen in nip.
          */
        public abstract ArrayList<int> valid_ids(int levels, int gsize, int lvl, PartialNIP partial_nip);
    }

    uint64 pip_to_ip(uchar[] pip)
    {
        uint64 ret = 0;
        for (int i = 0; i < pip.length; i++)
        {
            int c = (int)pip[pip.length-i-1];
            ret += c * (uint64)Math.pow(256, i);
        }
        return ret;
    }

    uchar[] ip_to_pip(uint64 ip, int numbytes)
    {
        uchar[] ret = new uchar[numbytes];
        int j = 0;
        for (int i = numbytes-1; i >= 0; i--)
        {
            uint64 modulo = (uint64)Math.pow(256, i+1);
            uint64 divisore = (uint64)Math.pow(256, i);
            uint64 resto = ip % modulo;
            uint64 quoziente = resto / divisore;
            int c = (int)quoziente;
            // int c = (int)((ip % (uint64)Math.pow(256, i+1)) / (uint64)Math.pow(256, i));
            ret[j++] = (uchar)c;
        }
        return ret;
    }

    /* TODO
    public class WholeNetworkIPv4NipToIp : Object, NipIpConverter
    {
    }
    */

    /** Represents class 10.* in IPv4
      */
    public class class10_nip_to_ip : Object, NipIpConverter
    {
        private int bitsperlevel(int gsize)
        {
            int ret = 0;
            while(1 << ret < gsize)
            {
                ret++;
            }
            return ret;
        }

        public int lvl_to_bits(int levels, int gsize, int lvl)
        {
            return 32 - lvl * bitsperlevel(gsize);
        }

        public int bits_to_lvl(int levels, int gsize, int bits)
        {
            return (32 - bits) / bitsperlevel(gsize);
        }

        public uchar[] nip_to_pip(int levels, int gsize, NIP nip)
        {
            uint64 ip = 0;
            for (int l = 0; l < levels; l++)
            {
                ip += nip.position_at(l) * (uint64)Math.pow(gsize, l);
            }
            return ip_to_pip(ip + 10 * (uint64)Math.pow(256, 3), 4);
        }

        public NIP pip_to_nip(int levels, int gsize, uchar[] pip) throws InetError
        {
            uint64 ip = pip_to_ip(pip);
            ip -= 10 * (int)Math.pow(256, 3);
            int[] _ret = new int[levels];
            for (int l = 0; l < levels; l++)
            {
                _ret[l] = (int)((ip % (uint64)Math.pow(gsize, l+1)) / (uint64)Math.pow(gsize, l));
            }
            if (ip / (uint64)Math.pow(gsize, levels) > 0) throw new InetError.NO_NIP("Outside of Netsukuku realm.");
            return new NIP(_ret);
        }

        public ArrayList<int> valid_ids(int levels, int gsize, int lvl, PartialNIP partial_nip)
        {
            ArrayList<int> ret = new ArrayList<int>();
            for (int i = 0; i < gsize; i++) ret.add(i);
            return ret;
        }
    }
}
