/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;
using zcd;

namespace Netsukuku
{
#if log_tasklet
    private string tasklet_id()
    {
        string ret = @"$(Tasklet.self().id)";
        int len = ret.length;
        for (int i = 0; i < 5-len; i++) ret = " " + ret;
        return @"[$(ret)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
                    tasklet_id() + "DEBUG "  + msg);}
    internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                    tasklet_id() + "INFO "   + msg);}
    internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                    tasklet_id() + "INFO+ "  + msg);}
    internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                    tasklet_id() + "INFO++ " + msg);}
    internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                    tasklet_id() + "ERROR "  + msg);}
    internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                    tasklet_id() + "ERROR+ " + msg);}

    public class Ntkd
    {
        static int verbosity;
        static bool nicreset;
        static string config_file;
        [CCode (array_length = false, array_null_terminated = true)]
        static string[] interfaces;
        [CCode (array_length = false, array_null_terminated = true)]
        static string[] excluded;

        public static int main(string[] args)
        {
            verbosity = -1;
            nicreset = false;
            config_file = "";
            OptionContext oc = new OptionContext("- Netsukuku routing daemon");
            OptionEntry[] entries = new OptionEntry[6];
            int index = 0;
            entries[index++] = {"config", 'c', 0, OptionArg.FILENAME, ref config_file, "Configuration file", null};
            entries[index++] = {"interface", 'i', 0, OptionArg.STRING_ARRAY, ref interfaces, "Interface to be managed", null};
            entries[index++] = {"exclude", 'e', 0, OptionArg.STRING_ARRAY, ref excluded, "Interface to be excluded", null};
            entries[index++] = {"verbosity", 'v', 0, OptionArg.INT, ref verbosity, "Verbosity level 0..4 (overrides configuration file, default: 0)", null};
            entries[index++] = {"nic-reset", 0, 0, OptionArg.NONE, ref nicreset, "Force NIC reset", null};
            entries[index++] = { null };
            oc.add_main_entries(entries, null);
            try {
                oc.parse(ref args);
            }
            catch (OptionError e) {
                print(@"Error parsing options: $(e.message)\n");
                return 1;
            }

            // CONFIGURATION_FILE has to be set (if ever) before any other access to Settings.
            if (config_file != "") Settings.init(config_file);

            // Initialize tasklets
            Tasklet.init(Settings.THREAD_STACK_SIZE);
            Tasklets.init_stats(
                /*Stat CreateTaskletStat()*/
                () => {
                    return new TaskletStats();
                },
                /*void TaskletEvent(Stat tasklet, EventType event_type)*/
                (tasklet, event_type) => {
                    TaskletStats st = (TaskletStats)tasklet;
                    if (event_type == EventType.STARTED) st.tasklet_started = new TimeCapsule(0);
                    if (event_type == EventType.CRASHED ||
                        event_type == EventType.ENDED ||
                        event_type == EventType.ABORTED) st.tasklet_ended = new TimeCapsule(0);
                });
            // error? Tasklet.declare_self("MAIN");

            // Initialize rpc library
            Serializer.init();

            // Register serializable types from sample model
            Andns.RpcAndns.init();
            RpcNtk.init();

            // Register OS-dependant modules
            ImplLinux.LinuxNIC.register();
            ImplLinux.LinuxRoute.register();
            ImplLinux.LinuxTunnel.register();

            if (verbosity >= 0)
            {
                if (verbosity > 4) verbosity = 4;
                Settings.VERBOSE_LEVEL = verbosity;
            }

            Posix.openlog("netsukuku", Posix.LOG_PID, Posix.LOG_USER);
            int loglevel = Posix.LOG_ERR;
            switch (Settings.VERBOSE_LEVEL)
            {
            case 1:
                loglevel = Posix.LOG_WARNING;
                break;
            case 2:
                loglevel = Posix.LOG_NOTICE;
                break;
            case 3:
                loglevel = Posix.LOG_INFO;
                break;
            case 4:
                loglevel = Posix.LOG_DEBUG;
                break;
            }
            Posix.setlogmask(Posix.LOG_UPTO(loglevel));

            Settings.NICS = new ArrayList<string>();
            Settings.EXCLUDE_NICS = new ArrayList<string>();
            foreach (string intf in interfaces)
                Settings.NICS.add(intf);
            foreach (string excl in excluded)
                Settings.EXCLUDE_NICS.add(excl);
            Settings.FORCE_NIC_RESET = nicreset;
            
            // start netsukuku daemon
            me = new NtkNode();
            me.run();
            // start andns server
            Andns.AndnsServer.start_operations();

            // register handlers for SIGHUP to reload hostnames and restart andna
            Posix.signal(Posix.SIGHUP, reload_andna);
            // register handlers for SIGINT and SIGTERM to exit
            Posix.signal(Posix.SIGINT, safe_exit);
            Posix.signal(Posix.SIGTERM, safe_exit);
            // waits for ever
            while (true)
            {
                Tasklet.nap(1, 0);
                if (do_me_exit)
                {
                    do_me_exit = false;
                    me.exit();
                }
            }
        }

        static NtkNode? me;
        static bool do_me_exit = false;

        public static void safe_exit(int signal)
        {
            // We got here because of a signal.
            // We could be stealing the control of the program from a pth
            //  that has not yet saved its data... or for any other reason
            //  we are in a state where we cannot reliably spawn a new tasklet.
            // When we return the control, we should be again in a coherent state.
            // For now, it is vital that we use a different approach to the spawning
            //  of a new tasklet.
            do_me_exit = true;
        }

        public static void reload_andna(int signal)
        {
            // TODO reload hostnames and restart andna
        }
    }
}

