/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    struct struct_helper_Etp_purge_incoming_nodes_periodically
    {
        public Etp self;
    }

    struct struct_helper_Etp_warn_my_gateways_periodically
    {
        public Etp self;
    }

    struct struct_helper_Etp_readvertise_periodically
    {
        public Etp self;
    }

    struct struct_helper_Etp_request_updates_dispatcher
    {
        public Etp self;
        public AggregatedNeighbour aggregated_neighbour;
    }

    struct struct_helper_Etp_collision_rehook
    {
        public Etp self;
        public NetworkID neighbour_netid;
    }

    struct struct_helper_Etp_advertise_modification_to_routes
    {
        public Etp self;
        public TracerPacketList? tpl;
        public REM? gwrem;
    }

    struct struct_helper_Etp_advertise_modification_to_gnodeid
    {
        public Etp self;
    }

    struct struct_helper_Etp_etp_exec_dispatcher
    {
        public Etp self;
    }

    struct struct_helper_Etp_etp_new_link
    {
        public Etp self;
        public AggregatedNeighbour aggregated_neighbour;
    }

    struct struct_helper_Etp_etp_changed_link
    {
        public Etp self;
        public AggregatedNeighbour aggregated_neighbour;
        public REM oldrem;
    }

    struct struct_helper_Etp_etp_dead_link
    {
        public Etp self;
        public AggregatedNeighbour aggregated_neighbour;
    }

    public class Etp : Object, IEtp
    {
        public weak AggregatedNeighbourManager aggregated_neighbour_manager {get; private set;}
        public weak MapRoute maproute {get; private set;}
        public weak AddressManager address_manager {get; private set;}
        public TracerPacketList? last_tpl {get; private set; default = null; }
        public REM? last_gwrem {get; private set; default = null; }
        // 1 because the first thing I have to say to my incoming nodes is the path to myself.
        public int my_curr_seq_num {get; private set; default = 1; }
        private Tasklet? readvertise_handle = null;
        private Tasklet? warn_my_gateways_handle = null;
        private Tasklet? purge_incoming_nodes_handle = null;
        public int num_etp_in {get; private set; default = 0; }
        public int num_etp_out {get; private set; default = 0; }
        private HashMap<AggregatedNeighbour, Channel> request_updates_dispatcher_channels;
        private Channel? etp_exec_dispatcher_channel = null;

        public signal void net_collision(Gee.List<AggregatedNeighbour> others);
        public signal void sent_etp(NIP nip_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller, ExtendedTracerPacket etp);
        public signal void etp_executed();

        public Etp(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, AddressManager address_manager)
        {
            this.aggregated_neighbour_manager = aggregated_neighbour_manager;
            this.maproute = maproute;
            this.address_manager = address_manager;

            this.aggregated_neighbour_manager.aggregated_neighbour_new.connect(etp_new_link);
            this.aggregated_neighbour_manager.aggregated_neighbour_colliding_new.connect(etp_new_link);
            this.aggregated_neighbour_manager.aggregated_neighbour_rem_chged.connect(etp_changed_link);
            this.aggregated_neighbour_manager.aggregated_neighbour_deleted.connect(etp_dead_link);
            this.aggregated_neighbour_manager.aggregated_neighbour_going_rem_chged.connect(etp_before_changed_link);
            this.aggregated_neighbour_manager.aggregated_neighbour_going_deleted.connect(etp_before_dead_link);

            readvertise_periodically();
            warn_my_gateways_periodically();
            purge_incoming_nodes_periodically();

            request_updates_dispatcher_channels = new HashMap<AggregatedNeighbour, Channel>(AggregatedNeighbour.hash_func, AggregatedNeighbour.equal_func);
        }

        public void etp_in()
        {
            num_etp_in++;
        }

        public void etp_out()
        {
            num_etp_out++;
        }

        public QspnStats report_qspn_stats()
        {
            log_debug("report_qspn_stats start");
            return new QspnStats(num_etp_in, num_etp_out, address_manager.is_mature);
        }

        public void stop_operations()
        {
            string ipstr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            log_debug(@"Etp: stopping operations for $(ipstr)");
            if (readvertise_handle != null)
            {
                log_debug(@"Etp: aborting readvertise for $(ipstr)");
                readvertise_handle.abort();
                readvertise_handle = null;
            }
            if (warn_my_gateways_handle != null)
            {
                log_debug(@"Etp: aborting warn_my_gateways for $(ipstr)");
                warn_my_gateways_handle.abort();
                warn_my_gateways_handle = null;
            }
            if (purge_incoming_nodes_handle != null)
            {
                log_debug(@"Etp: aborting purge_incoming_nodes for $(ipstr)");
                purge_incoming_nodes_handle.abort();
                purge_incoming_nodes_handle = null;
            }
        }

        private void impl_purge_incoming_nodes_periodically() throws Error
        {
            Tasklet.declare_self("Etp.purge_incoming_nodes_periodically");
            purge_incoming_nodes_handle = Tasklet.self();
            while (true)
            {
                ms_wait(2000);
                address_manager.incoming_nodes.purge();
            }
        }

        private static void * helper_purge_incoming_nodes_periodically(void *v) throws Error
        {
            struct_helper_Etp_purge_incoming_nodes_periodically *tuple_p = (struct_helper_Etp_purge_incoming_nodes_periodically *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_purge_incoming_nodes_periodically();
            // void method, return null
            return null;
        }

        public void purge_incoming_nodes_periodically()
        {
            struct_helper_Etp_purge_incoming_nodes_periodically arg = struct_helper_Etp_purge_incoming_nodes_periodically();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_purge_incoming_nodes_periodically, &arg);
        }

        private void impl_warn_my_gateways_periodically() throws Error
        {
            Tasklet.declare_self("Etp.warn_my_gateways_periodically");
            warn_my_gateways_handle = Tasklet.self();
            log_debug("warn_my_gateways_periodically: started.");
            string my_addr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            while (true)
            {
                ms_wait(20000);
                log_debug("warn_my_gateways_periodically: warn_my_gateways.");
                Gee.List<AggregatedNeighbour> current_nr_list = aggregated_neighbour_manager.neighbour_list(true);
                foreach (AggregatedNeighbour aggregated_neighbour in current_nr_list)
                {
                    // List my MACs (they could change)
                    ArrayList<string> my_macs = new ArrayList<string>();
                    foreach (NetworkInterfaceManager nic_man in address_manager.nics)
                    {
                        if (nic_man.to_be_managed)
                            my_macs.add(nic_man.nic_class.mac);
                    }
                    AddressManagerFakeRmt nclient = aggregated_neighbour.create_neighbour_client(my_addr, false);
                    nclient.etp.act_as_gateway(maproute.me,
                            address_manager.get_my_id(),
                            address_manager.get_main_netid(), my_macs);
                }
            }
        }

        private static void * helper_warn_my_gateways_periodically(void *v) throws Error
        {
            struct_helper_Etp_warn_my_gateways_periodically *tuple_p = (struct_helper_Etp_warn_my_gateways_periodically *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_warn_my_gateways_periodically();
            // void method, return null
            return null;
        }

        public void warn_my_gateways_periodically()
        {
            struct_helper_Etp_warn_my_gateways_periodically arg = struct_helper_Etp_warn_my_gateways_periodically();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_warn_my_gateways_periodically, &arg);
        }

        /** The caller (nip) is in my network and wants me to act
          * as a gateway for him.
          * I am told its MACs.
          */
        public void act_as_gateway(NIP nip_caller, int nodeid_caller, NetworkID netid_caller, Gee.List<string> macs)
        {
            // We choose if we want to be a gateway for this nip,netid.
            if (! address_manager.do_i_act_as_gateway_for(nip_caller, nodeid_caller, netid_caller))
                return;
            acting_as_gateway_for(nip_caller, nodeid_caller, netid_caller, macs);
        }

        private void acting_as_gateway_for(NIP nip_caller, int nodeid_caller, NetworkID netid_caller, Gee.List<string> macs)
        {
            // update our knowledge (of this AddressManager) about the incoming nodes.
            address_manager.incoming_nodes.update(nip_caller, nodeid_caller, macs);
            // eventually, events INCOMING_NODE_UPDATED will propagate.
        }

        // request_updates should be carried on (at the same time) by at most one tasklet per each
        // pair (address_manager, aggregated_neighbour).
        public void request_updates(AggregatedNeighbour aggregated_neighbour)
        {
            if (request_updates_dispatcher_channels.has_key(aggregated_neighbour))
            {
                request_updates_dispatcher_channels[aggregated_neighbour].send_async(0);
            }
            else
            {
                request_updates_dispatcher_channels[aggregated_neighbour] = new Channel();
                request_updates_dispatcher_channels[aggregated_neighbour].send_async(0);
                request_updates_dispatcher(aggregated_neighbour);
            }
        }

        private void impl_request_updates_dispatcher(AggregatedNeighbour aggregated_neighbour) throws Error
        {
            Tasklet.declare_self("Etp.request_updates_dispatcher");
            try
            {
                while (request_updates_dispatcher_channels[aggregated_neighbour].balance > 0)
                {
                    request_updates_dispatcher_channels[aggregated_neighbour].recv();
                    try
                    {
                        impl_request_updates(aggregated_neighbour);
                    }
                    catch (Error e)
                    {
                        log_error(@"Uncaught exception in request_updates $(aggregated_neighbour): $(e.message)");
                    }
                }
            }
            finally
            {
                request_updates_dispatcher_channels.unset(aggregated_neighbour);
            }
        }

        private static void * helper_request_updates_dispatcher(void *v) throws Error
        {
            struct_helper_Etp_request_updates_dispatcher *tuple_p = (struct_helper_Etp_request_updates_dispatcher *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            AggregatedNeighbour aggregated_neighbour_save = tuple_p->aggregated_neighbour;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_request_updates_dispatcher(aggregated_neighbour_save);
            // void method, return null
            return null;
        }

        public void request_updates_dispatcher(AggregatedNeighbour aggregated_neighbour)
        {
            struct_helper_Etp_request_updates_dispatcher arg = struct_helper_Etp_request_updates_dispatcher();
            arg.self = this;
            arg.aggregated_neighbour = aggregated_neighbour;
            Tasklet.spawn((FunctionDelegate)helper_request_updates_dispatcher, &arg);
        }

        public void impl_request_updates(AggregatedNeighbour aggregated_neighbour) throws Error
        {
            log_debug(@"request_updates: going to ask to $(aggregated_neighbour)");
            // List my MACs
            ArrayList<string> my_macs = new ArrayList<string>();
            foreach (NetworkInterfaceManager nic_man in address_manager.nics)
            {
                if (nic_man.to_be_managed)
                    my_macs.add(nic_man.nic_class.mac);
            }
            // Do the request
            ExtendedTracerPacket etp;
            try
            {
                log_debug("request_updates: doing request...");
                etp = aggregated_neighbour.neighbour_client.etp.request_etp(
                                aggregated_neighbour.mod_seq_num, maproute.me,
                                address_manager.get_my_id(),
                                address_manager.get_main_netid(), my_macs);
                log_debug(@"request_updates: got etp from $(aggregated_neighbour).");
                log_debug(@"I knew it by $(aggregated_neighbour.mod_seq_num), now it is $(etp.seq_num)");
            }
            catch (Error e)
            {
                if (e is QspnError.ALREADY_UP_TO_DATE) return;
                log_error(@"Uncaught exception in request_updates $(aggregated_neighbour): $(e.message)");
                // We should remove aggregated_neighbour as a valid neighbour...
                // TODO
                return;
            }
            log_debug("request_updates: processing etp...");
            etp_exec(aggregated_neighbour, etp);
        }

        /** This remotable is called to request an ETP
          */
        public ExtendedTracerPacket request_etp(int prev_mod_seq_num, NIP nip_caller,
                                int nodeid_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller) throws QspnError
        {
            // We choose if we want to be a gateway for this nip,netid.
            if (! address_manager.do_i_act_as_gateway_for(nip_caller, nodeid_caller, netid_caller))
                throw new QspnError.NOT_YOUR_GATEWAY("The node refuses to be your gateway.");
            if (! address_manager.is_in_my_network(netid_caller))
                throw new QspnError.NOT_YOUR_GATEWAY("The node is not in your network.");
            if (prev_mod_seq_num >= my_curr_seq_num)
                throw new QspnError.ALREADY_UP_TO_DATE("You already have my paths up to date.");

            // What were the previous destinations known by this incoming node through me?
            RoutesSet prev_dest = new RoutesSet(maproute.levels);
            if (address_manager.incoming_nodes.contains(nip_caller, nodeid_caller))
            {
                RoutesSet? x = null;
                try
                {
                    x = address_manager.incoming_nodes.get_knowledge(nip_caller, nodeid_caller, prev_mod_seq_num);
                }
                catch (IncomingNodesError e)
                {
                    log_warn(@"request_etp: reported an error: $(e.message); sending all routes again.");
                }
                if (x != null) prev_dest = x;
            }
            HCoord caller = maproute.nip_to_lvlid(nip_caller);
            log_debug(@"request_etp: we must send etp to caller $(nip_caller)");
            log_debug(@"request_etp: prev_dest $(prev_dest)");

            // Gather paths to each destination that we want to make accessible to nip_caller.
            RoutesSet destinations = gather_destinations(nip_caller);

            // We must add a route to ourself.
            destinations.add_replace(caller.lvl, maproute.me.position_at(caller.lvl),
                                    new NullREM(), new ArrayList<HCoord>(HCoord.equal_func),
                                    maproute.id_myself[caller.lvl]);
            log_debug("request_etp: maproute analyzed");
            log_debug(@"request_etp: destinations $(destinations)");
            // Compare with prev_dest
            RoutesSet changed_routes = new RoutesSet(maproute.levels);
            for (int lvl = 0; lvl < destinations.per_level.length; lvl++)
            {
                foreach (int dst in destinations.per_level[lvl])
                {
                    if (! prev_dest.per_level[lvl].has_key(dst))
                    {
                        RouteInSet? ris_dest = destinations.per_level[lvl].get_value(dst);
                        if (ris_dest == null)
                        {
                            // no paths to destination through me for this (g)node...
                            // TODO do we have to include DeadRem in this etp?
                        }
                        else
                        {
                            // a new path
                            changed_routes.add_replace(lvl, dst, ris_dest.rem, ris_dest.hops, ris_dest.gid);
                        }
                    }
                    else
                    {
                        RouteInSet? ris_dest = destinations.per_level[lvl].get_value(dst);
                        if (ris_dest == null)
                        {
                            // no paths to destination through me for this (g)node...
                            if (prev_dest.per_level[lvl].get_value(dst) == null)
                            {
                                // it was already known. nothing to communicate.
                            }
                            else
                            {
                                // we have to communicate a DeadRem.
                                changed_routes.add_replace_no_route(lvl, dst);
                            }
                        }
                        else
                        {
                            // there is a path. did it change?
                            // NOTE TODO: if we want to compress prev_dest (to save memory)
                            //  we just need to ensure that this comparison (equality)
                            //  reliably works most of time. When we cannot be sure we
                            //  could safely act as if they are not equal, but if this happens
                            //  too often then this become useless because we do not save
                            //  net trafic.
                            RouteInSet? ris_prev_dest = prev_dest.per_level[lvl].get_value(dst);
                            if (! RouteInSet.equal_func(ris_prev_dest, ris_dest))
                            {
                                // it is new, or it changed. we communicate.
                                changed_routes.add_replace(lvl, dst, ris_dest.rem, ris_dest.hops, ris_dest.gid);
                            }
                        }
                    }
                }
            }
            for (int lvl = 0; lvl < prev_dest.per_level.length; lvl++)
            {
                foreach (int dst in prev_dest.per_level[lvl])
                {
                    if (! destinations.per_level[lvl].has_key(dst))
                    {
                        // no paths to destination through me for this (g)node...
                        if (prev_dest.per_level[lvl].get_value(dst) == null)
                        {
                            // TODO do we have to include DeadRem in this etp?
                        }
                        else
                        {
                            // a path is no more.
                            changed_routes.add_replace_no_route(lvl, dst);
                        }
                    }
                }
            }
            log_debug("request_etp: changes analyzed");
            log_debug(@"request_etp: changed_routes $(changed_routes)");
            // prepare TPL.
            TracerPacketList tpl;
            if (last_tpl != null && prev_mod_seq_num == my_curr_seq_num+1)
            {
                tpl = last_tpl.clone();
                // TODO review the PAR-EER rule. In which cases do I have to create a new TPL?
                tpl.append(maproute.me.position_at(0), last_gwrem);
            }
            else
            {
                tpl = new TracerPacketList();
                tpl.append(maproute.me.position_at(0), new NullREM());
            }

            // prepare ETP.
            ExtendedTracerPacket etp = new ExtendedTracerPacket();
            etp.r = changed_routes;
            etp.tpl = tpl;
            etp.seq_num = my_curr_seq_num;
            log_debug(@"request_etp: final etp $(etp)");

            // signal that we are used as a gateway
            acting_as_gateway_for(nip_caller, nodeid_caller, netid_caller, macs_caller);
            // signal that we sent an ETP
            if (address_manager.incoming_nodes.contains(nip_caller, nodeid_caller))
            {
                try
                {
                    address_manager.incoming_nodes.log_knowledge(nip_caller, nodeid_caller, my_curr_seq_num, destinations);
                }
                catch (IncomingNodesError e)
                {
                    log_error(@"request_etp: reported an error: $(e.message); this shoud not happen.");
                }
            }
            log_debug("request_etp: sending etp");
            sent_etp(nip_caller, netid_caller, macs_caller, etp);
            etp_out();
            // we send the etp, as answer
            return etp;
        }

        public virtual RoutesSet gather_destinations(NIP nip_caller)
        {
            // Override in BnodeTunneling, return new RoutesSet(maproute.levels);
            // Evaluate, for all destinations, the best path whitout nip_caller
            // and use them in kernel routing table for packets coming from macs_caller.
            // This is done calling self.act_as_gateway.
            RoutesSet ret = new RoutesSet(maproute.levels);
            HCoord caller = maproute.nip_to_lvlid(nip_caller);
            // List my best-paths not containing this (g)node
            // for destinations we have in common
            for (int lvl = maproute.levels-1; lvl >= caller.lvl; lvl--)
            {
                for (int dst = 0; dst < maproute.gsize; dst++)
                {
                    // lvl,dst is a destination
                    RouteNode routes_to_v = maproute.node_get(lvl, dst);
                    if (routes_to_v.is_empty())
                    {
                        // destination does not exist...
                    }
                    else
                    {
                        // destination exists...
                        Route? best_without = routes_to_v.best_route_without(caller);
                        if (best_without == null)
                        {
                            // no paths to destination through me for this (g)node...
                            ret.add_replace_no_route(lvl, dst);
                        }
                        else
                        {
                            // a path to destination through me for this (g)node...
                            // purify hops:
                            Gee.List<HCoord> hops = best_without.hops_with_gw;
                            hops = maproute.list_lvl_id_to_nip(hops, nip_caller);
                            // save info
                            ret.add_replace(lvl, dst, best_without.rem, hops, best_without.gid);
                        }
                    }
                }
            }
            return ret;
        }

        /** Answer the size of my network.
          */
        public int request_size()
        {
            return evaluate_my_size();
        }

        /** Evaluate the size of my network.
          */
        public int evaluate_my_size()
        {
            // TODO find a way to estimate real size of the network.
            int tot = 1;
            for (int lvl = 0; lvl < maproute.levels; lvl++)
                tot *= maproute.busy_nodes_nb(lvl);
            return tot;
        }

        public void new_sequence_number(NIP nip_x, int nodeid_x, int seq_num_x)
        {
            /* If we have not x as a neighbour, ignore the message.
             * If we have the neighbour, and it is in another network, then decide if we have to rehook.
             * If we have the neighbour, but the current modification_sequence_number
             *    is greater than the one in the current message, ignore the message.
             * Otherwise, ask for updates.
            */
            log_debug(@"new_sequence_number: ($nip_x, $nodeid_x) changes state");
            AggregatedNeighbour? aggregated_neighbour =
                        aggregated_neighbour_manager.key_to_neighbour(nip_x, nodeid_x);
            if (aggregated_neighbour != null)
            {
                NetworkID my_netid = address_manager.get_main_netid();
                if (address_manager.is_in_my_network(aggregated_neighbour.netid))
                {
                    log_debug("new_sequence_number: it is in my network");
                    if (aggregated_neighbour.mod_seq_num < seq_num_x)
                    {
                        log_debug("new_sequence_number: asking updates...");
                        request_updates(aggregated_neighbour);
                    }
                }
                else
                {
                    // A collision. Ask for their size.
                    log_debug(@"new_sequence_number: a collision ($(aggregated_neighbour.netid), $(my_netid)), asking size...");
                    int their_sz;
                    try
                    {
                        their_sz = aggregated_neighbour.neighbour_client.etp.request_size();
                    }
                    catch (Error e)
                    {
                        log_info(@"Etp.new_sequence_number: tried to ask for size of network to $(aggregated_neighbour)"
                                + @" but got Exception $(e.message)");
                        return;
                    }
                    log_debug(@"new_sequence_number: their_sz = $their_sz");
                    int our_sz = evaluate_my_size();
                    log_debug(@"new_sequence_number: our_sz = $our_sz");
                    // decide what to do
                    if (their_sz > our_sz ||
                                (their_sz == our_sz &&
                                address_manager.is_preferred_network(aggregated_neighbour.netid)))
                    {
                        log_debug("new_sequence_number: going to rehook.");
                        collision_rehook(aggregated_neighbour.netid);
                    }
                    else
                    {
                        log_debug("new_sequence_number: we do not rehook.");
                    }
                }
            }
        }

        private void impl_readvertise_periodically() throws Error
        {
            Tasklet.declare_self("Etp.readvertise_periodically");
            readvertise_handle = Tasklet.self();
            log_debug("readvertise_periodically: started.");
            while (true)
            {
                try
                {
                    ms_wait(50000);
                    log_debug("readvertise_periodically: broadcast_my_sequence_number.");
                    broadcast_my_sequence_number();
                }
                catch (Error e)
                {
                    log_error(@"readvertise_periodically: reported an error: $(e.message).");
                }
            }
        }

        private static void * helper_readvertise_periodically(void *v) throws Error
        {
            struct_helper_Etp_readvertise_periodically *tuple_p = (struct_helper_Etp_readvertise_periodically *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_readvertise_periodically();
            // void method, return null
            return null;
        }

        public void readvertise_periodically()
        {
            struct_helper_Etp_readvertise_periodically arg = struct_helper_Etp_readvertise_periodically();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_readvertise_periodically, &arg);
        }

        private void impl_advertise_modification_to_routes(TracerPacketList? tpl, REM? gwrem) throws Error
        {
            Tasklet.declare_self("Etp.advertise_modification_to_routes");
            // override in BnodeTunneling, do nothing.
            my_curr_seq_num += 1;
            log_debug(@"Etp: advertise_modification_to_routes: my sequence number is now $my_curr_seq_num");
            last_tpl = tpl;
            last_gwrem = gwrem;
            broadcast_my_sequence_number();
        }

        private static void * helper_advertise_modification_to_routes(void *v) throws Error
        {
            struct_helper_Etp_advertise_modification_to_routes *tuple_p = (struct_helper_Etp_advertise_modification_to_routes *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            TracerPacketList? tpl_save = tuple_p->tpl;
            REM? gwrem_save = tuple_p->gwrem;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_advertise_modification_to_routes(tpl_save, gwrem_save);
            // void method, return null
            return null;
        }

        public virtual void advertise_modification_to_routes(TracerPacketList? tpl=null, REM? gwrem=null)
        {
            struct_helper_Etp_advertise_modification_to_routes arg = struct_helper_Etp_advertise_modification_to_routes();
            arg.self = this;
            arg.tpl = tpl;
            arg.gwrem = gwrem;
            Tasklet.spawn((FunctionDelegate)helper_advertise_modification_to_routes, &arg);
        }

        private void impl_advertise_modification_to_gnodeid() throws Error
        {
            Tasklet.declare_self("Etp.advertise_modification_to_gnodeid");
            my_curr_seq_num += 1;
            log_debug(@"Etp: advertise_modification_to_gnodeid: my sequence number is now $my_curr_seq_num");
            broadcast_my_sequence_number();
        }

        private static void * helper_advertise_modification_to_gnodeid(void *v) throws Error
        {
            struct_helper_Etp_advertise_modification_to_gnodeid *tuple_p = (struct_helper_Etp_advertise_modification_to_gnodeid *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_advertise_modification_to_gnodeid();
            // void method, return null
            return null;
        }

        public void advertise_modification_to_gnodeid()
        {
            struct_helper_Etp_advertise_modification_to_gnodeid arg = struct_helper_Etp_advertise_modification_to_gnodeid();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_advertise_modification_to_gnodeid, &arg);
        }

        public void broadcast_my_sequence_number() throws Error
        {
            // we need a BroadcastClient
            address_manager.get_broadcast_client()
                    .etp.new_sequence_number(maproute.me,
                    address_manager.get_my_id(), my_curr_seq_num);
        }

        // etp_exec should be carried on (at the same time) by at most one tasklet per address_manager.
        public void etp_exec(AggregatedNeighbour aggregated_neighbour, ExtendedTracerPacket etp)
        {
            if (etp_exec_dispatcher_channel != null)
            {
                etp_exec_dispatcher_channel.send_async(aggregated_neighbour);
                etp_exec_dispatcher_channel.send_async(etp);
            }
            else
            {
                etp_exec_dispatcher_channel = new Channel();
                etp_exec_dispatcher_channel.send_async(aggregated_neighbour);
                etp_exec_dispatcher_channel.send_async(etp);
                etp_exec_dispatcher();
            }
        }

        private void impl_etp_exec_dispatcher() throws Error
        {
            Tasklet.declare_self("Etp.etp_exec_dispatcher");
            try
            {
                assert(etp_exec_dispatcher_channel != null);
                while (etp_exec_dispatcher_channel.balance > 0)
                {
                    Value v1 = etp_exec_dispatcher_channel.recv();
                    Value v2 = etp_exec_dispatcher_channel.recv();
                    AggregatedNeighbour aggregated_neighbour = (AggregatedNeighbour)v1;
                    ExtendedTracerPacket etp = (ExtendedTracerPacket)v2;
                    try
                    {
                        impl_etp_exec(aggregated_neighbour, etp);
                    }
                    catch (Error e)
                    {
                        log_error(@"Uncaught exception in etp_exec $(aggregated_neighbour) $(etp): $(e.message)");
                    }
                }
            }
            finally
            {
                etp_exec_dispatcher_channel = null;
            }
        }

        private static void * helper_etp_exec_dispatcher(void *v) throws Error
        {
            struct_helper_Etp_etp_exec_dispatcher *tuple_p = (struct_helper_Etp_etp_exec_dispatcher *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_etp_exec_dispatcher();
            // void method, return null
            return null;
        }

        public void etp_exec_dispatcher()
        {
            struct_helper_Etp_etp_exec_dispatcher arg = struct_helper_Etp_etp_exec_dispatcher();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_etp_exec_dispatcher, &arg);
        }

        /** Executes an ETP received from a neighbour which is for sure of my same network.
          */
        public void impl_etp_exec(AggregatedNeighbour aggregated_neighbour, ExtendedTracerPacket etp) throws Error
        {
            // Since etp_exec and request_updates are executed in different tasklets, we could get called
            //  with aggregated_neighbour.mod_seq_num >= etp.seq_num . In that case, ignore the ETP.
            //  Otherwise there would be alot of ETPs.
            if (aggregated_neighbour.mod_seq_num >= etp.seq_num) return;

            try
            {
                etp_in();
                NIP gwnip = aggregated_neighbour.nip;
                int level = maproute.nip_cmp(gwnip.get_positions());

                // Purify map portion R from destinations that are not in
                // a common gnode
                for (int lvl = 0; lvl < level; lvl++)
                    etp.r.per_level[lvl] = new RoutesSetPerLevel();

                // Purify map portion R from hops that are not in
                // a common gnode
                for (int lvl = level; lvl < etp.r.per_level.length; lvl++)
                {
                    ArrayList<int> dsts = new ArrayList<int>();
                    foreach (int dst in etp.r.per_level[lvl]) dsts.add(dst);
                    foreach (int dst in dsts)
                    {
                        RouteInSet? ris = etp.r.get_value(lvl, dst);
                        Gee.List<HCoord> hops = maproute.list_lvl_id_from_nip(ris.hops, gwnip);
                        etp.r.add_replace(lvl, dst, ris.rem, hops, ris.gid);
                    }
                }

                // Group rule
                etp.tpl.group(level, gwnip.position_at(level));

                // Acyclic rule
                if (etp.tpl.contains(maproute.me))
                {
                    log_debug("ETP received: Executing: Ignore ETP because of Acyclic Rule.");
                    // drop the pkt
                    return;
                }

                log_debug(@"Translated ETP from $(nip_to_str(maproute.levels, maproute.gsize, gwnip)): $etp");

                if (etp.r.is_empty)
                {
                    log_debug("ETP received: Executing: Ignore ETP because no changed routes for me.");
                    // drop the pkt
                    return;
                }

                // ∀r ∈ ETP
                for (int lvl = 0; lvl < etp.r.per_level.length; lvl++)
                {
                    foreach (int dst in etp.r.per_level[lvl])
                    {
                        RouteInSet ris = etp.r.get_value(lvl, dst);
                        // for sure ris is not null. See TODO above about add_replace_no_route.
                        string str_hops = "<List ";
                        string next = "";
                        foreach (HCoord hop in ris.hops)
                        {
                            str_hops += next + hop.to_string();
                            next = ",";
                        }
                        str_hops += ">";
                        log_debug(@"Will process r received in a ETP by $(aggregated_neighbour)" +
                                     @": lvl, dst, rem, hops = ($lvl, $dst, $(ris.rem), $(str_hops))");
                        // rem is Bestᵗ(C → v) with v ∈ V
                        //   where C is our neighbour
                        //   and v = (lvl, dst)
                        // For proposition 3.2 hops should not contain ourself.
                        // D updates rᵗ⁺¹D (C → v) = Bestᵗ (C → v)
                        RouteNode routes_to_v = maproute.node_get(lvl, dst);
                        log_debug("before update:");
                        log_debug(@"  # routes = $(routes_to_v.nroutes())");
                        Route? prev_route = routes_to_v.route_get_by_gw(aggregated_neighbour);
                        if (prev_route != null)
                            log_debug(@"  rᵗD (C → v) = $(prev_route.rem) hops $(prev_route.hops_with_gw.size)" +
                                         @" gid $(prev_route.gid)");
                        Route? prev_best_route = routes_to_v.best_route();
                        if (prev_best_route != null)
                            log_debug(@"  Bestᵗ (D → v) = $(prev_best_route.rem) hops $(prev_best_route.hops_with_gw.size)" +
                                         @" gid $(prev_best_route.gid)");

                        maproute.update_route_by_gw(new HCoord(lvl, dst), aggregated_neighbour, ris.rem, ris.hops, ris.gid);
                        log_debug("after update:");
                        log_debug(@"  # routes = $(routes_to_v.nroutes())");
                        Route? curr_route = routes_to_v.route_get_by_gw(aggregated_neighbour);
                        if (curr_route != null)
                            log_debug(@"  rᵗ⁺¹D (C → v) = $(curr_route.rem) hops $(curr_route.hops_with_gw.size)" +
                                         @" gid $(curr_route.gid)");
                        Route? curr_best_route = routes_to_v.best_route();
                        if (curr_best_route != null)
                            log_debug(@"  Bestᵗ⁺¹ (D → v) = $(curr_best_route.rem) hops $(curr_best_route.hops_with_gw.size)" +
                                         @" gid $(curr_best_route.gid)");
                    }
                }

                log_info("ETP executed.");
                etp_executed();
                advertise_modification_to_routes(etp.tpl, aggregated_neighbour.rem);
            }
            finally
            {
                log_debug("request_updates: etp processed.");
                // Now we are informed from this neighbour up to this sequence number.
                aggregated_neighbour.mod_seq_num = etp.seq_num;
            }
        }

        public bool is_etp_executing()
        {
            return etp_exec_dispatcher_channel != null;
        }

        private void impl_etp_new_link(AggregatedNeighbour aggregated_neighbour) throws Error
        {
            Tasklet.declare_self("Etp.etp_new_link");
            // Asks for paths via this new link.
            aggregated_neighbour.mod_seq_num = 0;
            log_debug("calling new_sequence_number locally to request info to a new neighbour");
            new_sequence_number(aggregated_neighbour.nip, aggregated_neighbour.nodeid, 1);
        }

        private static void * helper_etp_new_link(void *v) throws Error
        {
            struct_helper_Etp_etp_new_link *tuple_p = (struct_helper_Etp_etp_new_link *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            AggregatedNeighbour aggregated_neighbour_save = tuple_p->aggregated_neighbour;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_etp_new_link(aggregated_neighbour_save);
            // void method, return null
            return null;
        }

        /** Detecting a variation (new link): advertise this modification.
          */
        public void etp_new_link(AggregatedNeighbour aggregated_neighbour)
        {
            struct_helper_Etp_etp_new_link arg = struct_helper_Etp_etp_new_link();
            arg.self = this;
            arg.aggregated_neighbour = aggregated_neighbour;
            Tasklet.spawn((FunctionDelegate)helper_etp_new_link, &arg);
        }

        /** Detecting a variation (changed link): possible hook before really doing the variation.
          */
        public void etp_before_changed_link(AggregatedNeighbour aggregated_neighbour)
        {
        }

        private void impl_etp_changed_link(AggregatedNeighbour aggregated_neighbour, REM oldrem) throws Error
        {
            Tasklet.declare_self("Etp.etp_changed_link");
            // step 1 of CLR
            // A computes rᵗ⁺¹A (→ B) = Rᵗ⁺¹ (AB)
            // It is already in aggregated_neighbour.rem

            // Now, ∀v ∈ V
            for (int lvl = 0; lvl < maproute.levels; lvl++)
            {
                for (int dst = 0; dst < maproute.gsize; dst++)
                {
                    RouteNode routes_to_v = maproute.node_get(lvl, dst);
                    // step 2 of CLR
                    // A computes Bestᵗ⁺¹ (A → v)
                    // Since aggregated_neighbour.rem is updated, each RouteNode.best_route is automatically up to date.
                    // We only need to emit signal that the routes are updated.
                    if (! routes_to_v.is_empty())
                        if (routes_to_v.route_get_by_gw(aggregated_neighbour) != null)
                            maproute.route_signal_rem_changed(lvl, dst);
                }
                advertise_modification_to_routes();
            }
        }

        private static void * helper_etp_changed_link(void *v) throws Error
        {
            struct_helper_Etp_etp_changed_link *tuple_p = (struct_helper_Etp_etp_changed_link *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            AggregatedNeighbour aggregated_neighbour_save = tuple_p->aggregated_neighbour;
            REM oldrem_save = tuple_p->oldrem;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_etp_changed_link(aggregated_neighbour_save, oldrem_save);
            // void method, return null
            return null;
        }

        /** Detecting a variation (changed link): advertise this modification.
          */
        public void etp_changed_link(AggregatedNeighbour aggregated_neighbour, REM oldrem)
        {
            struct_helper_Etp_etp_changed_link arg = struct_helper_Etp_etp_changed_link();
            arg.self = this;
            arg.aggregated_neighbour = aggregated_neighbour;
            arg.oldrem = oldrem;
            Tasklet.spawn((FunctionDelegate)helper_etp_changed_link, &arg);
        }

        /** Detecting a variation (dead link): possible hook before really doing the variation.
          */
        public void etp_before_dead_link(AggregatedNeighbour aggregated_neighbour)
        {
        }

        private void impl_etp_dead_link(AggregatedNeighbour aggregated_neighbour) throws Error
        {
            Tasklet.declare_self("Etp.etp_dead_link");
            // step 1 of CLR (similar to a dead link case)
            // A computes rᵗ⁺¹A (→ B) = Rᵗ⁺¹ (AB)   It is now a DeadRem.
            log_debug(@"Will remove routes per gateway $aggregated_neighbour");
            maproute.delete_routes_via_neighbour(aggregated_neighbour);
            // We omit step 2 of CLR: ∀v ∈ V, A computes Bestᵗ⁺¹ (A → v)
            // Since old routes (paths) have been deleted, each RouteNode.best_route is automatically up to date.
            log_debug(@"Done removing routes per gateway $aggregated_neighbour");
            advertise_modification_to_routes();
        }

        private static void * helper_etp_dead_link(void *v) throws Error
        {
            struct_helper_Etp_etp_dead_link *tuple_p = (struct_helper_Etp_etp_dead_link *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            AggregatedNeighbour aggregated_neighbour_save = tuple_p->aggregated_neighbour;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_etp_dead_link(aggregated_neighbour_save);
            // void method, return null
            return null;
        }

        /** Detecting a variation (dead link): advertise this modification.
          */
        public void etp_dead_link(AggregatedNeighbour aggregated_neighbour)
        {
            struct_helper_Etp_etp_dead_link arg = struct_helper_Etp_etp_dead_link();
            arg.self = this;
            arg.aggregated_neighbour = aggregated_neighbour;
            Tasklet.spawn((FunctionDelegate)helper_etp_dead_link, &arg);
        }

        private void impl_collision_rehook(NetworkID neighbour_netid) throws Error
        {
            Tasklet.declare_self("Etp.collision_rehook");
            log_debug(@"Etp: collision detected: rehook to netid $(neighbour_netid)");
            Gee.List<AggregatedNeighbour> the_others = aggregated_neighbour_manager.neighbour_list(null, neighbour_netid);
            // If we want to rehook with the_others, we have to make sure 
            // there is someone! We have to wait. And we could fail anyway.
            // In that case abort the processing of this request.
            TimeCapsule tc = new TimeCapsule(16000);
            while (the_others.is_empty)
            {
                if (tc.is_expired())
                {
                    log_info(@"Rehooking to netid = $neighbour_netid aborted: timeout.");
                    return;
                }
                ms_wait(50);
                the_others = aggregated_neighbour_manager.neighbour_list(null, neighbour_netid);
            }
            net_collision(the_others);
        }

        private static void * helper_collision_rehook(void *v) throws Error
        {
            struct_helper_Etp_collision_rehook *tuple_p = (struct_helper_Etp_collision_rehook *)v;
            // The caller function has to add a reference to the ref-counted instances
            Etp self_save = tuple_p->self;
            NetworkID neighbour_netid_save = tuple_p->neighbour_netid;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_collision_rehook(neighbour_netid_save);
            // void method, return null
            return null;
        }

        /** We are colliding and we are going to rehook.
          */
        public void collision_rehook(NetworkID neighbour_netid)
        {
            struct_helper_Etp_collision_rehook arg = struct_helper_Etp_collision_rehook();
            arg.self = this;
            arg.neighbour_netid = neighbour_netid;
            Tasklet.spawn((FunctionDelegate)helper_collision_rehook, &arg);
        }
    }
}

