/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    struct struct_helper_Map_periodically_check_all_nodes
    {
        public Map self;
    }

    struct struct_helper_Map_check_node
    {
        public Map self;
        public int lvl;
        public int pos;
    }

    /** A Map instance represents ...
      */
    public class Map<T> : Object
    {
        public signal void node_new(int lvl, int pos);
        public signal void node_deleted(int lvl, int pos);

        private int _levels;
        private int _gsize;
        private NIP _me;
        protected DataClass[,] node;
        private bool[,] node_prev_free;
        private Tasklet? check_all_nodes_handle = null;
        private bool stopped_operations = false;

        public int levels {
            get {
                return _levels;
            }
        }

        public int gsize {
            get {
                return _gsize;
            }
        }

        public NIP me {
            get {
                return _me;
            }
        }

        public Map(int levels, int gsize, NIP me)
        {
            this._levels = levels;    // Number of levels
            this._gsize = gsize;      // How many nodes are contained in a gnode
            assert(typeof(T).is_a(typeof(DataClass)));
            {
                int [] y = me.get_positions();
                // if I don't put this in a local variable strange things happen (?)
                assert(y.length == levels);
            }
            this._me = me;            // me.positions[lvl] is the ID of our (g)node of level lvl.

            // The member node[l, i] is a node of level l and its ID is i
            this.node = new DataClass[this._levels, this._gsize];
            this.node_prev_free = new bool[this._levels, this._gsize];
            for (int i = 0; i < this._levels; i++)
            {
                for (int j = 0; j < this._gsize; j++)
                {
                    this.node[i, j] = null;
                    this.node_prev_free[i, j] = true;
                }
            }

            this.check_all_nodes_handle = null;
            periodically_check_all_nodes();
        }

        private void impl_periodically_check_all_nodes() throws Error
        {
            Tasklet.declare_self("Map.periodically_check_all_nodes");
            if (stopped_operations) return;
            check_all_nodes_handle = Tasklet.self();
            log_debug(@"Map($(this.get_type().name())): periodically_check_all_nodes started.");
            while (true)
            {
                try
                {
                    Tasklet.nap(5, 0);
                    for (int lvl = 0; lvl < this._levels; lvl++)
                    {
                        for (int pos = 0; pos < this._gsize; pos++)
                        {
                            check_node(lvl, pos);
                        }
                    }
                }
                catch
                {
                    log_error(@"Map($(this.get_type().name())): Uncaught exception while periodically_check_all_nodes");
                }
            }
        }

        /* Decoration of microfunc */
        private static void * helper_periodically_check_all_nodes(void *v) throws Error
        {
            struct_helper_Map_periodically_check_all_nodes *tuple_p = (struct_helper_Map_periodically_check_all_nodes *)v;
            // The caller function has to add a reference to the ref-counted instances
            Map self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_periodically_check_all_nodes();
            // void method, return null
            return null;
        }

        public void periodically_check_all_nodes()
        {
            struct_helper_Map_periodically_check_all_nodes arg = struct_helper_Map_periodically_check_all_nodes();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_periodically_check_all_nodes, &arg);
        }

        public virtual void stop_operations()
        {
            string ipstr = nip_to_str(levels, gsize, me);
            log_debug(@"Map: [$(get_type().name())] stopping operations for $(ipstr)");
            if (check_all_nodes_handle != null)
            {
                log_debug(@"Map: aborting check_all_nodes for $(ipstr)");
                check_all_nodes_handle.abort();
                check_all_nodes_handle = null;
            }
            stopped_operations = true;
        }

        private void impl_check_node(int lvl, int pos) throws Error
        {
            if (stopped_operations) return;
            bool prev = node_prev_free[lvl, pos];
            bool curr = (node_get(lvl, pos) as DataClass).is_free();
            if (prev && !curr) node_add(lvl, pos);
            if (curr && !prev) node_del(lvl, pos);
            node_prev_free[lvl, pos] = curr;
        }

        /* Decoration of microfunc with dispatcher */
        private static void * helper_check_node(void *v) throws Error
        {
            Tasklet.declare_self("Map.check_node dispatcher");
            struct_channel *ch_cont_p = (struct_channel *)v;
            // The caller function has to add a reference to the ref-counted instances
            Channel ch = ch_cont_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
            Tasklet.schedule_back();
            // The actual dispatcher
            while (true)
            {
                string? doing = null;
                try
                {
                    struct_helper_Map_check_node tuple_p;
                    {
                        Value vv = ch.recv();
                        tuple_p = *((struct_helper_Map_check_node *)(vv.get_boxed()));
                    }
                    doing = @"check_node($(tuple_p.lvl), $(tuple_p.pos))";
                    Tasklet.declare_self(doing);
                    // The helper function should not need to copy values
                    tuple_p.self.impl_check_node(tuple_p.lvl, tuple_p.pos);
                }
                catch (Error e)
                {
                    log_warn(@"Map: check_node reported an error: $(e.message)");
                }
                if (doing != null) Tasklet.declare_finished(doing);
            }
        }

        public void check_node(int lvl, int pos)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((FunctionDelegate)helper_check_node);

            struct_helper_Map_check_node arg = struct_helper_Map_check_node();
            arg.self = this;
            arg.lvl = lvl;
            arg.pos = pos;

            // send the struct
            ch.send_async(arg);
        }

        /** Returns from the map a node of level `lvl' and id `pos'.
          *
          * An instance of type T will always be returned: if
          * it doesn't exist, it is created.
          */
        public virtual T node_get(int lvl, int pos)
        {
            if (node[lvl, pos] == null)
            {
                if (_me.position_at(lvl) == pos)
                {
                    DataClass ret = Object.new(typeof(T)) as DataClass;
                    ret.initialize(this, lvl, pos, true);
                    node[lvl, pos] = ret;
                }
                else
                {
                    DataClass ret = Object.new(typeof(T)) as DataClass;
                    ret.initialize(this, lvl, pos);
                    node[lvl, pos] = ret;
                }
            }
            return (T)node[lvl, pos];
        }
        public T node_get_hcoord(HCoord hc)
        {
            return node_get(hc.lvl, hc.pos);
        }

        /** Removes from the map a node of level `lvl' and id `pos'.
          */
        public void node_remove(int lvl, int pos)
        {
            node[lvl, pos] = null;
        }

        /** A node was previously free, and now it is busy. Emit a signal.
          */
        public void node_add(int lvl, int pos)
        {
            log_debug(@"Map($(this.get_type().name())): emit signal NODE_NEW($(lvl), $(pos)).");
            node_new(lvl, pos);
        }

        /** A node was previously busy, and now it is free. Emit a signal.
          */
        public void node_del(int lvl, int pos)
        {
            log_debug(@"Map($(this.get_type().name())): emit signal NODE_DELETED($(lvl), $(pos)).");
            node_deleted(lvl, pos);
        }

        /** Returns the number of free nodes of level `lvl'
          */
        public int free_nodes_nb(int lvl)
        {
            int [] x = free_nodes_list(lvl);
            return x.length;
        }

        /** Returns the list of free nodes of level `lvl'
          */
        public int[] free_nodes_list(int lvl)
        {
            ArrayList<int> ret = new ArrayList<int>();
            // valid_ids depends on the lvl and on the previous ids
            foreach (int pos in valid_ids(_levels, _gsize, lvl, _me))
            {
                if ((node_get(lvl, pos) as DataClass).is_free()) ret.add(pos);
            }
            return (int[])ret.to_array();
        }

        /** Returns the number of busy nodes of level `lvl'
          */
        public int busy_nodes_nb(int lvl)
        {
            int [] x = busy_nodes_list(lvl);
            return x.length;
        }

        /** Returns the list of busy nodes of level `lvl'
          */
        public int[] busy_nodes_list(int lvl)
        {
            ArrayList<int> ret = new ArrayList<int>();
            // valid_ids depends on the lvl and on the previous ids
            foreach (int pos in valid_ids(_levels, _gsize, lvl, _me))
            {
                if (! (node_get(lvl, pos) as DataClass).is_free()) ret.add(pos);
            }
            int[] retval = new int[ret.size];
            for (int i = 0; i < ret.size; i++) retval[i] = ret[i];
            return retval;
        }

        /** Returns a list of the number of busy nodes at every level
          */
        public int[] get_all_busy_nodes_nb()
        {
            int[] retval = new int[_levels];
            for (int lvl = 0; lvl < _levels; lvl++) retval[lvl] = busy_nodes_nb(lvl);
            return retval;
        }

        public int nip_cmp(int[] nipA)
        {
            return NIP.nip_cmp(nipA, _me.get_positions());
        }

        /** Converts a (lvl, pos) pair, referring to this map, to
          * its equivalent nip
          */
        public PartialNIP lvlid_to_nip(HCoord lvlid)
        {
            return lvlid.get_partialnip_relative_to(_me);
        }

        /** Finds a (lvl, pos) pair, referring to this map, from
          * its equivalent nip
          */
        public HCoord nip_to_lvlid(PartialNIP nip)
        {
            return nip.get_hcoord_relative_to(_me);
        }

        /** Given a list of pairs (lvl, id) received by a neighbour with a
          * given nip, returns the list of pairs (lvl, id) representing the same
          * list of nodes, as seen from the point of view of this node.
          */
        public Gee.List<HCoord> list_lvl_id_from_nip(Gee.List<HCoord> lvl_ids, NIP sender_nip)
        {
            HCoord sender_lvlid = nip_to_lvlid(sender_nip);
            ArrayList<HCoord> ret = new ArrayList<HCoord>(HCoord.equal_func);
            foreach (HCoord lvl_id in lvl_ids)
            {
                if (lvl_id.lvl < sender_lvlid.lvl)
                    lvl_id = sender_lvlid;
                ret.add(lvl_id);
            }
            // remove dups
            ArrayList<HCoord> ret_nodups = new ArrayList<HCoord>(HCoord.equal_func);
            HCoord? prec = null;
            foreach (HCoord x in ret)
            {
                if (!x.is_equal(prec))
                {
                    prec = x;
                    ret_nodups.add(x);
                }
            }
            return ret_nodups;
        }

        /** Given a list of pairs (lvl, id) being sent to a neighbour with a
          * given nip, returns the list of pairs (lvl, id) representing the same
          * list of nodes, as seen from the point of view of the neighbour.
          */
        public Gee.List<HCoord> list_lvl_id_to_nip(Gee.List<HCoord> lvl_ids, NIP to_nip)
        {
            HCoord to_lvlid = nip_to_lvlid(to_nip);
            ArrayList<HCoord> ret = new ArrayList<HCoord>(HCoord.equal_func);
            foreach (HCoord lvl_id in lvl_ids)
            {
                if (lvl_id.lvl < to_lvlid.lvl)
                    lvl_id = new HCoord(to_lvlid.lvl, _me.position_at(to_lvlid.lvl));
                ret.add(lvl_id);
            }
            // remove dups
            ArrayList<HCoord> ret_nodups = new ArrayList<HCoord>(HCoord.equal_func);
            HCoord? prec = null;
            foreach (HCoord x in ret)
            {
                if (!x.is_equal(prec))
                {
                    prec = x;
                    ret_nodups.add(x);
                }
            }
            return ret_nodups;
        }

        /* Method repr_me was used only for a debug message in MapPeerToPeer, where it was however overridden. */
    }
}
