/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    public abstract class BaseTunnel : Object
    {
        private static CreateTunnelDelegate? registered_class_linux = null;
        public static void register_class(string k, CreateTunnelDelegate create_new_nic)
        {
            if (k == "linux" && registered_class_linux == null) registered_class_linux = create_new_nic;
        }

        public static BaseTunnel create_instance()
        {
            string impl = Settings.NETWORK_IMPLEMENTATION;
            if (impl == "linux" && registered_class_linux != null) return registered_class_linux();
            error(@"No valid real implementation of class BaseTunnel for $impl. Is your system supported?");
        }

        private static BaseTunnel? _instance = null;
        public static BaseTunnel get_instance()
        {
            if (_instance == null) _instance = create_instance();
            return _instance;
        }

        protected BaseTunnel()
        {
        }

        /** Requests a tunnel.
          *  Usually, the other end (that we call the server) is executing in the meantime
          *  the method 'serve'.
          *  The implementor can use the callback functions cb_send(mesg) and cb_recv() to
          *  communicate with the other end during the phases of establishing the
          *  tunnel.
          *  When the method exits, if all goes well, there must exist a new NIC that
          *  represents the working tunnel.
          *  The return-value is the name of the new NIC.
          */
        public abstract string request(CallbackSendDelegate cb_send, CallbackRecvDelegate cb_recv, string my_address, string dest_address) throws TunnelError, RPCError;

        /** Serves a tunnel.
          *  Usually, the other end (that we call the client) is executing in the meantime
          *  the method 'request'.
          *  The implementor can use the callback functions cb_send(mesg) and cb_recv() to
          *  communicate with the other end during the phases of establishing the
          *  tunnel.
          *  When the method exits, if all goes well, there must exist a new NIC that
          *  represents the working tunnel.
          *  The return-value is the name of the new NIC.
          */
        public abstract string serve(CallbackSendDelegate cb_send, CallbackRecvDelegate cb_recv, string my_address, string dest_address) throws TunnelError, RPCError;

        /** Closes a tunnel.
          */
        public abstract void close(string nic_name, string my_address, string dest_address) throws TunnelError;
    }

    public delegate BaseTunnel CreateTunnelDelegate();
    public delegate void CallbackSendDelegate(ISerializable msg) throws TunnelError, RPCError;
    public delegate ISerializable CallbackRecvDelegate() throws TunnelError, RPCError;

    public class DummyTunnel : BaseTunnel
    {
        public DummyTunnel()
        {
            base();
        }

        public override string serve(CallbackSendDelegate cb_send, CallbackRecvDelegate cb_recv, string my_address, string dest_address) {return "dummy";}
        public override string request(CallbackSendDelegate cb_send, CallbackRecvDelegate cb_recv, string my_address, string dest_address) {return "dummy";}
        public override void close(string nic_name, string my_address, string dest_address) {}
    }

    struct struct_helper_TunnelManager_initiate
    {
        public TunnelManager self;
        public string dest_addr;
        public string my_addr;
        public int peer_handler_id;
        public int my_handler_id;
    }

    public class TunnelInfo : Object
    {
        public string nic_name {get; private set;}
        public string my_addr {get; private set;}
        public string dest_addr {get; private set;}
        public TunnelInfo(string nic_name, string my_addr, string dest_addr)
        {
            this.nic_name = nic_name;
            this.my_addr = my_addr;
            this.dest_addr = dest_addr;
        }
        public static bool equal_func(TunnelInfo? a, TunnelInfo? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.nic_name != b.nic_name) return false;
            if (a.my_addr != b.my_addr) return false;
            if (a.dest_addr != b.dest_addr) return false;
            return true;
        }
        public bool equals(TunnelInfo? other)
        {
            return TunnelInfo.equal_func(this, other);
        }
    }

    public class TunnelManager : Object, ITunnelManager
    {
        public string my_addr {get; private set;}
        private HashMap<int, Channel> handlers;
        private HashMap<string, TunnelInfo> active_tunnels; // key is nic_name

        public TunnelManager(string my_addr)
        {
            this.my_addr = my_addr;
            handlers = new HashMap<int, Channel>();
            active_tunnels = new HashMap<string, TunnelInfo>();
        }

        public Collection<TunnelInfo> get_active_tunnels()
        {
            return active_tunnels.values;
        }

        public string? choose_tunnel_protocol(Gee.List<string> protocols)
        {
            if (protocols.contains("tinc")) return "tinc";
            return null;
        }

        /** Remotable method. Called to instruct a node to serve a tunnel.
          */
        public int request_tunnel(string protocol, int peer_handler_id, CallerInfo? _rpc_caller = null) throws TunnelError
        {
            assert(_rpc_caller != null);
            assert(_rpc_caller.get_type().is_a(typeof(CallerInfo)));
            CallerInfo rpc_caller = (CallerInfo)_rpc_caller;

            if (protocol != "tinc") throw new TunnelError.GENERIC("Invalid protocol.");
            string my_addr = rpc_caller.my_ip;
            string dest_addr = rpc_caller.caller_ip;
            int my_handler_id = Random.int_range(0, (int)Math.pow(2,32) - 1);
            Channel handler_channel = new Channel();
            handlers[my_handler_id] = handler_channel;
            initiate(dest_addr, my_addr, peer_handler_id, my_handler_id);
            log_debug(@"TunnelManager: serving a request: $(my_addr) to $(dest_addr) handshaking in progress.");
            return my_handler_id;
        }

        /** Executed on a Tasklet to serve a tunnel.
          */
        private void impl_initiate(string dest_addr, string my_addr, int peer_handler_id, int my_handler_id) throws Error
        {
            Tasklet.declare_self("TunnelManager.initiate");
            AddressManagerTCPClient peer_client = new AddressManagerTCPClient(dest_addr, null, my_addr, false);
            CallbackSendDelegate cb_send = (mesg) => {
                peer_client.tunnel_manager.handshake(mesg, peer_handler_id);
            };
            CallbackRecvDelegate cb_recv = () => {
                return (ISerializable)handlers[my_handler_id].recv();
            };
            BaseTunnel ktunnel = BaseTunnel.get_instance();
            string nic_name = ktunnel.serve(cb_send, cb_recv, my_addr, dest_addr);
            log_debug(@"TunnelManager: $(my_addr) to $(dest_addr) tunnel nic is $(nic_name)");
            assert(! active_tunnels.has_key(nic_name));
            active_tunnels[nic_name] = new TunnelInfo(nic_name, my_addr, dest_addr);
        }

        private static void * helper_initiate(void *v) throws Error
        {
            struct_helper_TunnelManager_initiate *tuple_p = (struct_helper_TunnelManager_initiate *)v;
            // The caller function has to add a reference to the ref-counted instances
            TunnelManager self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            string dest_addr_save = tuple_p->dest_addr;
            string my_addr_save = tuple_p->my_addr;
            int peer_handler_id_save = tuple_p->peer_handler_id;
            int my_handler_id_save = tuple_p->my_handler_id;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_initiate(dest_addr_save, my_addr_save, peer_handler_id_save, my_handler_id_save);
            // void method, return null
            return null;
        }

        public void initiate(string dest_addr, string my_addr, int peer_handler_id, int my_handler_id)
        {
            struct_helper_TunnelManager_initiate arg = struct_helper_TunnelManager_initiate();
            arg.self = this;
            arg.dest_addr = dest_addr;
            arg.my_addr = my_addr;
            arg.peer_handler_id = peer_handler_id;
            arg.my_handler_id = my_handler_id;
            Tasklet.spawn((FunctionDelegate)helper_initiate, &arg);
        }

        /** Remotable method. Called to pass a message during the handshake phase,
          * following a particular protocol (e.g. tinc on linux).
          */
        public void handshake(ISerializable mesg, int handler_id)
        {
            if (handlers.has_key(handler_id))
                handlers[handler_id].send(mesg);
        }

        /** Remotable method. Called to instruct a node to close a tunnel.
          */
        public void close_tunnel(string nic_name, CallerInfo? _rpc_caller = null)
        {
            assert(_rpc_caller != null);
            assert(_rpc_caller.get_type().is_a(typeof(CallerInfo)));
            CallerInfo rpc_caller = (CallerInfo)_rpc_caller;

            string my_addr = rpc_caller.my_ip;
            string dest_addr = rpc_caller.caller_ip;
            log_debug(@"TunnelManager: $(my_addr): a request has been received from $(dest_addr) to close $(nic_name)");
            close_my_tunnel(nic_name, my_addr, dest_addr);
        }

        private void close_my_tunnel(string nic_name, string my_address, string dest_address)
        {
            if (active_tunnels.has_key(nic_name) &&
                active_tunnels[nic_name]
                    .equals(new TunnelInfo(nic_name, my_address, dest_address))
                )
            {
                BaseTunnel ktunnel = BaseTunnel.get_instance();
                try
                {
                    ktunnel.close(nic_name, my_address, dest_address);
                    log_debug(@"TunnelManager: $(my_address) to $(dest_address): $(nic_name) has been correctly closed.");
                }
                catch (TunnelError e)
                {
                    log_debug(@"TunnelManager: $(my_address) to $(dest_address): $(nic_name)" +
                            @" has encountered problem during close: $(e.message)");
                    log_debug("TunnelManager: Ignoring problem.");
                }
                active_tunnels.unset(nic_name);
            }
        }

        /** Helper client-side **/

        public string call_request_tunnel(string dest_addr) throws TunnelError, RPCError
        {
            log_debug(@"TunnelManager: $(my_addr): request a tunnel to $(dest_addr)");
            string nic_name;
            AddressManagerTCPClient peer_server = new AddressManagerTCPClient(dest_addr, null, my_addr);
            ArrayList<string> protocols = new ArrayList<string>();
            protocols.add("tinc");
            string proto = peer_server.tunnel_manager.choose_tunnel_protocol(protocols);
            if (proto == "tinc")
            {
                log_debug(@"TunnelManager: $(my_addr): $(dest_addr) allows tinc protocol.");
                int my_handler_id = Random.int_range(0, (int)Math.pow(2,32) - 1);
                Channel handler_channel = new Channel();
                handlers[my_handler_id] = handler_channel;
                int peer_handler_id = peer_server.tunnel_manager.request_tunnel(proto, my_handler_id);
                log_debug(@"TunnelManager: $(my_addr) to $(dest_addr) handshaking in progress.'");
                CallbackSendDelegate cb_send = (mesg) => {
                    peer_server.tunnel_manager.handshake(mesg, peer_handler_id);
                };
                CallbackRecvDelegate cb_recv = () => {
                    return (ISerializable)handlers[my_handler_id].recv();
                };
                BaseTunnel ktunnel = BaseTunnel.get_instance();
                nic_name = ktunnel.request(cb_send, cb_recv, my_addr, dest_addr);
            }
            else
            {
                log_debug(@"TunnelManager: invalid protocol '$(proto)'.");
                throw new TunnelError.GENERIC("Invalid protocol");
            }
            log_debug(@"TunnelManager: $(my_addr) to $(dest_addr) tunnel nic is $(nic_name)");
            active_tunnels[nic_name] = new TunnelInfo(nic_name, my_addr, dest_addr);
            return nic_name;
        }

        public void call_close_tunnel(string dest_addr, string nic_name)
        {
            // First, see if it's still there
            if (active_tunnels.has_key(nic_name) &&
                active_tunnels[nic_name]
                    .equals(new TunnelInfo(nic_name, my_addr, dest_addr))
                )
            {
                log_debug(@"TunnelManager: $(my_addr) closes tunnel $(nic_name): try to communicate to $(dest_addr)");
                bool done = false;
                try
                {
                    AddressManagerTCPClient peer_server = new AddressManagerTCPClient(dest_addr, null, my_addr, false);
                    peer_server.tunnel_manager.close_tunnel(nic_name);
                    log_debug(@"TunnelManager: $(my_addr) communication ok to $(dest_addr)");
                    done = true;
                }
                catch
                {
                    log_debug(@"TunnelManager: $(my_addr) communication fail to $(dest_addr)");
                }
                // close my side
                close_my_tunnel(nic_name, my_addr, dest_addr);
                if (! done)
                {
                    try
                    {
                        log_debug(@"TunnelManager: $(my_addr): try again to communicate to $(dest_addr)");
                        AddressManagerTCPClient peer_server = new AddressManagerTCPClient(dest_addr, null, my_addr, false);
                        peer_server.tunnel_manager.close_tunnel(nic_name);
                        log_debug(@"TunnelManager: $(my_addr) communication ok to $(dest_addr)");
                    }
                    catch
                    {
                        log_debug(@"TunnelManager: $(my_addr) communication fail to $(dest_addr)");
                    }
                }
            }
        }
    }
}

