/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    /** A path to a known destination.
      *
      * An instance of this class represents a path to a known
      * destination d.
      * The instance members are:
      * `gw': the gateway of the route. Our next hop. It is an instance of the
      *       class AggregatedNeighbour.
      * `rem_at_gw': a REM (Route Efficience Measure) of the path from the 
      *       gateway to the destination d. It is an instance of the class 
      *       REM or a derivative.
      * `hops': a List of HCoord of the hops represented by this
      *       path from the gateway gw to the destination d.
      *       gw and d are not included.
      * `gid': a instance of GNodeid, it is the ID of the gnode that this class
      *       represents, as it is reported by this path.
      */
    public class Route : Object
    {
        public weak RouteNode routenode;
        public AggregatedNeighbour gw;
        public REM rem_at_gw;
        public Gee.List<HCoord> hops;
        public GNodeID gid;

        public Route(AggregatedNeighbour gw, REM rem_at_gw, Gee.List<HCoord> hops, GNodeID gid, RouteNode routenode)
        {
            this.gw = gw;
            this.rem_at_gw = rem_at_gw;
            this.hops = hops;
            this.gid = gid;
            this.routenode = routenode;
        }

        private REM _rem;
        public REM rem {
            get {
                _rem = rem_at_gw.add_segment(gw.rem);
                return _rem;
            }
        }

        private HCoord _first_hop;
        public HCoord first_hop {
            get {
                _first_hop = routenode.maproute.nip_to_lvlid(gw.nip);
                return _first_hop;
            }
        }

        private HCoord _dest;
        public HCoord dest {
            get {
                _dest = new HCoord(routenode.lvl, routenode.pos);
                return _dest;
            }
        }

        private Gee.List<HCoord> _hops_with_gw;
        // this contains gateway, but not destination in any case.
        public Gee.List<HCoord> hops_with_gw {
            get {
                _hops_with_gw = new ArrayList<HCoord>(HCoord.equal_func);
                if (! first_hop.is_equal(dest))
                {
                    _hops_with_gw.add(first_hop);
                    _hops_with_gw.add_all(hops);
                }
                return _hops_with_gw;
            }
        }

        /** The route 'this' is better (greater) than 'other' if its rem is better
          */
        public static int compare_func(Route a, Route b)
        {
            return a.compare_to(b);
        }
        private int compare_to(Route other)
        {
            return rem.compare_to(other.rem);
        }

        /** Used to test between 2 routes in the same RouteNode
          * (e.g. to remove a route from the collection of routes)
          * In such a comparison the only member we can check is the gateway.
          */
        public static bool equal_func_in_routenode(Route a, Route b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            return a.gw == b.gw;
        }

        /** The path contains a specified hop?
          */
        public bool contains(HCoord hop)
        {
            if (hop.is_equal(dest)) return true;
            if (hop.is_equal(first_hop)) return true;
            if (hops.contains(hop)) return true;
            return false;
        }

        public REM rem_modify(REM newrem_at_gw, Gee.List<HCoord> new_hops, GNodeID new_gid)
        {
            REM oldrem_at_gw = rem_at_gw;
            rem_at_gw = newrem_at_gw;
            hops = new_hops;
            gid = new_gid;
            return oldrem_at_gw;
        }

        /** Returns an instance of CloneRoute that is a copy of the current
          * status of this Route. If this Route changes in some way, the cloned
          * instance will not.
          * The class CloneRoute has some of the methods that Route has,
          * those needed to perform the task of choosing the settings to
          * impose in the kernel routing tables.
          */
        public CloneRoute clone()
        {
            return new CloneRoute(this);
        }
    }

    public class CloneRoute : Object
    {
        public string gwdev;
        public REM gwrem;
        public NIP gwnip;
        public string gwipstr;
        public REM rem;
        public REM rem_at_gw;
        public HCoord dest;
        public HCoord first_hop;
        public Gee.List<HCoord> hops;
        public Gee.List<HCoord> hops_with_gw;
        public NIP mymap_me;

        public CloneRoute(Route route)
        {
            try
            {
                gwdev = route.gw.dev;
                gwrem = route.gw.rem;
                gwipstr = nip_to_str(route.gw.levels, route.gw.gsize, route.gw.nip);

                gwnip = (NIP)ISerializable.deserialize(route.gw.nip.serialize());
                rem = (REM)ISerializable.deserialize(route.rem.serialize());
                rem_at_gw = (REM)ISerializable.deserialize(route.rem_at_gw.serialize());
                dest = (HCoord)ISerializable.deserialize(route.dest.serialize());
                first_hop = (HCoord)ISerializable.deserialize(route.first_hop.serialize());

                hops = new ArrayList<HCoord>(HCoord.equal_func);
                foreach (HCoord hop in route.hops)
                {
                    HCoord hop2 = (HCoord)ISerializable.deserialize(hop.serialize());
                    hops.add(hop2);
                }
                hops_with_gw = new ArrayList<HCoord>(HCoord.equal_func);
                foreach (HCoord hop in route.hops_with_gw)
                {
                    HCoord hop2 = (HCoord)ISerializable.deserialize(hop.serialize());
                    hops_with_gw.add(hop2);
                }

                mymap_me = route.routenode.maproute.me;
            }
            catch (SerializerError e)
            {
                error(@"CloneRoute: Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }

        public static bool equal_func(CloneRoute? a, CloneRoute? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (@"$(a.dest)" != @"$(b.dest)") return false;
            if (a.gwipstr != b.gwipstr) return false;
            if (a.gwdev != b.gwdev) return false;
            string a_list = "";
            foreach (HCoord hop in a.hops) a_list += @"$(hop)";
            string b_list = "";
            foreach (HCoord hop in b.hops) b_list += @"$(hop)";
            if (a_list != b_list) return false;
            return true;
        }

        public static uint hash_func(CloneRoute a)
        {
            return @"$(a.dest)_$(a.gwipstr)_$(a.gwdev)".hash();
        }

        public string to_string()
        {
            return @"<CloneRoute to $dest via $gwnip dev $gwdev rem $gwrem>";
        }

        /** The route 'this' is better (greater) than 'other' if its rem is better
          */
        public static int compare_func(CloneRoute a, CloneRoute b)
        {
            return a.compare_to(b);
        }
        public int compare_to(CloneRoute other)
        {
            return rem.compare_to(other.rem);
        }

        /** The path contains a specified hop?
          */
        public bool contains(HCoord hop)
        {
            if (hop.is_equal(dest)) return true;
            if (hop.is_equal(first_hop)) return true;
            if (hops.contains(hop)) return true;
            return false;
        }
        public bool contains_nip(NIP nip)
        {
            return contains(this.nip_to_lvlid(nip));
        }

        public HCoord nip_to_lvlid(NIP nip)
        {
            return nip.get_hcoord_relative_to(mymap_me);
        }
        public PartialNIP lvlid_to_nip(HCoord lvlid)
        {
            return lvlid.get_partialnip_relative_to(mymap_me);
        }
    }


    /** List of paths to a known destination.
      *
      * This class is basically a list of Route instances, where the
      * destination node and its level are fixed and known.
      *
      * Note: for each gateway G there's only one route in self.routes,
      *       which has the same gateway G
      */
    public class RouteNode : DataClass
    {
            public weak Map maproute;
            public int lvl;
            public int pos;
            public bool its_me;
            public bool busy;
            public ArrayList<Route> routes;

            public override void initialize(Object map, int lvl, int pos, bool its_me = false)
            {
                this.maproute = map as Map;
                this.lvl = lvl;
                this.pos = pos;
                this.its_me = its_me;
                busy = false;
                routes = new ArrayList<Route>(Route.equal_func_in_routenode);
            }

            /** Returns the route having as gateway `gw'
              */
            public Route? route_get_by_gw(AggregatedNeighbour gw)
            {
                foreach (Route r in routes)
                {
                    if (r.gw == gw)
                    {
                        return r;
                    }
                }
                return null;
            }

            public void update_route_by_gw(AggregatedNeighbour nr, REM rem_at_gw, Gee.List<HCoord> hops, GNodeID gid)
            {
                Route? r = route_get_by_gw(nr);
                if (r == null)
                {
                    routes.add(new Route(nr, rem_at_gw, hops, gid, this));
                }
                else
                {
                    r.rem_modify(rem_at_gw, hops, gid);
                }
                busy = true; // For sure now we are busy.
                sort();
            }

            /** Delete a route.
              * Returns 1 if the route has been deleted, otherwise 0
              */
            public int route_del_by_gw(AggregatedNeighbour gw)
            {
                Route? r = route_get_by_gw(gw);
                if (r != null)
                {
                    routes.remove(r);
                    return 1;
                }
                return 0;
            }

            private static int reverse_compare_func(Route a, Route b)
            {
                return - Route.compare_func(a, b);
            }
            /** Order the routes
              * Order the routes in decrescent order of efficiency, so that
              *  routes[0] is the best one
              */
            public void sort()
            {
                routes.sort(reverse_compare_func);
            }

            public bool is_empty()
            {
                return routes.is_empty;
            }

            public override bool is_free()
            {
                if (its_me) return false;
                return !busy;
            }

            public GNodeID? get_eldest_gid()
            {
                GNodeID? ret = null;
                foreach (Route r in routes)
                {
                    if (ret == null) ret = r.gid;
                    else ret = ret.will_bubble(r.gid);
                }
                return ret;
            }

            public Gee.List<Route> all_valid_routes()
            {
                // a route is valid if gid is the oldest
                ArrayList<Route> ret = new ArrayList<Route>(Route.equal_func_in_routenode);
                sort();
                if (!routes.is_empty)
                {
                    GNodeID eldest = get_eldest_gid();
                    foreach (Route r in routes)
                    {
                        if (r.gid.ident == eldest.ident)
                        {
                            ret.add(r);
                        }
                    }
                }
                return ret;
            }

            public int nroutes()
            {
                return all_valid_routes().size;
            }

            public Route? best_route()
            {
                Gee.List<Route> filtered = all_valid_routes();
                if (filtered.is_empty) return null;
                return filtered.get(0);
            }

            public Route? best_route_without(HCoord hop)
            {
                Gee.List<Route> filtered = all_valid_routes();
                foreach (Route r in filtered)
                {
                    if (! r.contains(hop)) return r;
                }
                return null;
            }
    }

    public class GIDChecker : Object
    {
        public int my_req_id;
        public TimeCapsule ttl;
        public ArrayList<int> received_req_ids;
        public GIDChecker()
        {
            my_req_id = Random.int_range(0, (int)Math.pow(2,32) - 1);
            ttl = new TimeCapsule(120000);
            received_req_ids = new ArrayList<int>();
        }
    }

    struct struct_helper_MapRoute_periodically_check_gid
    {
        public MapRoute self;
        public int level;
    }

    /** Map of routes, all of a same Rem type.
      *
      * MapRoute.node[lvl][id] is a RouteNode class, i.e. a list of routes
      * having as destination the node (lvl, id)
      */
    public class MapRoute : Map<RouteNode>, IMapRoute
    {
        public weak AddressManager address_manager {get; private set;}
        public GNodeID[] id_myself {get; private set;}
        private HashMap<int, GIDChecker> checking_gid;
        private ArrayList<Tasklet?> check_gid_handle;

        public MapRoute(int levels, int gsize, NIP me, GNodeID[] id_myself, AddressManager address_manager)
        {
            base(levels, gsize, me);
            this.id_myself = id_myself;
            this.address_manager = address_manager;
            
            checking_gid = new HashMap<int, GIDChecker>();
            check_gid_handle = new ArrayList<Tasklet?>();
            for (int i = 0; i < levels; i++) check_gid_handle.add(null);
        }

        public signal void routes_updated(HCoord lvl_pos);
        public signal void gnode_splitted(Gee.List<AggregatedNeighbour> passed_neighbour_list, Gee.List<int> queue_of_request_ids, GNodeID actual_gid);
        public signal void id_gnode_changed(int level);

        public override void stop_operations()
        {
            string ipstr = nip_to_str(levels, gsize, me);
            log_debug(@"MapRoute: stopping operations for $(ipstr)");
            log_debug(@"Coord: calling base");
            base.stop_operations();
            log_debug(@"Coord: base done");
            for (int l = 1; l < levels; l++)
            {
                Tasklet? t = check_gid_handle[l];
                if (t != null)
                {
                    log_debug(@"MapRoute: aborting check_gid level $(l) for $(ipstr)");
                    t.abort();
                    check_gid_handle[l] = null;
                }
            }
        }

        public Gee.List<InfoRoute> report_routes()
        {
            log_debug("report_routes start");
            ArrayList<InfoRoute> ret = new ArrayList<InfoRoute>();
            for (int level = levels-1; level >= 0; level--)
            {
                for (int dest = 0; dest < gsize; dest++)
                {
                    if (me.position_at(level) != dest)
                    {
                        RouteNode node = node_get(level, dest);
                        if (node.nroutes() > 0)
                        {
                            foreach (Route r in node.all_valid_routes())
                            {
                                if (! r.rem.get_type().is_a(typeof(DeadREM)))
                                {
                                    log_debug("report_routes: found one good route, adding...");
                                    ret.add(new InfoRoute(new HCoord(level, dest), r.gid, r.gw.nip, r.gw.dev, r.hops_with_gw, r.rem));
                                }
                            }
                        }
                    }
                }
                Tasklet.nap(0, 1000); // be gentle
            }
            log_debug(@"report_routes returns $(ret.size) routes");
            return ret;
        }

        public Gee.List<GNodeID> report_gid_list()
        {
            log_debug("report_gid_list start");
            ArrayList<GNodeID> ret = new ArrayList<GNodeID>();
            foreach (GNodeID g in get_gid_list())
            {
                ret.add(g);
            }
            log_debug(@"report_gid_list returns $(ret.size) IDs");
            return ret;
        }

        public InfoNode report_yourself()
        {
            return new InfoNode(get_main_netid(),
                         me,
                         address_manager.keypair.pub_key.to_pubkey(),
                         levels, gsize);
        }

        /** My GNode ID list
          */
        public GNodeID[] get_gid_list()
        {
            GNodeID[] ret = new GNodeID[id_myself.length];
            for (int i = 0; i < id_myself.length; i++)
            {
                ret[i] = id_myself[i];
            }
            return ret;
        }

        public GNodeID[] get_gid_uppermost_list(int level_of_gnode)
        {
            GNodeID[] ret = new GNodeID[id_myself.length - level_of_gnode];
            for (int i = level_of_gnode; i < id_myself.length; i++)
            {
                ret[i - level_of_gnode] = id_myself[i];
            }
            return ret;
        }

        public GNodeID get_gid_at_level(int lvl)
        {
            return id_myself[lvl];
        }

        /** My node ID
          */
        public int get_my_id()
        {
            return id_myself[0].ident;
        }

        /** This method returns a Networkid. The aim is to be able to
          *  testify to a neighbour that we already hooked in its same network.
          * Since the network could split, and we could detect the change
          *  moments before our neighbour does, then we must compare not just the
          *  GID of level 'levels' -- which is the min of GIDs of level 'levels-1'
          *  but we must compare all the GIDs of level levels-1.
          * Add to this, that at the very beginning of our hook when we did not
          *  receive an etp yet, we do not have in the maproute the necessary
          *  gids, we just received the main one in self.id_myself[-1]
          * Furthermore we have to consider self.id_myself[-2] for our GID at
          *  level 'levels-1'.
          * TODO We still have a weakness here. 2 neighbours that belong to the same
          *  network but in different gnodes at the uppermost level (eg 1.1.1 and 2.2.2
          *  in a network of 3 levels) have in common only the gnodes of level 2, so
          *  they can compare the GIDs at level 2 and the one GID at level 3 (that is
          *  the min of GIDs at level 2). If the gsize is quite small (say gsize=4) we
          *  have very few IDs (4) that we can compare. They exist for the lifespan of
          *  few single nodes. If all these nodes die quite at the same time,
          *  then we'll experience a massive rehook.
          */
        public NetworkID get_main_netid()
        {
            ArrayList<GNodeID> ret = new ArrayList<GNodeID>(GNodeID.equal_func);
            RouteNode[] r1 = {};
            for (int pos = 0; pos < gsize; pos++)
                r1 += node_get(levels-1, pos);
            foreach (RouteNode node in r1)
                if (! node.is_empty())
                    ret.add(node.get_eldest_gid());
            ret.add(id_myself[id_myself.length - 2]);
            ret.add(id_myself[id_myself.length - 1]);
            return new NetworkID(ret);
        }

        /** Is this netid in my network
          */
        public bool is_in_my_network(NetworkID netid)
        {
            return get_main_netid().is_same_network(netid);
        }

        /** Is this netid preferred over mine
          */
        public bool is_preferred_network(NetworkID netid)
        {
            return netid.is_preferred_over(get_main_netid());
        }

        /** Verify the validity of the GNode ID.
          * This has to be called just once after a ROUTES_UPDATED event,
          * to keep updated the GNodeid at various levels.
          */
        public void evaluate_changed_netid()
        {
            if (! address_manager.is_mature)
            {
                log_debug("Maproute for " + me.to_string() + ": DO NOT recalculate gnode IDs because it is too early.");
                return;
            }
            log_debug("Maproute for " + me.to_string() + ": recalculate gnode IDs.");
            for (int j = 0; j < levels; j++)
            {
                // new_ID_myself(j+1) = min({ID((j, pos)) | (j,pos) in maproute} + ID_myself(j))
                GNodeID eldest_internal_to_j_plus_1 = id_myself[j];
                for (int pos = 0; pos < gsize; pos++)
                {
                    RouteNode jpos = node_get(j, pos);
                    if (! jpos.is_empty())
                        eldest_internal_to_j_plus_1 = 
                                eldest_internal_to_j_plus_1
                                .will_bubble(jpos.get_eldest_gid());
                }
                if (id_myself[j+1].change_values(eldest_internal_to_j_plus_1))
                {
                    // Emit signal ID_GNODE_CHANGED
                    id_gnode_changed(j + 1);
                    // If the gnode which changed its ID is not the entire network...
                    if (j < levels-1)
                    {
                        // ... start (in a while) the procedure to check for a gnode split
                        log_debug("Maproute for " + me.to_string() + ": possible gnode split detected at level " + (j+1).to_string());
                        handle_check_gid(j+1);
                    }
                }
            }
        }

        /** A node which wants to rest assured that its GID at level lvl is valid will launch handle_check_gid. This
          *  will start operations (if not already in progress) and make sure that for a certain time
          *  from now the check will be periodically sent in broadcast until a response (ok or rehook)
          *  is received or the time expires.
          *  Note that handle_check_gid is not a microfunc and when it returns we can immediately add an id to its answer_queue.
          * When a node receives such a request and it is in the same gnode, it will launch handle_check_gid. Then it adds
          *  the received request_id to the answer_queue for the level.
          * To formulate the request to be sent, a node generates a request_id. It will use this request_id in all messages
          *  sent until response/expiration.
          * When a response is received that is for us and is ok, the same answer is sent again in broadcast to all the id
          *  that we have in answer_queue.
          * When a response is received that is for us and is rehook, the maproute emits a signal that will cause the
          *  ntkd_node to start a rehook. The signal contains all data needed to reply the same answer after the rehook has
          *  been completed. The reply will occurr only after the node has is_mature, because, until then, the node will not
          *  be able to serve a hook request.
          *  The exact sequence of operations when ntkd_node detects the signal is:
          *    . (assert the splitted was a primary_address)
          *    . start rehook from primary_address to [answering_nip]
          *    . tc=TimeCapsule(xx)
          *    . while True:
          *    .   if tc.expired(): break
          *    .   if rehook failed: break
          *    .   if addresses.primary_address.is_mature:
          *    .     for queued_id in queue:
          *    .       broadcast_client.maproute.answer_gid( \
          *    .              addresses.primary_address.maproute.me, \
          *    .              queued_id, \
          *    .              actual_gid)
          *    .     break
          *    .   swait a bit
          *  So, the data needed with the signal 'GNODE_SPLITTED' are: (neighbours=[answering_nip], queue, actual_gid)
          * Note_1: An address_manager nom-autonomous will ignore messages request_gid.
          * Note_2: An address_manager nom-autonomous that detects a possible gnode split will start handle_check_gid, but if
          *          it receives a rehook then it just dies. (this check is in ntkd_node)
          * Note_3: An address_manager that has not is_mature will ignore messages request_gid.
          */

        public void handle_check_gid(int level)
        {
            // If this same request is already pending, refresh the TTL and do nothing more.
            if (checking_gid.has_key(level))
            {
                int req_id = checking_gid[level].my_req_id;
                log_debug(@"Maproute for $me : refreshed a check_gid at level $level, my request_id is $req_id");
                checking_gid[level].ttl = new TimeCapsule(120000);
                return;
            }
            checking_gid[level] = new GIDChecker();
            periodically_check_gid(level);
            int req_id = checking_gid[level].my_req_id;
            log_debug(@"Maproute for $me : begin to handle a check_gid at level $level, my request_id is $req_id");
        }

        private void impl_periodically_check_gid(int level) throws Error
        {
            Tasklet.declare_self("MapRoute.periodically_check_gid");
            check_gid_handle.set(level, Tasklet.self());
            try
            {
                log_debug(@"Maproute for $me at level $level : periodically_check_gid started");
                ms_wait(1000);
                while (true)
                {
                    try
                    {
                        if (! checking_gid.has_key(level))
                        {
                            log_debug(@"Maproute for $me at level $level : periodically_check_gid completed");
                            break;
                        }
                        if (checking_gid[level].ttl.is_expired())
                        {
                            log_debug(@"Maproute for $me at level $level : periodically_check_gid time out");
                            break;
                        }
                        send_request_gid(level);
                        ms_wait(5000);
                    }
                    catch (Error e)
                    {
                        log_debug(@"Maproute for $me at level $level : periodically_check_gid error: $(e.message)");
                    }
                }
            }
            finally
            {
                log_debug(@"Maproute for $me at level $level : periodically_check_gid stopped");
            }
        }

        private static void * helper_periodically_check_gid(void *v) throws Error
        {
            struct_helper_MapRoute_periodically_check_gid *tuple_p = (struct_helper_MapRoute_periodically_check_gid *)v;
            // The caller function has to add a reference to the ref-counted instances
            MapRoute self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            int level_save = tuple_p->level;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_periodically_check_gid(level_save);
            // void method, return null
            return null;
        }

        public void periodically_check_gid(int level)
        {
            struct_helper_MapRoute_periodically_check_gid arg = struct_helper_MapRoute_periodically_check_gid();
            arg.self = this;
            arg.level = level;
            Tasklet.spawn((FunctionDelegate)helper_periodically_check_gid, &arg);
        }

        public void send_request_gid(int level) throws RPCError
        {
            // prepare udpclient for broadcast
            AddressManagerFakeRmt bclient = address_manager.get_broadcast_client();
            PartialNIP gprefix = me.get_gnode_at_level(level);
            NetworkID gnetid = get_main_netid();
            // Ask neighbours
            int request_id = checking_gid[level].my_req_id;
            log_debug(@"Maproute for $me at level $level : sending request_gid");
            bclient.maproute.request_gid(request_id, gnetid, gprefix);
        }

        public void request_gid(int request_id, NetworkID gnetid, PartialNIP gprefix) throws RPCError
        {
            // Override in Auxiliary, do nothing.
            // Check that I am mature
            if (! address_manager.is_mature)
            {
                log_debug(@"Maproute for $me : request_gid: I am not mature. Ignore.");
                return;
            }
            // Check that I am in the same network
            if (! is_in_my_network(gnetid))
            {
                log_debug(@"Maproute for $me : request_gid: not in my network. Abort.");
                return;
            }
            // Check that I am operative
            if (! address_manager.operative)
            {
                log_debug(@"Maproute for $me : request_gid: not operative. Ignore.");
                return;
            }
            // Do I am outside G?
            int level = gprefix.level_of_gnode();
            if (me.compare(gprefix) < level)
            {
                // yes.
                log_debug(@"Maproute for $me : request_gid: reached outer gnode from $gprefix");
                // Do I have the exact G in my maproute?
                if (me.compare(gprefix) < level+1)
                {
                    // no.
                    log_debug(@"Maproute for $me : request_gid: but I do not know this prefix.");
                    return;
                }
                else
                {
                    // yes. answer.
                    GNodeID? actual_gid = node_get(level, gprefix.position_at(level)).get_eldest_gid();
                    if (actual_gid == null)
                    {
                        log_debug(@"Maproute for $me : request_gid: let be someone else to answer.");
                        // let be someone else to answer.
                        return;
                    }
                    log_debug(@"Maproute for $me : request_gid: to my knowledge this prefix has gid $actual_gid");
                    // answer actual gid, propose myself to hook if necessary.
                    ArrayList<int> queue = new ArrayList<int>();
                    queue.add(request_id);
                    send_answer_gid(queue, actual_gid);
                    log_debug(@"Maproute for $me : request_gid: sent my answer in broadcast.");
                }
            }
            else
            {
                // I am inside G.
                // route the request
                log_debug(@"Maproute for $me : ");
                handle_check_gid(level);
                checking_gid[level].received_req_ids.add(request_id);
            }
        }

        public void send_answer_gid(Gee.List<int> queue, GNodeID actual_gid) throws RPCError
        {
            // prepare udpclient for broadcast
            AddressManagerFakeRmt bclient = address_manager.get_broadcast_client();
            foreach (int queued_id in queue)
            {
                log_debug(@"Maproute for $me : routing answer to id $queued_id");
                bclient.maproute.answer_gid(me, queued_id, actual_gid);
            }
        }

        public void answer_gid(NIP answering_nip, int request_id, GNodeID actual_gid) throws RPCError
        {
            log_debug(@"Maproute for $me : answer_gid: received an answer.");
            foreach (int level in checking_gid.keys)
            {
                if (checking_gid[level].my_req_id == request_id)
                {
                    log_debug(@"Maproute for $me : answer_gid: it is for me.");
                    ArrayList<int> queue = checking_gid[level].received_req_ids;
                    if (actual_gid.min_will_not_rehook < id_myself[level].min_will_not_rehook)
                    {
                        log_debug(@"Maproute for $me : answer_gid: we must rehook.");
                        // Rehook with answering_nip.
                        // If we do not find our neighbour instance, our next send_request_gid will do; just abort.
                        Gee.List<AggregatedNeighbour> neighbours0 = address_manager.aggregated_neighbour_manager.neighbour_list(true);
                        ArrayList<AggregatedNeighbour> neighbours = new ArrayList<AggregatedNeighbour>(AggregatedNeighbour.equal_func);
                        foreach (AggregatedNeighbour n in neighbours0)
                            if (n.nip.compare(answering_nip) == -1)
                                neighbours.add(n);
                        if (neighbours.is_empty)
                        {
                            log_debug(@"Maproute for $me : answer_gid: but we do not find the answering_nip.");
                            return;
                        }
                        // Emit a signal to rehook because of a gnode-split.
                        //  The data needed with the signal 'GNODE_SPLITTED' are: (neighbours=[answering_nip], queue, actual_gid)
                        gnode_splitted(neighbours, queue, actual_gid);
                        checking_gid.unset(level);
                        return;
                    }
                    else
                    {
                        log_debug(@"Maproute for $me : answer_gid: we are still ok.");
                        checking_gid.unset(level);
                        // try to propagate answer (keep in mind that we could fail,
                        //  but the other nodes will continue to ask for a while anyway)
                        send_answer_gid(queue, id_myself[level]);
                    }
                    return;
                }
            }
            log_debug(@"Maproute for $me : answer_gid: not for me.");
        }

        /** Evaluation methods **/

        /** Returns the REM of the best path that we have to
          *  reach a destination which is inside our gnode of level level_of_gnode
          *  and which has the worst REM among such destinations.
          */
        public REM worst_internal_bestrem(int level_of_gnode)
        {
            REM worstrem = new NullREM();
            for (int lvl = 0; lvl < level_of_gnode; lvl++)
            {
                for (int dst = 0; dst < gsize; dst++)
                {
                    // Don't try to measure the rem towards myself.
                    if (me.position_at(lvl) != dst)
                    {
                        RouteNode node = node_get(lvl, dst);
                        if (! node.is_free())
                        {
                            REM bestrem = node.best_route().rem;
                            // destination has to be alive
                            if (bestrem.get_type() != typeof(AlmostDeadREM) && bestrem.get_type() != typeof(DeadREM))
                            {
                                if (worstrem.compare_to(bestrem) > 0) worstrem = bestrem;
                            }
                        }
                    }
                }
            }
            return worstrem;
        }

        /** Returns minimum distance in number of hops
          */
        public int? get_distance(int lvl, int pos)
        {
            // If destination is me, distance is 0.
            if (me.position_at(lvl) == pos) return 0;
            RouteNode node = node_get(lvl, pos);
            // If not reachable / not existant, return null.
            if (node.is_free()) return null;
            int ret = -1;
            foreach (Route r in node.all_valid_routes())
            {
                int dist = r.hops_with_gw.size;
                // TODO we should consider for each hop the level and then,
                //      given an estimate of the number of nodes at each level,
                //      sum that number.
                if (ret == -1 || dist < ret)
                    ret = dist;
            }
            return ret;
        }

        /** Methods that modify routes **/

        public void update_route_by_gw(HCoord dest, AggregatedNeighbour nr, REM rem_at_gw, Gee.List<HCoord> hops, GNodeID? gid)
        {
            int lvl = dest.lvl;
            int dst = dest.pos;
            // If dest is me, strange thing. TODO review.
            if (me.position_at(lvl) == dst) return;

            RouteNode paths = node_get(lvl, dst);
            if (rem_at_gw.get_type() == typeof(DeadREM))
                paths.route_del_by_gw(nr);
            else
                paths.update_route_by_gw(nr, rem_at_gw, hops, gid);

            if (paths.is_empty())
            {
                // No more routes to reach the node (lvl, dst).
                // Consider it dead
                node_remove(lvl, dst);
            }

            // Emit event
            check_node(lvl, dst);

            _emit_routes_updated(lvl, dst);
        }

        public void route_del_by_gw(int lvl, int dst, AggregatedNeighbour gw)
        {
            // If destination is me I won't delete a route. Pretend it didn't happen.
            if (me.position_at(lvl) == dst) return;

            RouteNode paths = node_get(lvl, dst);
            paths.route_del_by_gw(gw);

            if (paths.is_empty())
            {
                // No more routes to reach the node (lvl, dst).
                // Consider it dead
                node_remove(lvl, dst);
            }

            // Emit event
            check_node(lvl, dst);

            _emit_routes_updated(lvl, dst);
        }

        public void route_signal_rem_changed(int lvl, int dst)
        {
            _emit_routes_updated(lvl, dst);
        }

        public void _emit_routes_updated(int lvl, int dst)
        {
            routes_updated(new HCoord(lvl, dst));
        }

        /** Neighbour stuff **/

        /** Delete from the MapRoute all the routes passing from the
          * gateway `neighbour' and delete the node `neighbour' itself
          * (if present)
          */
        public void delete_routes_via_neighbour(AggregatedNeighbour aggregated_neighbour)
        {
            for (int lvl = 0; lvl < levels; lvl++)
            {
                for (int dst = 0; dst < gsize; dst++)
                {
                    // Don't try deleting a route towards myself.
                    if (me.position_at(lvl) != dst)
                    {
                        RouteNode node = node_get(lvl, dst);
                        if (! node.is_free())
                        {
                            if (node.route_get_by_gw(aggregated_neighbour) != null)
                            {
                                route_del_by_gw(lvl, dst, aggregated_neighbour);
                            }
                        }
                    }
                }
            }
        }

        /** Helpers **/

        public PartialNIP? choose_between(Gee.List<PartialNIP> choose_from)
        {
            int32 probabilitysum = 0;
            ArrayList<int32> numbers = new ArrayList<int32>();
            foreach (PartialNIP nip in choose_from)
            {
                if (me.is_equal(nip))
                {
                    // it's me, choose me!
                    return nip;
                }
                else
                {
                    HCoord lvl_pos = nip.get_hcoord_relative_to(me);
                    RouteNode rn = node_get(lvl_pos.lvl, lvl_pos.pos);
                    if (rn.is_free())
                    {
                        // not this one!
                    }
                    else
                    {
                        Route best = rn.best_route();
                        if (best == null ||
                            best.rem.get_type() == typeof(DeadREM) ||
                            best.rem.get_type() == typeof(AlmostDeadREM))
                        {
                            // not this one!
                        }
                        else
                        {
                            // a possibility.
                            assert(best.rem.get_type() == typeof(RTT));
                            probabilitysum += (int32)100000 / (best.rem as RTT).delay;
                        }
                    }
                }
                numbers.add(probabilitysum);
            }
            // now the choice
            int32 chosen = Random.int_range(0, probabilitysum);
            int pos = 0;
            foreach (PartialNIP nip in choose_from)
            {
                int32 threshold = numbers[pos++];
                if (chosen <= threshold) return nip;
            }
            return null;
        }

        public PartialNIP? choose_fast(Gee.List<PartialNIP> choose_from)
        {
            return choose_between(choose_from);
        }
    }
}

