/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
	public errordomain RadarError {
		DUPL
	}

    struct struct_helper_Radar_run
    {
        public Radar self;
    }

    /** An instance of Radar uses a certain network interface on behalf of a certain AddressManager.
      * The radar sends in broadcast a bouquet of MAX_BOUQUET packets and waits
      * for the reply of the alive nodes. It then recollects the replies and builds
      * a small statistic.
      * Then it reports the detected neighbours to the NeighbourManager(s) interested
      * in that network interface.
      *
      * A radar is fired periodically by Radar.run(), which is started as a
      * microfunc.
      */
    public class Radar : Object
    {
        protected string dev_name;
        protected AddressManager address_manager;
        protected int bouquet_numb;
        private HashMap<string, ArrayList<int?>> gather_rtts;
        private HashMap<string, ArrayList<string>> gather_macs;
        private HashMap<string, int> gather_levels;
        private HashMap<string, int> gather_gsize;
        private HashMap<string, NetworkID> gather_netid;
        private HashMap<string, bool> gather_is_primary;
        private HashMap<string, bool> gather_is_auxiliary;
        protected int max_bouquet;
        protected int wait_time;
        private int max_neighbours;
        protected AddressManagerBroadcastClient bcastclient;
        private Tasklet? tasklet = null;
        protected ArrayList<int> radar_ids;
        private HashMap<int, int64?> bcast_send_time;
        // Send a SCAN_DONE event each time a sent bouquet has been completely
        // collected
        public signal void scan_done(int bouquet_numb, HashMap<string, Neighbour> detected_neighbours);
        public signal void radar_stop();

        protected Radar.fake() {}  // to allow derivation of Glue
        public Radar(string dev_name, AddressManager address_manager)
        {
            this.dev_name = dev_name;
            this.address_manager = address_manager;

            // how many bouquet we have already sent
            this.bouquet_numb = 0;
            // when the replies arrived (many per item)
            this.gather_rtts = new HashMap<string, ArrayList<int?>>();
            // MACs gathered (many per item)
            this.gather_macs = new HashMap<string, ArrayList<string>>();
            // levels, gsize, netid gathered (one per item)
            this.gather_levels = new HashMap<string, int>();
            this.gather_gsize = new HashMap<string, int>();
            this.gather_netid = new HashMap<string, NetworkID>();
            this.gather_is_primary = new HashMap<string, bool>();
            this.gather_is_auxiliary = new HashMap<string, bool>();
            // max_bouquet: how many packets does each bouquet contain?
            this.max_bouquet = Settings.MAX_BOUQUET;
            // wait_time: the time we wait for a reply, in seconds
            this.wait_time = Settings.RADAR_WAIT_TIME;
            // max_neighbours: maximum number of neighbours we can have
            this.max_neighbours = Settings.MAX_NEIGHBOURS;

            bcastclient = new AddressManagerBroadcastClient(new BroadcastID(),
                                                            new string[] {this.dev_name});
            this.tasklet = null;
        }

        private void impl_run() throws Error
        {
            Tasklet.declare_self("Radar.run");
            if (tasklet != null)
            {
                throw new RadarError.DUPL("An instance of Radar is already running");
            }
            tasklet = Tasklet.self();
            while (true)
            {
                radar();
            }
        }

        private static void * helper_run(void *v) throws Error
        {
            struct_helper_Radar_run *tuple_p = (struct_helper_Radar_run *)v;
            // The caller function has to add a reference to the ref-counted instances
            Radar self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_run();
            // void method, return null
            return null;
        }

        public void run()
        {
            struct_helper_Radar_run arg = struct_helper_Radar_run();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_run, &arg);
        }

        /** Stop the radar scanner
          */
        public void stop()
        {
            if (tasklet != null)
            {
                tasklet.abort();
                tasklet = null;
                radar_stop();
            }
        }

        public virtual void radar()
        {
            try
            {
                // We have to send self.max_bouquet packets in self.wait_time seconds.
                // We send a packet at each time in self.wait_time/2 seconds, and then wait.
                radar_ids = new ArrayList<int>();
                bcast_send_time = new HashMap<int, int64?>();
                double interval = (double)(wait_time) / (double)(max_bouquet * 2);
                for (int i = 0; i < max_bouquet; i++)
                {
                    int radar_id = Random.int_range(0, ((int)Math.pow(2, 32))-1);
                    TimeVal time_now = TimeVal();
                    time_now.get_current_time();
                    int64 msec_now = (int64)time_now.tv_sec * (int64)1000 + (int64)time_now.tv_usec / (int64)1000;
                    radar_ids.add(radar_id);
                    bcast_send_time[radar_id] = msec_now;
                    NIP nip = address_manager.maproute.me;
                    NetworkID netid = address_manager.get_main_netid();
                    int nodeid = address_manager.get_my_id();
                    bcastclient.aggregated_neighbour_manager.reply(radar_id, nip, nodeid, netid);
                    ms_wait((long)(interval * 1000));
                }
                ms_wait(wait_time * 500);

                // Notify the detected neighbours
                bouquet_numb += 1;
                HashMap<string, Neighbour> detected_neighbours = get_all_avg_rtt();
                log_debug(@"radar: emits SCAN_DONE for $dev_name");
                scan_done(bouquet_numb, detected_neighbours);
            }
            catch (Error e)
            {
                log_warn("Exception %s while doing a radar scan. We ignore it. Soon another scan."
                                .printf("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message)));
                bcastclient = new AddressManagerBroadcastClient(new BroadcastID(),
                                                                new string[] {this.dev_name});
                ms_wait(500);
            }

            radar_reset();
        }

        /** Clean the objects needed by radar()
          */
        public void radar_reset()
        {
            // Clean some stuff
            this.gather_rtts = new HashMap<string, ArrayList<int?>>();
        }

        /** save each node's rtt
          */
        public virtual void time_register(int radar_id, int levels, int gsize, NIP nip, int nodeid,
                        NetworkID netid, string mac, bool is_primary, bool is_auxiliary)
        {
            if (! radar_ids.contains(radar_id))
            {
                // drop. It isn't a reply to our current bouquet
                return;
            }

            int64 msec_send_time = bcast_send_time[radar_id];
            string key = tnip_nodeid_key(nip, nodeid);
            // this is the rtt
            TimeVal time_now = TimeVal();
            time_now.get_current_time();
            int64 msec_now = (int64)time_now.tv_sec * (int64)1000 + (int64)time_now.tv_usec / (int64)1000;
            int time_elapsed = (int)(msec_now - msec_send_time);
            // let's store it in the gather_rtts table
            if (gather_rtts.has_key(key))
            {
                gather_rtts[key].add(time_elapsed);
                if (! gather_macs[key].contains(mac))
                    gather_macs[key].add(mac);
            }
            else
            {
                gather_rtts[key] = new ArrayList<int?>();
                gather_rtts[key].add(time_elapsed);
                gather_macs[key] = new ArrayList<string>();
                gather_macs[key].add(mac);
                gather_levels[key] = levels;
                gather_gsize[key] = gsize;
                gather_netid[key] = netid;
                gather_is_primary[key] = is_primary;
                gather_is_auxiliary[key] = is_auxiliary;
                log_debug(@"Radar: IP $(nip_to_str(levels, gsize, nip)) from network $netid detected");
            }
        }

        /** Calculates the average RTT of the neighbour for the device associated to this radar.
          */
        public int get_avg_rtt(string key)
        {
            ArrayList<int?> rtts = gather_rtts[key];
            int sum = 0;
            foreach (int rtt in rtts) sum += rtt;
            return sum / rtts.size;
        }

        protected string tnip_nodeid_key(NIP key_nip, int key_nodeid)
        {
            string ret = "";
            int[] nip = key_nip.get_positions();
            foreach (int i in nip) ret += "%d_".printf(i);
            ret += "%d".printf(key_nodeid);
            return ret;
        }
        protected void key_tnip_nodeid(string key, out NIP key_nip, out int key_nodeid)
        {
            string[] nums = key.split("_");
            int[] pos = new int[nums.length - 1];
            for (int i = 0; i < pos.length; i++) pos[i] = int.parse(nums[i]);
            key_nodeid = int.parse(nums[nums.length - 1]);
            key_nip = new NIP(pos);
        }

        /** Calculate the average rtt of all the neighbours
          */
        public virtual HashMap<string, Neighbour> get_all_avg_rtt()
        {
            HashMap<string, Neighbour> all_avg = new HashMap<string, Neighbour>();
            foreach (string key in gather_rtts.keys)
            {
                NIP nip;
                int nodeid;
                key_tnip_nodeid(key, out nip, out nodeid);
                int avg = get_avg_rtt(key);
                all_avg[key] = new Neighbour.with_rtt(gather_levels[key], gather_gsize[key],
                                   nip, nodeid, gather_netid[key],
                                   gather_is_primary[key], gather_is_auxiliary[key],
                                   dev_name, gather_macs[key], avg);
            }
            return all_avg;
        }

        public string to_string()
        {
            return @"<Radar for device $(dev_name)>";
        }
    }
}
