/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    class SettingsInstance : Object
    {
        public static string init_file = null;
        private static SettingsInstance _settings = null;
        public static SettingsInstance settings {
            get {
                if (_settings == null)
                {
                    if (init_file == null) init_file = "ntkd.ini";
                    _settings = new SettingsInstance(init_file);
                }
                return _settings;
            }
        }

        private SettingsInstance(string config_file)
        {
            // per-OS default settings
#if linux
            CONFIGURATION_DIR = "/etc/netsukuku";
            DATA_DIR = "/usr/share/netsukuku";
            LOG_DIR = "/var/log";
            STATUS_XML_DIR = "/var/tmp";
            PID_DIR = "/var/run";
            TMP_DIR = "/tmp";
            SERVNO_PATH = "/etc/services";
            RESOLV_PATH = "/etc/resolv.conf";
            NETWORK_IMPLEMENTATION = "linux";
            // Size of the RAM to be allocated as the stack of new tasklets when
            //  not specified by the program. Default for GNU Pth is 64 kB. One could play
            //  with this value if he has a device with few RAM and see if this helps,
            //  but beware that a low value could lead to unexpected behaviour.
            THREAD_STACK_SIZE = 64; // kB
#elif win32
            error("Your platform is not supported yet");
#else
            error("Your platform is not supported yet");
#endif

            // common default settings
            CONFIGURATION_FILE = config_file;
            VERBOSE_LEVEL = 0;
            LOGS_ON_SCREEN = true;
            LOG_FILE = "netsukuku.log";
            STATUS_XML_FILE = "node_status.xml";
            // inet
            IP_VERSION = 4;
            // Dimensions
            BITS_PER_LEVEL = 3;
            LEVELS = 8;
            // Radar
            MAX_BOUQUET = 8;
            MAX_NEIGHBOURS = 16;
            RADAR_WAIT_TIME = 5; // seconds
            MULTIPATH = true;
            // Specify if you want to avoid tunneling between bnodes.
            DO_NOT_TUNNEL = true;
            // Specify if you want to avoid migrations. Use carefully one of the following:
            // MIGRATION_CAPABILITY = 1 => I am a crucial server, try to avoid to change address
            // MIGRATION_CAPABILITY = 2 => I am used occasionally as a server, be gentle
            // MIGRATION_CAPABILITY = 3 => I am a client, I can come and go quite easily
            MIGRATION_CAPABILITY = 3;
            // Replication code is experimental
            ANDNA_REPLICA_ACTIVATED = true;
            COUNTER_REPLICA_ACTIVATED = true;
            // Internet Sharing options (IGS)
            IGS_MODE = "NONE";
            // NONE:    do not share my Internet link, do not use others' ones (use my own if available)
            // SHARE:   share and use my Internet link (when available), do not use others' ones
            // USE:     have not an Internet link, use others' ones
            // BOTH:    share and use my Internet link (when available), use also others' ones
            // depending on the implementations, some mode might not work
            // e.g. current linux cannot use BOTH
            IGS_ANNOUNCE_INTERNET = "ALWAYS";
            // only used when IGS_MODE is SHARE or BOTH (otherwise I never announce my Internet link)
            // <IP Address in dot notation>:  announce only when this IP is pingable
            // ALWAYS:  announce always
            IGS_TUNNEL = 4;
            // only used when IGS_MODE is USE or BOTH (otherwise I do not set a default path, it might
            //   be set by someone else in advance)
            // N, with 0 < N <= MAX_IGS: try to create a tunnel with up to N inet-gateway
            //   and multipath to Internet through them

            load_configuration();

            if (KEY_PAIR_DIR == null)
                KEY_PAIR_DIR = CONFIGURATION_DIR + Path.DIR_SEPARATOR_S + "keys";
            if (KEY_PAIR_PATH == null)
                KEY_PAIR_PATH = KEY_PAIR_DIR + Path.DIR_SEPARATOR_S + "my_key_pair";
            if (SNSD_NODES_PATH == null)
                SNSD_NODES_PATH = CONFIGURATION_DIR + Path.DIR_SEPARATOR_S + "snsd_nodes";
            if (PUB_KEY_PATH == null)
                PUB_KEY_PATH = DATA_DIR + Path.DIR_SEPARATOR_S + "andna_lcl_keyring";
            if (LOCAL_CACHE_PATH == null)
                LOCAL_CACHE_PATH = CONFIGURATION_DIR + Path.DIR_SEPARATOR_S + "local_cache";

            TESTING = false;
        }

        void load_configuration()
        {
            string fname;
            if (Path.is_absolute(CONFIGURATION_FILE)) fname = CONFIGURATION_FILE;
            else fname = CONFIGURATION_DIR + Path.DIR_SEPARATOR_S + CONFIGURATION_FILE;
            KeyFile conf = new KeyFile();
            try
            {
                bool file_found = false;
                try
                {
                    conf.load_from_file(fname, KeyFileFlags.NONE);
                    file_found = true;
                }
                catch (KeyFileError e)
                {
                    if (e is KeyFileError.NOT_FOUND)
                    {
                        // No configuration file. Ignore it.
                    }
                    else
                    {
                        error(e.message);
                    }
                }
                if (file_found)
                {
                    if (conf.has_group("NTKD"))
                    {
                        // reading value CONFIGURATION_FILE makes no sense, but reading value CONFIGURATION_DIR does.
                        if (conf.has_key("NTKD", "CONFIGURATION_DIR")) CONFIGURATION_DIR = conf.get_string("NTKD", "CONFIGURATION_DIR");
                        if (conf.has_key("NTKD", "DATA_DIR")) DATA_DIR = conf.get_string("NTKD", "DATA_DIR");
                        if (conf.has_key("NTKD", "LOG_DIR")) LOG_DIR = conf.get_string("NTKD", "LOG_DIR");
                        if (conf.has_key("NTKD", "STATUS_XML_DIR")) STATUS_XML_DIR = conf.get_string("NTKD", "STATUS_XML_DIR");
                        if (conf.has_key("NTKD", "PID_DIR")) PID_DIR = conf.get_string("NTKD", "PID_DIR");
                        if (conf.has_key("NTKD", "TMP_DIR")) TMP_DIR = conf.get_string("NTKD", "TMP_DIR");
                        if (conf.has_key("NTKD", "SERVNO_PATH")) SERVNO_PATH = conf.get_string("NTKD", "SERVNO_PATH");
                        if (conf.has_key("NTKD", "RESOLV_PATH")) RESOLV_PATH = conf.get_string("NTKD", "RESOLV_PATH");
                        if (conf.has_key("NTKD", "VERBOSE_LEVEL")) VERBOSE_LEVEL = conf.get_integer("NTKD", "VERBOSE_LEVEL");
                        if (conf.has_key("NTKD", "LOGS_ON_SCREEN")) LOGS_ON_SCREEN = conf.get_boolean("NTKD", "LOGS_ON_SCREEN");
                        if (conf.has_key("NTKD", "LOG_FILE")) LOG_FILE = conf.get_string("NTKD", "LOG_FILE");
                        if (conf.has_key("NTKD", "STATUS_XML_FILE")) STATUS_XML_FILE = conf.get_string("NTKD", "STATUS_XML_FILE");
                        if (conf.has_key("NTKD", "MULTIPATH")) MULTIPATH = conf.get_boolean("NTKD", "MULTIPATH");
                        if (conf.has_key("NTKD", "DO_NOT_TUNNEL")) DO_NOT_TUNNEL = conf.get_boolean("NTKD", "DO_NOT_TUNNEL");
                        if (conf.has_key("NTKD", "MIGRATION_CAPABILITY")) MIGRATION_CAPABILITY = conf.get_integer("NTKD", "MIGRATION_CAPABILITY");
                        if (conf.has_key("NTKD", "ANDNA_REPLICA_ACTIVATED")) ANDNA_REPLICA_ACTIVATED = conf.get_boolean("NTKD", "ANDNA_REPLICA_ACTIVATED");
                        if (conf.has_key("NTKD", "COUNTER_REPLICA_ACTIVATED")) COUNTER_REPLICA_ACTIVATED = conf.get_boolean("NTKD", "COUNTER_REPLICA_ACTIVATED");
                        if (conf.has_key("NTKD", "IGS_MODE")) IGS_MODE = conf.get_string("NTKD", "IGS_MODE");
                        if (conf.has_key("NTKD", "IGS_ANNOUNCE_INTERNET")) IGS_ANNOUNCE_INTERNET = conf.get_string("NTKD", "IGS_ANNOUNCE_INTERNET");
                        if (conf.has_key("NTKD", "IGS_TUNNEL")) IGS_TUNNEL = conf.get_integer("NTKD", "IGS_TUNNEL");
                        if (conf.has_key("NTKD", "SNSD_NODES_PATH")) SNSD_NODES_PATH = conf.get_string("NTKD", "SNSD_NODES_PATH");
                        if (conf.has_key("NTKD", "LOCAL_CACHE_PATH")) LOCAL_CACHE_PATH = conf.get_string("NTKD", "LOCAL_CACHE_PATH");
                        if (conf.has_key("NTKD", "NETWORK_IMPLEMENTATION")) NETWORK_IMPLEMENTATION = conf.get_string("NTKD", "NETWORK_IMPLEMENTATION");
                        if (conf.has_key("NTKD", "KEY_PAIR_DIR")) KEY_PAIR_DIR = conf.get_string("NTKD", "KEY_PAIR_DIR");
                        if (conf.has_key("NTKD", "KEY_PAIR_PATH")) KEY_PAIR_PATH = conf.get_string("NTKD", "KEY_PAIR_PATH");
                        if (conf.has_key("NTKD", "PUB_KEY_PATH")) PUB_KEY_PATH = conf.get_string("NTKD", "PUB_KEY_PATH");
                        if (conf.has_key("NTKD", "IPROUTE_PATH")) IPROUTE_PATH = conf.get_string("NTKD", "IPROUTE_PATH");
                        if (conf.has_key("NTKD", "IPTABLES_PATH")) IPTABLES_PATH = conf.get_string("NTKD", "IPTABLES_PATH");
                        if (conf.has_key("NTKD", "PING_PATH")) PING_PATH = conf.get_string("NTKD", "PING_PATH");
                        if (conf.has_key("NTKD", "IP_VERSION")) IP_VERSION = conf.get_integer("NTKD", "IP_VERSION");
                        if (conf.has_key("NTKD", "BITS_PER_LEVEL")) BITS_PER_LEVEL = conf.get_integer("NTKD", "BITS_PER_LEVEL");
                        if (conf.has_key("NTKD", "LEVELS")) LEVELS = conf.get_integer("NTKD", "LEVELS");
                        // FORCE_NIC_RESET can be set only by command line
                        if (conf.has_key("NTKD", "TESTING")) TESTING = conf.get_boolean("NTKD", "TESTING");
                        if (conf.has_key("NTKD", "MAX_BOUQUET")) MAX_BOUQUET = conf.get_integer("NTKD", "MAX_BOUQUET");
                        if (conf.has_key("NTKD", "RADAR_WAIT_TIME")) RADAR_WAIT_TIME = conf.get_integer("NTKD", "RADAR_WAIT_TIME");
                        if (conf.has_key("NTKD", "MAX_NEIGHBOURS")) MAX_NEIGHBOURS = conf.get_integer("NTKD", "MAX_NEIGHBOURS");
                        if (conf.has_key("NTKD", "THREAD_STACK_SIZE")) THREAD_STACK_SIZE = conf.get_integer("NTKD", "THREAD_STACK_SIZE");
                    }
                }
            }
            catch (Error e) {error(e.message);}
        }

        public string CONFIGURATION_FILE;
        public string CONFIGURATION_DIR;
        public string DATA_DIR;
        public string LOG_DIR;
        public string STATUS_XML_DIR;
        public string PID_DIR;
        public string TMP_DIR;
        public string SERVNO_PATH;
        public string RESOLV_PATH;
        public int VERBOSE_LEVEL;
        public bool LOGS_ON_SCREEN;
        public string LOG_FILE;
        public string STATUS_XML_FILE;
        public bool MULTIPATH;
        public bool DO_NOT_TUNNEL;
        public int MIGRATION_CAPABILITY;
        public bool ANDNA_REPLICA_ACTIVATED;
        public bool COUNTER_REPLICA_ACTIVATED;
        public string IGS_MODE;
        public string IGS_ANNOUNCE_INTERNET;
        public int IGS_TUNNEL;
        public string SNSD_NODES_PATH;
        public string LOCAL_CACHE_PATH;
        public ArrayList<string> NICS;
        public ArrayList<string> EXCLUDE_NICS;
        public string NETWORK_IMPLEMENTATION;
        public string KEY_PAIR_DIR;
        public string KEY_PAIR_PATH;
        public string PUB_KEY_PATH;
        public string IPROUTE_PATH;
        public string IPTABLES_PATH;
        public string PING_PATH;
        public int IP_VERSION;
        public int BITS_PER_LEVEL;
        public int LEVELS;
        public bool FORCE_NIC_RESET;
        public bool TESTING;
        public int MAX_BOUQUET;
        public int RADAR_WAIT_TIME;
        public int MAX_NEIGHBOURS;
        public int THREAD_STACK_SIZE;
    }

    public class Settings : Object
    {
        // makes sense only if we set the value before accessing any other value
        public static void init(string config_file)
        {
            SettingsInstance.init_file = config_file;
        }

        public static string CONFIGURATION_FILE {
            get {return SettingsInstance.settings.CONFIGURATION_FILE;}
        }

        public static string CONFIGURATION_DIR {
            get {return SettingsInstance.settings.CONFIGURATION_DIR;}
            set {SettingsInstance.settings.CONFIGURATION_DIR = value;}
        }

        public static string DATA_DIR {
            get {return SettingsInstance.settings.DATA_DIR;}
            set {SettingsInstance.settings.DATA_DIR = value;}
        }

        public static string LOG_DIR {
            get {return SettingsInstance.settings.LOG_DIR;}
            set {SettingsInstance.settings.LOG_DIR = value;}
        }

        public static string STATUS_XML_DIR {
            get {return SettingsInstance.settings.STATUS_XML_DIR;}
            set {SettingsInstance.settings.STATUS_XML_DIR = value;}
        }

        public static string PID_DIR {
            get {return SettingsInstance.settings.PID_DIR;}
            set {SettingsInstance.settings.PID_DIR = value;}
        }

        public static string TMP_DIR {
            get {return SettingsInstance.settings.TMP_DIR;}
            set {SettingsInstance.settings.TMP_DIR = value;}
        }

        public static string SERVNO_PATH {
            get {return SettingsInstance.settings.SERVNO_PATH;}
            set {SettingsInstance.settings.SERVNO_PATH = value;}
        }

        public static string RESOLV_PATH {
            get {return SettingsInstance.settings.RESOLV_PATH;}
            set {SettingsInstance.settings.RESOLV_PATH = value;}
        }

        public static int VERBOSE_LEVEL {
            get {return SettingsInstance.settings.VERBOSE_LEVEL;}
            set {SettingsInstance.settings.VERBOSE_LEVEL = value;}
        }

        public static bool LOGS_ON_SCREEN {
            get {return SettingsInstance.settings.LOGS_ON_SCREEN;}
            set {SettingsInstance.settings.LOGS_ON_SCREEN = value;}
        }

        public static string LOG_FILE {
            get {return SettingsInstance.settings.LOG_FILE;}
            set {SettingsInstance.settings.LOG_FILE = value;}
        }

        public static string STATUS_XML_FILE {
            get {return SettingsInstance.settings.STATUS_XML_FILE;}
            set {SettingsInstance.settings.STATUS_XML_FILE = value;}
        }

        public static bool MULTIPATH {
            get {return SettingsInstance.settings.MULTIPATH;}
            set {SettingsInstance.settings.MULTIPATH = value;}
        }

        public static bool DO_NOT_TUNNEL {
            get {return SettingsInstance.settings.DO_NOT_TUNNEL;}
            set {SettingsInstance.settings.DO_NOT_TUNNEL = value;}
        }

        public static int MIGRATION_CAPABILITY {
            get {return SettingsInstance.settings.MIGRATION_CAPABILITY;}
            set {SettingsInstance.settings.MIGRATION_CAPABILITY = value;}
        }

        public static bool ANDNA_REPLICA_ACTIVATED {
            get {return SettingsInstance.settings.ANDNA_REPLICA_ACTIVATED;}
            set {SettingsInstance.settings.ANDNA_REPLICA_ACTIVATED = value;}
        }

        public static bool COUNTER_REPLICA_ACTIVATED {
            get {return SettingsInstance.settings.COUNTER_REPLICA_ACTIVATED;}
            set {SettingsInstance.settings.COUNTER_REPLICA_ACTIVATED = value;}
        }

        public static string IGS_MODE {
            get {return SettingsInstance.settings.IGS_MODE;}
            set {SettingsInstance.settings.IGS_MODE = value;}
        }

        public static string IGS_ANNOUNCE_INTERNET {
            get {return SettingsInstance.settings.IGS_ANNOUNCE_INTERNET;}
            set {SettingsInstance.settings.IGS_ANNOUNCE_INTERNET = value;}
        }

        public static int IGS_TUNNEL {
            get {return SettingsInstance.settings.IGS_TUNNEL;}
            set {SettingsInstance.settings.IGS_TUNNEL = value;}
        }

        public static string SNSD_NODES_PATH {
            get {return SettingsInstance.settings.SNSD_NODES_PATH;}
            set {SettingsInstance.settings.SNSD_NODES_PATH = value;}
        }

        public static string LOCAL_CACHE_PATH {
            get {return SettingsInstance.settings.LOCAL_CACHE_PATH;}
            set {SettingsInstance.settings.LOCAL_CACHE_PATH = value;}
        }

        public static ArrayList<string> NICS {
            get {return SettingsInstance.settings.NICS;}
            set {SettingsInstance.settings.NICS = value;}
        }

        public static ArrayList<string> EXCLUDE_NICS {
            get {return SettingsInstance.settings.EXCLUDE_NICS;}
            set {SettingsInstance.settings.EXCLUDE_NICS = value;}
        }

        public static string NETWORK_IMPLEMENTATION {
            get {return SettingsInstance.settings.NETWORK_IMPLEMENTATION;}
            set {SettingsInstance.settings.NETWORK_IMPLEMENTATION = value;}
        }

        public static string KEY_PAIR_DIR {
            get {return SettingsInstance.settings.KEY_PAIR_DIR;}
            set {SettingsInstance.settings.KEY_PAIR_DIR = value;}
        }

        public static string KEY_PAIR_PATH {
            get {return SettingsInstance.settings.KEY_PAIR_PATH;}
            set {SettingsInstance.settings.KEY_PAIR_PATH = value;}
        }

        public static string PUB_KEY_PATH {
            get {return SettingsInstance.settings.PUB_KEY_PATH;}
            set {SettingsInstance.settings.PUB_KEY_PATH = value;}
        }

        public static string IPROUTE_PATH {
            get {return SettingsInstance.settings.IPROUTE_PATH;}
            set {SettingsInstance.settings.IPROUTE_PATH = value;}
        }

        public static string IPTABLES_PATH {
            get {return SettingsInstance.settings.IPTABLES_PATH;}
            set {SettingsInstance.settings.IPTABLES_PATH = value;}
        }

        public static string PING_PATH {
            get {return SettingsInstance.settings.PING_PATH;}
            set {SettingsInstance.settings.PING_PATH = value;}
        }

        public static int IP_VERSION {
            get {return SettingsInstance.settings.IP_VERSION;}
            set {SettingsInstance.settings.IP_VERSION = value;}
        }

        public static int BITS_PER_LEVEL {
            get {return SettingsInstance.settings.BITS_PER_LEVEL;}
            set {SettingsInstance.settings.BITS_PER_LEVEL = value;}
        }

        public static int LEVELS {
            get {return SettingsInstance.settings.LEVELS;}
            set {SettingsInstance.settings.LEVELS = value;}
        }

        public static bool FORCE_NIC_RESET {
            get {return SettingsInstance.settings.FORCE_NIC_RESET;}
            set {SettingsInstance.settings.FORCE_NIC_RESET = value;}
        }

        public static bool TESTING {
            get {return SettingsInstance.settings.TESTING;}
            set {SettingsInstance.settings.TESTING = value;}
        }

        public static int MAX_BOUQUET {
            get {return SettingsInstance.settings.MAX_BOUQUET;}
            set {SettingsInstance.settings.MAX_BOUQUET = value;}
        }

        public static int RADAR_WAIT_TIME {
            get {return SettingsInstance.settings.RADAR_WAIT_TIME;}
            set {SettingsInstance.settings.RADAR_WAIT_TIME = value;}
        }

        public static int MAX_NEIGHBOURS {
            get {return SettingsInstance.settings.MAX_NEIGHBOURS;}
            set {SettingsInstance.settings.MAX_NEIGHBOURS = value;}
        }

        public static int THREAD_STACK_SIZE {
            get {return SettingsInstance.settings.THREAD_STACK_SIZE;}
            set {SettingsInstance.settings.THREAD_STACK_SIZE = value;}
        }
    }
}

