/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    /** This namespace groups functions related to crypto (libgcrypt)
      */
    namespace Crypto
    {

        public errordomain GCryptError {
            FAILED
        }

        /** 32 bit Fowler/Noll/Vo hash
          */
        public uint32 fnv_32(uchar[] buf)
        {
            uint32 hval = (uint32)2166136261;
            foreach (uchar c in buf)
            {
                hval += (hval<<1) + (hval<<4) + (hval<<7) + (hval<<8) + (hval<<24);
                hval ^= (uint8)c;
            }
            return hval;
        }

        public errordomain FileError {GENERIC}
        void write_file(string pathname, uint8[] buf) throws FileError
        {
            FileStream fout = FileStream.open(pathname, "w"); // truncate to 0 | create
            if (fout == null) throw new FileError.GENERIC(@"Unable to write $pathname");
            fout.write(buf);
        }

        uint8[]? read_file(string pathname, int maxlen)
        {
            uint8[] buf = new uint8[maxlen];
            FileStream? fin = FileStream.open(pathname, "r"); // read from 0
            if (fin == null) return null;
            size_t len = fin.read(buf);
            uint8[] ret = new uint8[len];
            for (int i = 0; i < len; i++) ret[i] = buf[i];
            return ret;
        }

        GCrypt.SExp getpubkey(GCrypt.SExp keypair)
        {
            GCrypt.SExp ret = keypair.find_token("public-key");
            return ret;
        }

        GCrypt.SExp getprivkey(GCrypt.SExp keypair)
        {
            GCrypt.SExp ret = keypair.find_token("private-key");
            return ret;
        }

        GCrypt.SExp recreate_sexp(uint8[] buffer) throws GCryptError
        {
            void *data = (void *)buffer;
            int size = buffer.length;
            GCrypt.SExp sexp;
            GCrypt.Error err = GCrypt.SExp.new(out sexp, data, size, 1);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return sexp;
        }

        uint8[] serialize_sexp(GCrypt.SExp sexp)
        {
            char[] buffer2 = new char[2000];
            size_t len = sexp.sprint(GCrypt.SExp.Format.CANON, buffer2);
            uint8[] buffer = new uint8[len];
            for (int i = 0; i < len; i++) buffer[i] = buffer2[i];
            return buffer;
        }

        GCrypt.SExp generate_keypair() throws GCryptError
        {
            GCrypt.SExp parms;
            GCrypt.Error err = GCrypt.SExp.build(out parms, null, "(genkey (rsa (nbits 4:1024)(transient-key)))");
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            GCrypt.SExp keypair;
            err = GCrypt.PublicKey.genkey(out keypair, parms);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return keypair;
        }

        uint8[] md5(uint8[] buffer) throws GCryptError
        {
            GCrypt.Hash hash;
            GCrypt.Error err = GCrypt.Hash.open(out hash, GCrypt.Hash.Algorithm.MD5, (GCrypt.Hash.Flag)0);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            hash.write((uchar[])buffer);
            hash.final();
            unowned uchar *tmp = hash.read(GCrypt.Hash.Algorithm.MD5);
            size_t len = GCrypt.Hash.Algorithm.MD5.get_digest_length();
            uint8[] ret = new uint8[len];
            for (int i = 0; i < len; i++) ret[i] = *(tmp+i);
            return ret;
        }

        GCrypt.SExp sign(uint8[] buffer, GCrypt.SExp skey) throws GCryptError
        {
            GCrypt.SExp ret;
            GCrypt.SExp data;

            GCrypt.Error err = GCrypt.SExp.build(out data, null, "(data (flags pkcs1) (hash md5 %b ))", buffer.length, (char*)buffer);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }

            err = GCrypt.PublicKey.sign(out ret, data, skey);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return ret;
        }

        bool verify(GCrypt.SExp sig, uint8[] buffer, GCrypt.SExp pkey) throws GCryptError
        {
            GCrypt.SExp data;

            GCrypt.Error err = GCrypt.SExp.build(out data, null, "(data (flags pkcs1) (hash md5 %b ))", buffer.length, (char*)buffer);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }

            bool ret = true;
            err = GCrypt.PublicKey.verify(sig, data, pkey);
            if (err != 0)
            {
                if (err.code() == GCrypt.ErrorCode.BAD_SIGNATURE)
                {
                    ret = false;
                }
                else
                {
                    throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
                }
            }
            return ret;
        }

        GCrypt.SExp encrypt_session_key(uchar[] buffer, GCrypt.SExp pkey) throws GCryptError
        {
            GCrypt.SExp ret;
            GCrypt.SExp data;

            GCrypt.Error err = GCrypt.SExp.build(out data, null, "(data (flags pkcs1) (value %b ))", buffer.length, buffer);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            err = GCrypt.PublicKey.encrypt(out ret, data, pkey);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return ret;
        }

        char[] encrypt_session_key_get_rsa_mpi(uchar[] buffer, GCrypt.SExp pkey) throws GCryptError
        {
            GCrypt.SExp ciph = encrypt_session_key(buffer, pkey);
            GCrypt.SExp rsapart = ciph.find_token("rsa").find_token("a");
            unowned char[] data = rsapart.nth_data(1);
            char[] ret = new char[data.length];
            for (int i = 0; i < data.length; i++)
                ret[i] = data[i];
            return ret;
        }

        uchar[] decrypt_session_key_from_rsa_mpi(char[] rsapart, GCrypt.SExp skey) throws GCryptError
        {
            GCrypt.SExp encval;
            GCrypt.Error err = GCrypt.SExp.build(out encval, null, "(enc-val (flags pkcs1) (rsa (a %b )))", rsapart.length, rsapart);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return decrypt_session_key_from_encval(encval, skey);
        }

        uchar[] decrypt_session_key_from_ciph(GCrypt.SExp ciph, GCrypt.SExp skey) throws GCryptError
        {
            GCrypt.SExp encval;
            GCrypt.SExp rsapart = ciph.find_token("rsa");
            GCrypt.Error err = GCrypt.SExp.build(out encval, null, "(enc-val (flags pkcs1) %S )", rsapart);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return decrypt_session_key_from_encval(encval, skey);
        }

        uchar[] decrypt_session_key_from_encval(GCrypt.SExp encval, GCrypt.SExp skey) throws GCryptError
        {
            uchar[] ret = new uchar[0];
            GCrypt.SExp plain;

            GCrypt.Error err = GCrypt.PublicKey.decrypt(out plain, encval, skey);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }

            unowned char[] data = plain.nth_data(1);
            if (Settings.TESTING) {stdout.printf("\ndecrypt_session_key_from_encval: nth_data(1) follows:\n"); hex_dump((uint8[])data);}
            bool zerofound = false;
            int validpos = 0;
            for (int i = 0; i < data.length; i++)
            {
                if (zerofound)
                {
                    ret[validpos++] = data[i];
                }
                else
                {
                    if (data[i] == '\0')
                    {
                        zerofound = true;
                        ret = new uchar[data.length - i - 1];
                    }
                }
            }
            if (!zerofound)
            {
                ret = new uchar[data.length];
                for (int i = 0; i < data.length; i++)
                    ret[i] = data[i];
            }

            return ret;
        }

        uchar[] encrypt(uchar[] message, uchar[] key, uchar[] iv) throws GCryptError
        {
            GCrypt.Cipher.Cipher hd;
            GCrypt.Error err = GCrypt.Cipher.Cipher.open(out hd, GCrypt.Cipher.Algorithm.ARCFOUR, GCrypt.Cipher.Mode.STREAM, (GCrypt.Cipher.Flag)0);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            err = hd.set_key(key);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            err = hd.set_iv(iv);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            uchar[] ciph = new uchar[message.length];
            err = hd.encrypt(ciph, message);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return ciph;
        }

        uchar[] decrypt(uchar[] ciph, uchar[] key, uchar[] iv) throws GCryptError
        {
            GCrypt.Cipher.Cipher hd;
            GCrypt.Error err = GCrypt.Cipher.Cipher.open(out hd, GCrypt.Cipher.Algorithm.ARCFOUR, GCrypt.Cipher.Mode.STREAM, (GCrypt.Cipher.Flag)0);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            err = hd.set_key(key);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            err = hd.set_iv(iv);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            uchar[] plain = new uchar[ciph.length];
            err = hd.encrypt(plain, ciph);
            if (err != 0)
            {
                throw new GCryptError.FAILED(@"libgcrypt: source: $(err.source_to_string()) code: $(err)\n");
            }
            return plain;
        }

        uchar[] create_random_session_key(int l)
        {
            uchar[] ret = new uchar[l];
            GCrypt.Random.nonce(ret);
            return ret;
        }

        void hex_dump(uint8[] buf)
        {
            /* dumps buf to stdout. Looks like:
             * [0000] 75 6E 6B 6E 6F 77 6E 20
             *                  30 FF 00 00 00 00 39 00 unknown 0.....9.
             * (in a single line of course)
             */

            uchar *p = buf;
            string addrstr = "";
            string hexstr = "";
            string charstr = "";

            for (int n = 1; n <= buf.length; n++)
            {
                if (n % 16 == 1)
                {
                    addrstr = "%.4x".printf((uint)p-(uint)buf);
                }

                uchar c = *p;
                if (c == '\0' || "( ):-_qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM1234567890".index_of_char(c) < 0)
                    c = '.';

                // store hex str (for left side)
                hexstr += "%02X ".printf(*p);
                charstr += "%c".printf(c);

                if (n % 16 == 0)
                {
                    // line completed
                    stdout.printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
                    hexstr = "";
                    charstr = "";
                }
                else if (n % 16 == 8)
                {
                    // half line: add whitespaces
                    hexstr += "  ";
                    charstr += " ";
                }

                p++; // next byte
            }

            if (hexstr.length > 0)
            {
                // print rest of buffer if not empty
                stdout.printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
            }
        }

        uchar[] asym_encrypt(uchar[] message, GCrypt.SExp pkey) throws GCryptError
        {
            uchar[] session_key = create_random_session_key(32);
            if (Settings.TESTING) {stdout.printf("\nasym_encrypt: session_key follows:\n"); hex_dump((uint8[])session_key);}
            uchar[] asym_enc_session_key = (uchar[])encrypt_session_key_get_rsa_mpi(session_key, pkey);
            uchar[] sym_enc_message = encrypt(message, session_key, new uchar[]{});
            uchar[] len_0 = ("%d:".printf(asym_enc_session_key.length)).data;
            uchar[] len_1 = ("%d:".printf(sym_enc_message.length)).data;
            uchar[] ret = new uchar[len_0.length + asym_enc_session_key.length + len_1.length + sym_enc_message.length];
            int j = 0;
            for (int i = 0; i < len_0.length; i++) ret[j++] = len_0[i];
            for (int i = 0; i < asym_enc_session_key.length; i++) ret[j++] = asym_enc_session_key[i];
            for (int i = 0; i < len_1.length; i++) ret[j++] = len_1[i];
            for (int i = 0; i < sym_enc_message.length; i++) ret[j++] = sym_enc_message[i];
            return ret;
        }

        uchar[] asym_decrypt(uchar[] ciph, GCrypt.SExp skey) throws GCryptError
        {
            if (ciph.length < 6)
            {
                throw new GCryptError.FAILED("Failure: malformed encrypted message.");
            }
            int i = 0;
            int len_0 = 0;
            int j_0 = 0;
            uchar[] asym_enc_session_key = new uchar[]{};
            int len_1 = 0;
            int j_1 = 0;
            uchar[] sym_enc_message = new uchar[]{};
            int phase = 0;
            while (i < ciph.length)
            {
                uchar c = ciph[i];
                switch (phase)
                {
                    case 0:
                        if (c == ':')
                        {
                            if (len_0 == 0)
                            {
                                throw new GCryptError.FAILED("Failure: malformed encrypted message.");
                            }
                            phase = 1;
                            asym_enc_session_key = new uchar[len_0];
                            break;
                        }
                        if ("0123456789".index_of_char(c) < 0)
                        {
                            throw new GCryptError.FAILED("Failure: malformed encrypted message.");
                        }
                        len_0 *= 10;
                        len_0 += (int)(c-'0');
                        break;
                    case 1:
                        asym_enc_session_key[j_0++] = c;
                        if (j_0 == len_0) phase = 2;
                        break;
                    case 2:
                        if (c == ':')
                        {
                            if (len_1 == 0)
                            {
                                throw new GCryptError.FAILED("Failure: malformed encrypted message.");
                            }
                            phase = 3;
                            sym_enc_message = new uchar[len_1];
                            break;
                        }
                        if ("0123456789".index_of_char(c) < 0)
                        {
                            throw new GCryptError.FAILED("Failure: malformed encrypted message.");
                        }
                        len_1 *= 10;
                        len_1 += (int)(c-'0');
                        break;
                    case 3:
                        sym_enc_message[j_1++] = c;
                        if (j_1 == len_1) phase = 4;
                        break;
                    case 4:
                        throw new GCryptError.FAILED("Failure: malformed encrypted message.");
                }
                i++;
            }
            if (phase != 4)
            {
                throw new GCryptError.FAILED("Failure: malformed encrypted message.");
            }
            if (Settings.TESTING) {stdout.printf("\nasym_decrypt: asym_enc_session_key follows:\n"); hex_dump((uint8[])asym_enc_session_key);}
            if (Settings.TESTING) {stdout.printf("\nasym_decrypt: sym_enc_message follows:\n"); hex_dump((uint8[])sym_enc_message);}
            uchar[] session_key = decrypt_session_key_from_rsa_mpi((char[])asym_enc_session_key, skey);
            if (Settings.TESTING) {stdout.printf("\nasym_decrypt: session_key follows:\n"); hex_dump((uint8[])session_key);}
            return decrypt(sym_enc_message, session_key, new uchar[]{});
        }

        string short_repr(uint8 *pubkey)
        {
            string ret = "...";
            uchar *p = (uchar*)(pubkey + 80);
            for (int n = 1; n <= 16; n++)
            {
                ret += "%02X".printf(*p);
                p++; // next byte
            }
            return ret + "...";
        }
    }

    /**
      */
    public class KeyPair : Object
    {
        private GCrypt.SExp keypair;
        private GCrypt.SExp pkey;
        private GCrypt.SExp skey;
        public PublicKeyWrapper pub_key {get; private set;}
        public KeyPair(string? from_file=null) throws Crypto.GCryptError
        {
            if (from_file == null)
            {
                // generate keypair
                keypair = Crypto.generate_keypair();
            }
            else
            {
                // recreate
                uint8[]? buffer = Crypto.read_file(from_file, 2000);
                keypair = Crypto.recreate_sexp(buffer);
            }
            pkey = Crypto.getpubkey(keypair);
            skey = Crypto.getprivkey(keypair);
            pub_key = new PublicKeyWrapper.from_pem(Crypto.serialize_sexp(pkey));
        }

        public void save_pair(string keys_path) throws Crypto.GCryptError
        {
            uint8[] buffer = Crypto.serialize_sexp(keypair);
            try
            {
                Crypto.write_file(keys_path, buffer);
            }
            catch (Crypto.FileError e)
            {
                throw new Crypto.GCryptError.FAILED(@"Failed to save key pair: $(e.message)");
            }
        }

        public void save_pub_key(string pk_path) throws Crypto.GCryptError
        {
            uint8[] buffer = Crypto.serialize_sexp(pkey);
            try
            {
                Crypto.write_file(pk_path, buffer);
            }
            catch (Crypto.FileError e)
            {
                throw new Crypto.GCryptError.FAILED(@"Failed to save public key: $(e.message)");
            }
        }

        public uchar[] sign(uchar[] msg) throws Crypto.GCryptError
        {
            GCrypt.SExp signature = Crypto.sign(Crypto.md5((uint8[])msg), skey);
            uint8[] buffer = Crypto.serialize_sexp(signature);
            return (uchar[])buffer;
        }

        public uchar[] decrypt(uchar[] cypher) throws Crypto.GCryptError
        {
            return Crypto.asym_decrypt(cypher, skey);
        }

        public string to_string()
        {
            return @"<KeyPair pubkey='$(pub_key.to_pubkey())'>";
        }
    }

    public class PublicKeyWrapper : Object
    {
        private GCrypt.SExp pkey;
        private PublicKeyWrapper(uint8[]? from_pem,
                                string? from_file=null,
                                PublicKey? from_pubk=null) throws Crypto.GCryptError
        {
            uint8[] buffer;
            if (from_pem != null)
            {
                buffer = from_pem;
                pkey = Crypto.recreate_sexp(buffer);
            }
            else if (from_file != null)
            {
                buffer = Crypto.read_file(from_file, 2000);
                pkey = Crypto.recreate_sexp(buffer);
            }
            else
            {
                pkey = Crypto.recreate_sexp(from_pubk.x.buffer);
            }
        }

        public PublicKeyWrapper.from_pem(uint8[] from_pem) throws Crypto.GCryptError
        {
            this(from_pem);
        }

        public PublicKeyWrapper.from_file(string from_file) throws Crypto.GCryptError
        {
            this(null, from_file);
        }

        public PublicKeyWrapper.from_pubk(PublicKey from_pubk) throws Crypto.GCryptError
        {
            this(null, null, from_pubk);
        }

        public void save_pub_key(string pk_path) throws Crypto.GCryptError
        {
            uint8[] buffer = Crypto.serialize_sexp(pkey);
            try
            {
                Crypto.write_file(pk_path, buffer);
            }
            catch (Crypto.FileError e)
            {
                throw new Crypto.GCryptError.FAILED(@"Failed to save public key: $(e.message)");
            }
        }

        public bool verify(uchar[] msg, uchar[] signature) throws Crypto.GCryptError
        {
            GCrypt.SExp sig = Crypto.recreate_sexp((uint8[])signature);
            return Crypto.verify(sig, Crypto.md5((uint8[])msg), pkey);
        }

        public uchar[] encrypt(uchar[] message) throws Crypto.GCryptError
        {
            return Crypto.asym_encrypt(message, pkey);
        }

        public PublicKey to_pubkey()
        {
            return new PublicKey(new SerializableBuffer(Crypto.serialize_sexp(pkey)));
        }

        public string to_string()
        {
            return @"<PublicKeyWrapper pubk='$(to_pubkey())'>";
        }
    }
}

