/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011-2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Netsukuku;
using Tasklets;
using zcd;
using Ntk.Test;

namespace Netsukuku
{
#if log_tasklet
    private string tasklet_id()
    {
        return @"[$(Tasklet.self().id)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_info(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_notice(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_warn(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_error(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_critical(string msg) {print_out(tasklet_id()+msg+"\n");}

    public errordomain PeerToPeerError {
        REGISTER,
        GENERIC
    }

    public class RmtPeer : Object
    {
        private DistributedHashTable realdest;
        public RmtPeer(PeerToPeer peer_to_peer_service, Object? key=null, NIP? hIP=null, AggregatedNeighbour? aggregated_neighbour=null)
        {
            MapRoute mr = peer_to_peer_service.maproute;
            // I have to reach dest
            NIP dest;
            if (hIP == null)
                dest = peer_to_peer_service.h(key);
            else
                dest = hIP;
            // The best approximation alive is realdest
            realdest = Ntk.Test.best_approximation(mr.levels, mr.gsize, dest, 1);
        }

        public virtual ISerializable rmt(RemoteCall data) throws Error
        {
            print_out(@"RmtPeer.rmt(data $(data))\n");
            print_out(@"sent to $(realdest.maproute.me)\n");
            
            return realdest._dispatch(null, data);
        }
    }

    public class PeerToPeerAll : Object
    {
        public virtual void peer_to_peer_register(PeerToPeer peer_to_peer) throws PeerToPeerError
        {
        }
    }

    public class PeerToPeer : Object
    {
        public PeerToPeer(AggregatedNeighbourManager aggregated_neighbour_manager,
                    MapRoute maproute, int pid)
        {
            this.maproute = maproute;
        }

        public weak MapRoute maproute;

        public virtual ISerializable _dispatch (Object? caller, RemoteCall data) throws GLib.Error
        {
            return null;
        }

        public signal void map_peer_to_peer_validated();

        public virtual NIP h(Object key)
        {
            return (NIP)key;
        }

        public Gee.List<int> list_ids(int center, int sign)
        {
            print_out(@"[$(this.maproute.me)]: list_ids(center $(center), sign $(sign))\n");
            ArrayList<int> ret = Ntk.Test.list_positions(center, sign, maproute.gsize);
            string strret = "";
            foreach (int r in ret) strret += @" $(r) ";
            print_out(@"[$(this.maproute.me)]: list_ids = $(strret)\n");
            return ret;
        }

        // Simulate a network with a number of nodes and a number of records
        // in the DHT database. Simulate various cases, of a new node with a
        // certain NIP. Based on the NIP of the new node the method
        // find_hook_peers of PeerToPeer has to return the NIP of the
        // participants that are first_forward, first_back and last_back at
        // a particular level and a particular num_dupl.
        // At the same time the method rmt of RmtPeer, which
        // will eventually get called with RemoteCall <"get_cache">, has to
        // return a ListISerializable backed by a Gee.List<DHTRecord> that
        // will contain records based on the NIP passed to the constructor
        // of RmtPeer. This combination should allow to test the right behaviour
        // of the method hook_to_service of DistributedHashTable.
        //
        // 
        public void find_hook_peers(out int? ret_first_forward,
                                    out int? ret_first_back,
                                    out int? ret_last_back,
                                    int lvl, int num_dupl, int timeout=120000)
        {
            print_out(@"[$(this.maproute.me)]: find_hook_peers(lvl $(lvl), num_dupl $(num_dupl))\n");
            ret_first_forward = null;
            ret_first_back = null;
            ret_last_back = null;
            // check positions from me forward
            Gee.List<int> ids_me_forward = list_ids(maproute.me.position_at(lvl), 1);
            ids_me_forward.remove_at(0);
            foreach (int _id in ids_me_forward)
            {
                int num = 0;
                try {
                    num = get_number_of_participants(lvl, _id, timeout);
                }
                catch (RPCError e) {}
                if (num > 0)
                {
                    ret_first_forward = _id;
                    break;
                }
            }
            // Maybe no nodes at all participate. It would return 3 null.
            // Otherwise...
            if (ret_first_forward != null)
            {
                // check positions from me back
                Gee.List<int> ids_me_back = list_ids(maproute.me.position_at(lvl), -1);
                ids_me_back.remove_at(0);
                int remaining = num_dupl;
                foreach (int _id in ids_me_back)
                {
                    int num = 0;
                    try {
                        num = get_number_of_participants(lvl, _id, timeout);
                    }
                    catch (RPCError e) {}
                    if (num > 0 && ret_first_back == null) ret_first_back = _id;
                    remaining -= num;
                    if (remaining <= 0)
                    {
                        ret_last_back = _id;
                        break;
                    }
                }
                if (remaining > 0)
                {
                    // not enough nodes participate.
                    ret_first_back = null;
                    // I will reach the first forward and be a replica for any
                    // record that it has.
                }
            }
            // One possible situation is that ret_first_back == ret_last_back
            // This would mean that that gnode has a number of participants
            // greater than num_dupl.
        }

        public int get_number_of_participants(int lvl, int pos, int timeout)
        {
            print_out(@"[$(this.maproute.me)]: get_number_of_participants(lvl $(lvl), pos $(pos))\n");
            int ret = 0;
            foreach (NIP n in Ntk.Test.nodes.keys)
            {
                HCoord hcn = n.get_hcoord_relative_to(maproute.me);
                if (hcn.lvl == lvl && hcn.pos == pos) ret++;
            }
            print_out(@"[$(this.maproute.me)]:   $(ret)\n");
            return ret;
        }

        public Gee.List<NIP> find_nearest_to_register(NIP hash_nip, int num_dupl)
        {
            print_out(@"[$(this.maproute.me)]: find_nearest_to_register(hash_nip $(hash_nip), num_dupl $(num_dupl))\n");
            ArrayList<NIP> ret = new ArrayList<NIP>(PartialNIP.equal_func);
            ArrayList<NIP> lst = 
                Ntk.Test.ordered_lookup(maproute.levels, maproute.gsize, hash_nip, 1);
            foreach (NIP s in lst)
            {
                if (Ntk.Test.nodes.has_key(s)) ret.add(s);
                if (ret.size >= num_dupl) break;
            }
            foreach (NIP s in ret) print_out(@"[$(this.maproute.me)]:   $(s)\n");
            return ret;
        }

        public HCoord? search_participant(NIP hIP, int path_sign=1) throws PeerToPeerError
        {
            print_out(@"[$(this.maproute.me)]: search_participant(hIP $(hIP), path_sign $(path_sign))\n");
            // Must find the participant at this moment in the network (I will check
            //  the collection Ntk.Test.nodes) starting at hIP and going path_sign.
            // Then must get HCoord of the result based on this.maproute.
            // If it's me, then null.
            NIP? first = Ntk.Test.best_approximation_nip(maproute.levels,
                         maproute.gsize, hIP, path_sign);
            if (first == null)
                print_out(@"[$(this.maproute.me)]: search_participant: first null\n");
            else
                print_out(@"[$(this.maproute.me)]: search_participant: first $(first)\n");
            if (first == null) throw new PeerToPeerError.GENERIC("No participants.");
            if (first.is_equal(maproute.me)) return null;
            return first.get_hcoord_relative_to(maproute.me);
        }
    }

    public class OptionalPeerToPeer : PeerToPeer
    {
        //public MapPeerToPeer map_peer_to_peer;
        public bool will_participate;
        public OptionalPeerToPeer(AggregatedNeighbourManager aggregated_neighbour_manager,
                    MapRoute maproute, int pid)
        {
            base(aggregated_neighbour_manager, maproute, pid);
        }

        public bool hooked = false;
        public void participate()
        {
            hooked = true;
        }
    }

    public class AggregatedNeighbourManager : Object
    {
    }

    public class AggregatedNeighbour : Object
    {
    }

    public class MapRoute : Object
    {
        public MapRoute(int l, int s, NIP n)
        {
            levels = l;
            gsize = s;
            me = n;
        }
        public int levels;
        public int gsize;
        public NIP me;
    }
}

namespace Ntk.Test
{
    string logger;
    const bool output = false;
    public void print_out(string s)
    {
        if (output) print(s);
    }

    public HashMap<NIP, DistributedHashTable> nodes;

    public NIP? best_approximation_nip(int levels, int gsize, NIP hIP, int sign)
    {
        // The best approximation alive
        NIP? ret = null;
        ArrayList<NIP> lst = 
            Ntk.Test.ordered_lookup(levels, gsize, hIP, sign);
        foreach (NIP s in lst)
        {
            if (Ntk.Test.nodes.has_key(s)) {ret = s; break;}
        }
        return ret;
    }

    public DistributedHashTable best_approximation(int levels, int gsize, NIP hIP, int sign)
    {
        NIP nip = best_approximation_nip(levels, gsize, hIP, sign);
        assert(nip != null);
        assert(Ntk.Test.nodes.has_key(nip));
        return Ntk.Test.nodes[nip];
    }

    public ArrayList<int> list_positions(int startpos, int sign, int gsize)
    {
        // Given gsize, startpos and sign this method will return the ordered list
        //  of all the positions (in one level) that will be contacted as peers
        ArrayList<int> ret = new ArrayList<int>();
        for (int i = 0; i < gsize; i++)
            ret.add((startpos + gsize + i * sign) % gsize);
        return ret;
    }

    public ArrayList<NIP> ordered_lookup(int levels, int gsize, NIP hIP, int sign)
    {
        // Given levels, gsize, hIP and sign this method will return the ordered list
        //  of all the NIPs that will be contacted as peers in our peertopeer
        ArrayList<NIP> tot = new ArrayList<NIP>();
        for (int lvl = 0; lvl < levels; lvl++)
        {
            ArrayList<NIP> tmp = new ArrayList<NIP>();
            foreach (int n in list_positions(hIP.position_at(lvl), sign, gsize))
            {
                if (tot.is_empty) tmp.add(new NIP({n}));
                else foreach (NIP s in tot)
                {
                    int[] n_s_pos = new int[lvl+1];
                    int[] s_pos = s.get_positions();
                    for (int l = 0; l < lvl; l++) n_s_pos[l] = s_pos[l];
                    n_s_pos[lvl] = n;
                    tmp.add(new NIP(n_s_pos));
                }
            }
            tot = tmp;
        }
        return tot;
    }

    public class DhtTester : Object
    {
        public void set_up ()
        {
            logger = "";
        }

        public void tear_down ()
        {
            logger = "";
        }

        public void test_hook_and_replica()
        {
            print_out("test_hook_and_replica started\n");
            int int_set_1_counter = 0;
            int[] int_set_1 =
            {
                /* levels */ 3,
                /* gsize */ 4,
                /* positions */
                3, 0, 2,     /* node 2 in gnode 0 in ggnode 3 */
                  0,  /* how many record to store */
                  0,  /* how many keys to lookup */
                2, 0, 1, 0, 0,
                1, 1, 3, 2, 0,
                0, 3, 2, 0, 1,
                0, 2, 2, 0, 0,
                3, 1, 3, 0, 0,
                2, 1, 1, 0, 0,
                3, 0, 3, 0, 0,
                2, 0, 2, 0, 0,
                3, 1, 1, 0, 0,
                2, 2, 2, 0, 0,
                0, 2, 3, 0, 0,
                1, 1, 1, 0, 0,
                2, 2, 1, 0, 0,
                0, 0, 2, 0, 0,
                2, 3, 0, 0, 0,
                3, 3, 1, 0, 0,
                3, 2, 2, 0, 0,
                0, 2, 0, 0, 2,
                3, 3, 2, 7, 3
            };
            string[] records = 
            {
                "k1", "v1",   /* key, value */
                "kn", "vn",
                "pegasus", "192.168.14.32",
                "zeus", "192.168.14.22",
                "andromeda", "192.168.14.124",
                "athena", "192.168.14.56",
                "sirio", "192.168.14.3",
                "euliste", "192.168.14.5",
                "dedalo", "192.168.14.1"
            };
            string[] searchkeys = 
            {
                "k1", "1",       /* key, should-find */
                "kn", "1",
                "pippo", "",     /* key, should-not-find */
                "athena", "1",
                "zeus", "1",
                "dedalo", "1"
            };

            // This collection will contain the nodes in the network
            nodes = new HashMap<NIP, DistributedHashTable>(PartialNIP.hash_func, PartialNIP.equal_func);

            NIP[] nips = {};
            int levels = int_set_1[int_set_1_counter++];
            int gsize = int_set_1[int_set_1_counter++];
            while (true)
            {
                int[] positions = new int[levels];
                for (int i = levels-1; i >= 0; i--)
                    positions[i] = int_set_1[int_set_1_counter++];
                NIP nip = new NIP(positions);
                nips += nip;
                int_set_1_counter++;  // store record?
                int_set_1_counter++;  // read record?
                if (int_set_1_counter >= int_set_1.length) break;
            }

            MapRoute[] mrs = {};
            int_set_1_counter = 2;
            int records_counter = 0;
            int searchkeys_counter = 0;
            foreach (NIP nip in nips)
            {
                AggregatedNeighbourManager anm = new AggregatedNeighbourManager();
                MapRoute mr = new MapRoute(levels, gsize, nip);
                mrs += mr;
                PeerToPeerAll ptpall = new PeerToPeerAll();
                DistributedHashTable dht = new DistributedHashTable(anm, mr, ptpall);
                dht.map_peer_to_peer_validated();
                while (!dht.hooked) Tasklet.nap(0, 10000);
                nodes[mr.me] = dht;

                int_set_1_counter += levels;
                int store = int_set_1[int_set_1_counter++];
                int search = int_set_1[int_set_1_counter++];
                // do I want to store?
                while (store-- > 0)
                {
                    DHTKey k = new DHTKey(records[records_counter++]);
                    DHTRecord rec = new DHTRecord(k, records[records_counter++]);
                    print_out(@"store $(k.key) $(rec.@value)\n");
                    dht.peer(dht.h(k)).store(rec);
                    Tasklet.nap(0, 10000);  // delay for replication
                }
                // do I want to search?
                while (search-- > 0)
                {
                    DHTKey k = new DHTKey(searchkeys[searchkeys_counter++]);
                    print_out(@"search $(k.key)\n");
                    DHTRecord? rec = dht.peer(dht.h(k)).retrieve(k);
                    if (rec == null) print_out("not found\n");
                    else print_out(@"found: $(rec.@value)\n");
                    if (searchkeys[searchkeys_counter++] == "") assert(rec == null);
                    else assert(rec != null);
                }
            }

            // Any record has at least DHT_DUPLICATION presences
            HashMap<string, int> number_hits = new HashMap<string, int>();
            foreach (DistributedHashTable dht in Ntk.Test.nodes.values)
            {
                print_out(@"Node $(dht.maproute.me) has $(dht.cache.size) records:\n");
                foreach (DHTRecord rec in dht.cache.values)
                {
                    print_out(@"  $(rec.key.key) = $(rec.@value)\n");
                    if (number_hits.has_key(rec.key.key))
                        number_hits[rec.key.key] = number_hits[rec.key.key] + 1;
                    else
                        number_hits[rec.key.key] = 1;
                }
            }
            foreach (int num in number_hits.values) assert(num >= DHT_DUPLICATION);
        }

        public void test_impl_hook_to_service()
        {
            print_out("test_impl_hook_to_service started\n");

            // This collection will contain the nodes in the network
            nodes = new HashMap<NIP, DistributedHashTable>(PartialNIP.hash_func, PartialNIP.equal_func);

            // A first node appears.
            AggregatedNeighbourManager anm = new AggregatedNeighbourManager();
            MapRoute mr = new MapRoute(3, 4, new NIP({2, 0, 3}));
            PeerToPeerAll ptpall = new PeerToPeerAll();
            DistributedHashTable dht = new DistributedHashTable(anm, mr, ptpall);
            dht.map_peer_to_peer_validated();
            while (!dht.hooked) Tasklet.nap(0, 10000);
            nodes[mr.me] = dht;
            dht.store(new DHTRecord(new DHTKey("k1"), "v1"), true);

            AggregatedNeighbourManager anm2 = new AggregatedNeighbourManager();
            MapRoute mr2 = new MapRoute(3, 4, new NIP({1, 0, 2}));
            PeerToPeerAll ptpall2 = new PeerToPeerAll();
            DistributedHashTable dht2 = new DistributedHashTable(anm2, mr2, ptpall2);
            dht2.map_peer_to_peer_validated();
            while (!dht2.hooked) Tasklet.nap(0, 10000);
            nodes[mr2.me] = dht2;
            var key1 = new DHTKey("k1");
            DHTRecord? rec = dht2.retrieve(key1);
            assert(rec != null);
            assert(rec.@value == "v1");

            // another record, whose key "Ohz7thie" hashes to NIP {3,1,1}, which
            // is best approximated by {1, 0, 2}
            DHTRecord another = new DHTRecord(new DHTKey("Ohz7thie"), "secret");
            dht2.store(another, true);
            Tasklet.nap(0, 100000); // a delay to allow replication
            // We should have an available replica in dht
            var key2 = new DHTKey("Ohz7thie");
            assert(dht.cache.has_key(key2));
            assert(dht.cache[key2].@value == "secret");

            AggregatedNeighbourManager anm3 = new AggregatedNeighbourManager();
            MapRoute mr3 = new MapRoute(3, 4, new NIP({3, 1, 1}));
            PeerToPeerAll ptpall3 = new PeerToPeerAll();
            DistributedHashTable dht3 = new DistributedHashTable(anm3, mr3, ptpall3);
            dht3.map_peer_to_peer_validated();
            while (!dht3.hooked) Tasklet.nap(0, 10000);
            nodes[mr3.me] = dht3;
            // still few nodes, all records should be anywhere
            assert(dht3.cache.has_key(key1));
            assert(dht3.cache[key1].@value == "v1");
            assert(dht3.cache.has_key(key2));
            assert(dht3.cache[key2].@value == "secret");
        }

        public void test_hash()
        {
            print_out("test_hash started\n");
            // Given a DHTKey, a value for levels and a value for gsize,
            // the function h should compute the same NIP in any platform
            // and any architecture.
            AggregatedNeighbourManager anm = new AggregatedNeighbourManager();
            MapRoute mr = new MapRoute(3, 4, new NIP({2, 0, 3}));
            PeerToPeerAll ptpall = new PeerToPeerAll();
            DistributedHashTable dht = new DistributedHashTable(anm, mr, ptpall);
            NIP n1 = dht.h(new DHTKey("k1"));
            assert(n1.is_equal(new NIP({1,0,0})));
            NIP n2 = dht.h(new DHTKey("another value"));
            assert(n2.is_equal(new NIP({3,0,3})));
            NIP n3 = dht.h(new DHTKey("lorem ipsum"));
            assert(n3.is_equal(new NIP({2,0,3})));
            mr = new MapRoute(4, 256, new NIP({141, 250, 0, 3}));
            dht = new DistributedHashTable(anm, mr, ptpall);
            NIP n4 = dht.h(new DHTKey("k1"));
            assert(n4.is_equal(new NIP({65,120,89,0})));
            NIP n5 = dht.h(new DHTKey("another value"));
            assert(n5.is_equal(new NIP({51,199,181,83})));
            NIP n6 = dht.h(new DHTKey("lorem ipsum"));
            assert(n6.is_equal(new NIP({50,222,90,29})));
        }

        public void test_hash_equal_func()
        {
            print_out("test_hash_equal_func started\n");
            var cache = new HashMap<DHTKey, DHTRecord>(DHTKey.hash_func, DHTKey.equal_func);
            var recone = new DHTRecord(new DHTKey("k1"), "v1");
            cache[recone.key] = recone;
            assert(cache.has_key(new DHTKey("k1")));
            var rectwo = new DHTRecord(new DHTKey("Ohz7thie"), "secret");
            cache[rectwo.key] = rectwo;
            assert(cache.has_key(new DHTKey("Ohz7thie")));
            assert(!cache.has_key(new DHTKey("pippo")));
        }

        public static int main(string[] args)
        {
            GLib.Test.init(ref args);
            Tasklet.init();
            GLib.Test.add_func ("/DistributedHashTable/Hash", () => {
                var x = new DhtTester();
                x.set_up();
                x.test_hash();
                x.tear_down();
            });
            GLib.Test.add_func ("/DistributedHashTable/Collections", () => {
                var x = new DhtTester();
                x.set_up();
                x.test_hash_equal_func();
                x.tear_down();
            });
            GLib.Test.add_func ("/DistributedHashTable/Hooking", () => {
                var x = new DhtTester();
                x.set_up();
                x.test_impl_hook_to_service();
                x.tear_down();
            });
            GLib.Test.add_func ("/DistributedHashTable/HookingAndReplica", () => {
                var x = new DhtTester();
                x.set_up();
                x.test_hook_and_replica();
                x.tear_down();
            });
            GLib.Test.run();
            Tasklet.kill();
            return 0;
        }
    }
}
