/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Implementation of the PeerToPeer Over Ntk RFC. See {-PeerToPeerNtk-}
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    /** Helper functions for replication of records **/

    class HelperCallbacks : Object
    {
        public AcceptRecordCallback accept_callback;
        public ForwardRecordCallback forward_callback;
        public RefuseRecordCallback? refuse_callback;
    }
    delegate bool AcceptRecordCallback() throws PeerRefuseServiceError;
    delegate void RefuseRecordCallback() throws PeerRefuseServiceError;
    delegate void ForwardRecordCallback(
                        Object? obj1,
                        owned Gee.List<NIP>? replica_nodes=null) throws PeerRefuseServiceError;
    void check_hash_and_start_replica(
                        PeerToPeer service,
                        NIP hash_nip,
                        bool replicate,
                        Object? obj1,
                        int replica_nodes_max,
                        AcceptRecordCallback accept_callback,
                        ForwardRecordCallback forward_callback,
                        RefuseRecordCallback? refuse_callback=null)
                        throws PeerRefuseServiceError
    {
        HelperCallbacks callbacks = new HelperCallbacks();
        callbacks.accept_callback = accept_callback;
        callbacks.forward_callback = forward_callback;
        callbacks.refuse_callback = refuse_callback;
        // First, I check if I am the exact best participant hash_nip.
        // This check doesnt need to send messages to other nodes.
        HCoord? participant_hash_nip = service.search_participant(hash_nip);
        if (participant_hash_nip == null)
        {
            bool confirm = accept_callback();
            if (!confirm) return;
            if (!replicate) return;
            // Launch forward_callback in a new tasklet and reply right away.
            Tasklet.tasklet_callback(
                (tpar1, tpar2) => {
                    Object tasklet_obj1 = tpar1;
                    HelperCallbacks tasklet_callbacks = (HelperCallbacks)tpar2;
                    Tasklet.declare_self("peer_to_peer_start_replica");
                    tasklet_callbacks.forward_callback(tasklet_obj1);
                },
                obj1,
                callbacks);
        }
        else
        {
            // If I am not the best, it's convenient for the client that I
            // do some research from here. If I am in the replica nodes I accept
            // the record.
            Gee.List<NIP> replica_nodes =
                    service.find_nearest_to_register(hash_nip, replica_nodes_max);
            if (service.maproute.me in replica_nodes)
            {
                bool confirm = accept_callback();
                if (!confirm) return;
                if (!replicate) return;
                // Launch forward_callback in a new tasklet and reply right away.
                Tasklet.tasklet_callback(
                    (tpar1, tpar2, tpar3) => {
                        Object tasklet_obj1 = tpar1;
                        Gee.List<NIP>? tasklet_replica_nodes = (Gee.List<NIP>?)tpar2;
                        HelperCallbacks tasklet_callbacks = (HelperCallbacks)tpar3;
                        Tasklet.declare_self("peer_to_peer_start_replica");
                        tasklet_callbacks.forward_callback(tasklet_obj1, tasklet_replica_nodes);
                    },
                    obj1,
                    replica_nodes,
                    callbacks);
            }
            else
            {
                // If it is a request of a replica, drop it silently
                if (!replicate) return;
                // We should send an Exception to the client. If you have
                //  a specific one throw it here. Should be of domain PeerRefuseService.
                if (refuse_callback != null) refuse_callback();
                // If it doesnt, we throw a generic instance.
                throw new PeerRefuseServiceError.GENERIC("The node is not one of the nearest to the perfect hash.");
            }
        }
    }
}
