/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTPEERTOPEERPID {
        INT_PID,
        NUMCOLUMNS
    }

    public class NodePeerToPeer : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private InfoNode info_node;
        private ListStore liststore_services;
        private TreeView tv_services;
        private Label lbl_participation;
        private ListStore liststore_participation;
        private TreeView tv_participation;
        private SetOptionalServiceParticipants set_peertopeer_part;

        public NodePeerToPeer(Builder builder, Box box_parent, AddressManagerFakeRmtGetter client_getter)
        {
            this.client_getter = client_getter;
            builder.connect_signals (this);
            liststore_services = builder.get_object ("liststore_services") as ListStore;
            tv_services = builder.get_object ("tv_services") as TreeView;
            lbl_participation = builder.get_object ("lbl_participation") as Label;
            liststore_participation = builder.get_object ("liststore_participation") as ListStore;
            tv_participation = builder.get_object ("tv_participation") as TreeView;
            Widget widget_peertopeer = builder.get_object ("widget_root") as Widget;
            widget_peertopeer.reparent(box_parent);

            TreeSelection sel_services = tv_services.get_selection();
            sel_services.set_mode(SelectionMode.SINGLE);
            sel_services.changed.connect(() => {tv_services_selection_changed();});

            TreeSelection sel_participation = tv_participation.get_selection();
            sel_participation.set_mode(SelectionMode.NONE);

            info_node = client_getter.get_client().maproute.report_yourself();
            CellRenderer rend_lvl = new CellRendererText();
            TreeViewColumn col_lvl = 
                    new TreeViewColumn.with_attributes(
                        "Lvl/Pos",
                        rend_lvl,
                        "text",
                        0);
            tv_participation.append_column(col_lvl);
            for (int pos = 0; pos < info_node.gsize; pos++)
            {
                CellRenderer rend_pos = new CellRendererToggle();
                rend_pos.set_alignment(0,0);
                TreeViewColumn col_pos = 
                        new TreeViewColumn.with_attributes(
                            @"$(pos)",
                            rend_pos,
                            "active",
                            pos+1);
                tv_participation.append_column(col_pos);
            }
            Type[] coltypes = {};
            coltypes += typeof(int);
            for (int pos = 0; pos < info_node.gsize; pos++)
            {
                coltypes += typeof(bool);
            }
            liststore_participation = new ListStore.newv(coltypes);
            tv_participation.set_model(liststore_participation);
        }

        private int impl_start_operations()
        {
            while (true)
            {
                try { refresh_peertopeer();
                } catch (Error e) {}

                if (nap_until_condition(1000,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
                t_op = null;
            }
        }

        /** retrieve and display data
          */
        private void refresh_peertopeer() throws Error
        {
            try
            {
                set_peertopeer_part =
                        client_getter.get_client().peer_to_peer_all.get_optional_participants();
                MainContext.@default().invoke(() => {
                        display_services();
                        lbl_participation.label = "Participation";
                        return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {
                        display_error(e_message);
                        return false;});
            }
        }

        public void tv_services_selection_changed()
        {
            update_selected_services();
        }

        void update_selected_services()
        {
            TreePath? path;
            unowned TreeViewColumn? column;
            TreeIter iter;
            tv_services.get_cursor(out path, out column);
            if (path != null)
            {
                if (liststore_services.get_iter(out iter, path))
                {
                    int i;
                    liststore_services.@get(iter, LISTPEERTOPEERPID.INT_PID, out i);
                    display_participation(i);
                }
                else
                {
                    // not found
                    clear_participation();
                }
            }
            else
            {
                // not found
                clear_participation();
            }
        }

        void clear_participation()
        {
            liststore_participation.clear();
        }

        void display_participation(int pid)
        {
            foreach (OptionalServiceParticipants osp in set_peertopeer_part.o_s_participants)
            {
                if (osp.pid == pid)
                {
                    liststore_participation.clear();
                    for (int lvl = info_node.levels-1; lvl >= 0; lvl--)
                    {
                        TreeIter iter;
                        // append a new list item
                        liststore_participation.append(out iter);
                        liststore_participation.@set(iter,
                                0, lvl);
                        for (int pos = 0; pos < info_node.gsize; pos++)
                        {
                            liststore_participation.@set(iter,
                                    pos+1, osp.ppnodes.packednodes[lvl][pos].participant);
                        }
                    }
                    return;
                }
            }
            clear_participation();
        }

        void display_services()
        {
            TreeIter iter;
            foreach (OptionalServiceParticipants osp in set_peertopeer_part.o_s_participants)
            {
                int pid = osp.pid;
                // scan treemodel for it
                TreeIter? found_iter = null;
                liststore_services.@foreach((model, path, iter) => {
                    int i;
                    model.@get(iter, LISTPEERTOPEERPID.INT_PID, out i);
                    if (i == pid)
                    {
                        found_iter = iter;
                        return true;
                    }
                    return false;
                });
                if (found_iter != null)
                {
                    // modify a list item
                }
                else
                {
                    // append a new list item
                    liststore_services.append(out iter);
                    liststore_services.@set(iter,
                            LISTPEERTOPEERPID.INT_PID, pid);
                }
            }
            // remove absent items
            TreeIter[] iters_toremove = {};
            liststore_services.@foreach((model, path, iter) => {
                int iter_pid;
                model.@get(iter, LISTPEERTOPEERPID.INT_PID, out iter_pid);
                bool found = false;
                foreach (OptionalServiceParticipants osp in set_peertopeer_part.o_s_participants)
                {
                    int pid = osp.pid;
                    if (iter_pid == pid)
                    {
                        found = true;
                        break;
                    }
                }
                if (!found) iters_toremove += iter;
                return false;
            });
            foreach (TreeIter i in iters_toremove) liststore_services.remove(i);
            update_selected_services();
        }

        void display_error(string e_message)
        {
            TreeIter iter;
            liststore_services.clear();
            liststore_participation.clear();
            lbl_participation.label = e_message;
        }
    }
}
