/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Ntkresolv;
using Netsukuku.InetUtils;

const string LOG_LEAD = "";

string? get_andnaserver()
{
    string? ret = null;
    string fname = "/etc/ntkresolv/ntkresolv.ini";
    KeyFile conf = new KeyFile();
    try
    {
        bool file_found = false;
        try
        {
            conf.load_from_file(fname, KeyFileFlags.NONE);
            file_found = true;
        }
        catch (KeyFileError e)
        {
            if (e is KeyFileError.NOT_FOUND)
            {
                // No configuration file. Ignore it.
            }
            else
            {
                Posix.syslog(Posix.LOG_WARNING, LOG_LEAD + @"KeyFileError $(e.message)");
            }
        }
        if (file_found)
        {
            if (conf.has_group("NTKRESOLV") && conf.has_key("NTKRESOLV", "ANDNASERVER"))
                ret = conf.get_string("NTKRESOLV", "ANDNASERVER");
            else Posix.syslog(Posix.LOG_WARNING, LOG_LEAD + "Missing server in ini file.");
        }
        else
        {
            Posix.syslog(Posix.LOG_WARNING, LOG_LEAD + "Didn't read ini file.");
        }
    }
    catch (Error e)
    {
        Posix.syslog(Posix.LOG_WARNING, LOG_LEAD + @"Error $(e.message)");
    }
    return ret;
}

int main(string[] args)
{
    OptionContext oc = new OptionContext("- ANDNA Resolve tool");
    OptionEntry[] entries = new OptionEntry[3];
    bool reverse = false;
    bool ipv6 = false;
    int index = 0;
    entries[index++] = {"inverse", 'x', 0, OptionArg.NONE, ref reverse, "Inverse resolution", null};
    entries[index++] = {"ipv6", '6', 0, OptionArg.NONE, ref ipv6, "Use IPv6", null};
    entries[index++] = { null };
    oc.add_main_entries(entries, null);
    try {
        oc.parse(ref args);
    }
    catch (OptionError e) {
        print(@"Error parsing options: $(e.message)\n");
        return 1;
    }

    if (!reverse)
    {
        // TODO IPv6 support
        if (args.length != 2 && args.length != 3)
        {
            print(@"Usage: $(args[0]) hostname [ip_of_andna_server]\n");
            return 1;
        }
        string node = args[1];
        string? service = null;
        NtkAddrInfo? hints=null;
        string? tcpaddress = null;
        if (args.length == 3) tcpaddress = args[2];
        else tcpaddress = get_andnaserver();
        Gee.List<NtkAddrInfo> resp;
        try {
            if (tcpaddress == null)
                resp = resolv(node, service, hints);
            else
                resp = resolv(node, service, hints, tcpaddress);
            if (resp.size == 0)
            {
                print(" No such domain.\n");
            }
            else foreach (NtkAddrInfo ainfo in resp)
            {
                NtkInetAddr addr = ainfo.address as NtkInetAddr;
                if (addr == null)
                {
                    print(" * Not supported address.\n");
                }
                else
                {
                    // an IPv4 address.
                    string ipstr = address_v4_bytes_to_str(addr.addr);
                    print(@" * $(ipstr):$(addr.port)\n");
                }
            }
        } catch (Error e) {
            print(@"$(args[0]): $(e.message)\n");
        }
    }
    else
    {
        // TODO IPv6 support
        // reverse lookup
        if (args.length != 2 && args.length != 3)
        {
            print(@"Usage: $(args[0]) -x addr [ip_of_andna_server]\n");
            return 1;
        }
        IpFamily family = IpFamily.IPV4;
        uint8 addr_bytes[4];
        address_v4_str_to_bytes(args[1], addr_bytes);
        NtkAddr addr = new NtkInetAddr(addr_bytes, 0);
        string? tcpaddress = null;
        if (args.length == 3) tcpaddress = args[2];
        else tcpaddress = get_andnaserver();
        Gee.List<string> resp;
        try {
            if (tcpaddress == null)
                resp = inverse(family, addr);
            else
                resp = inverse(family, addr, tcpaddress);
            if (resp.size == 0)
            {
                print(" No hostnames.\n");
            }
            else foreach (string hname in resp)
            {
                print(@" * $(hname)\n");
            }
        } catch (Error e) {
            print(@"$(args[0]): $(e.message)\n");
        }
    }

    return 0;
}

