/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;


namespace Tasklets
{
    /** Class for "timeouts" or "timespans"
      */
    public class Timer : Object
    {
        protected TimeVal exp;
        public Timer(int64 msec_ttl)
        {
            set_time(msec_ttl);
        }

        protected void set_time(int64 msec_ttl)
        {
            exp = TimeVal();
            exp.get_current_time();
            long milli = (long)(msec_ttl % (int64)1000);
            long seconds = (long)(msec_ttl / (int64)1000);
            int64 check_seconds = (int64)exp.tv_sec;
            check_seconds += (int64)seconds;
            assert(check_seconds <= long.MAX);
            exp.add(milli*1000);
            exp.tv_sec += seconds;
        }

        public int64 get_msec_ttl()
        {
            // It's dangerous to public as API get_msec_ttl
            //  because if it is used in order to compare 2 timers
            //  the caller program cannot take into consideration the
            //  time passed from the 2 calls to this method.
            // The right way to compare 2 timers is the method is_younger.
            TimeVal now = TimeVal();
            now.get_current_time();
            long sec = exp.tv_sec - now.tv_sec;
            long usec = exp.tv_usec - now.tv_usec;
            while (usec < 0)
            {
                usec += 1000000;
                sec--;
            }
            return (int64)sec * (int64)1000 + (int64)usec / (int64)1000;
        }

        public bool is_younger(Timer t)
        {
            if (exp.tv_sec > t.exp.tv_sec) return true;
            if (exp.tv_sec < t.exp.tv_sec) return false;
            if (exp.tv_usec > t.exp.tv_usec) return true;
            return false;
        }

        public bool is_expired()
        {
            return get_msec_ttl() < 0;
        }

        public string get_string_msec_ttl()
        {
            return @"$(get_msec_ttl())";
        }
    }

    public class Stat : Object
    {
        public int id;
        public int parent;
        public string funcname = "";
        public Status status;
        public string crash_message = "";

        public static bool equal_func(Stat a, Stat b)
        {
            return a.id == b.id;
        }
    }

    public enum Status {
        SPAWNED,
        STARTED,
        ENDED,
        CRASHED,
        ABORTED
    }
}

namespace zcd
{
    /** Packs a message of variable size together with the indication of its size
      */
    uchar[] data_pack(uchar[] data)
    {
        uint data_sz = data.length;
        Variant data_hdr = new Variant.uint32(data_sz);
        uint data_hdr_sz = (uint)data_hdr.get_size();
        uchar []ser = null;
        ser = new uchar[data_hdr_sz + data_sz];
        for (int i = 0; i < data_sz; i++) ser[data_hdr_sz + i] = data[i];
            if (!endianness_network) data_hdr = data_hdr.byteswap();
        data_hdr.store(ser);
        return ser;
    }

    /** Reads from a stream a message of variable size (use with tcp streams)
      */
    uchar[] data_unpack_from_stream(IConnectedStreamSocket socket)
    {
        try
        {
            int i;
            uchar[] readBuffer = new uchar[0];
            Variant data_hdr_tmp = new Variant.uint32(0);
            uint data_hdr_sz = (uint)data_hdr_tmp.get_size();
            while (true)
            {
                uchar[] rawPacket = socket.recv((int)(data_hdr_sz - readBuffer.length));
                uchar[] tempBuffer = new uchar[readBuffer.length + rawPacket.length];
                for (i = 0; i < readBuffer.length; i++) tempBuffer[i] = readBuffer[i];
                for (i = 0; i < rawPacket.length; i++) tempBuffer[readBuffer.length + i] = rawPacket[i];
                readBuffer = tempBuffer;
                if (readBuffer.length == data_hdr_sz)
                {
                    Variant data_hdr = Variant.new_from_data<uchar[]>(VariantType.UINT32, readBuffer, false);
                    if (!endianness_network) data_hdr = data_hdr.byteswap();
                    uint data_sz = (uint)data_hdr;
                    // TODO Do I have enough memory?
                    readBuffer = new uchar[0];
                    while (readBuffer.length != data_sz)
                    {
                        rawPacket = socket.recv((int)(data_sz - readBuffer.length));
                        tempBuffer = new uchar[readBuffer.length + rawPacket.length];
                        for (i = 0; i < readBuffer.length; i++) tempBuffer[i] = readBuffer[i];
                        for (i = 0; i < rawPacket.length; i++) tempBuffer[readBuffer.length + i] = rawPacket[i];
                        readBuffer = tempBuffer;
                    }
                    return readBuffer;
                }
            }
        }
        catch (Error e)
        {
            return new uchar[0];
        }
    }

    /** Informations about an host that sent us a message.
      * This is useless at the moment in zcd-tcpclient because is used only
      * on server side, but it is present as an argument (nullable) in some
      * methods' signature.
      */
    public class CallerInfo : Object
    {
        public string? caller_ip;
        public string? my_ip;
        public string? dev;
        public CallerInfo(string? caller_ip, string? my_ip, string? dev)
        {
            this.caller_ip = caller_ip;
            this.my_ip = my_ip;
            this.dev = dev;
        }
    }

    /** An instance of this class is used when we want to send a message via TCP.
      */
    public class TCPClient : Object, FakeRmt
    {
        public string dest_addr {get; private set;}
        public uint16 dest_port;
        private string my_addr;
        private bool wait_response;
        private bool connected;
        public bool calling {get; private set;}
        public bool retry_connect {get; set; default=true;}
        private IConnectedStreamSocket? socket;
        public TCPClient(string dest_addr, uint16? dest_port=null, string? my_addr=null, bool wait_response=true)
        {
            if (dest_port == null) dest_port = (uint16)269;
            this.dest_addr = dest_addr;
            this.dest_port = dest_port;
            this.my_addr = my_addr;
            this.wait_response = wait_response;
            socket = null;
            connected = false;
            calling = false;
        }

        public ISerializable rmt(RemoteCall data) throws RPCError
        {
            // TODO locking for thread safety
            if (calling)
            {
                // A TCPClient instance cannot handle more than one rpc call at a time.
                // When a new RPC is needed and another one is 'calling' with this
                // instance of TCPClient, then another instance should be created.
                // Anyway, this control will tolerate it and emit a WARNING.
                while (calling) Thread.usleep(1000);
            }
            try
            {
                // Now other threads cannot make a RPC call
                // until this one has finished
                calling = true;

                TCPRequest message = new TCPRequest(wait_response, data);
                rpc_send(message.serialize());
                if (wait_response)
                    return rpc_receive();
                return new SerializableNone();
            }
            finally
            {
                calling = false;
            }
        }

        public void rpc_send(uchar[] serdata) throws RPCError
        {
            // 30 seconds max to try connecting
            Timer timeout = new Timer();
            int interval = 5;
            while (!connected)
            {
                connect();
                if (!connected)
                {
                    if (!retry_connect || timeout.elapsed() > 30)
                    {
                        throw new RPCError.NETWORK_ERROR(
                                @"Failed connecting to (\"$(dest_addr)\", $(dest_port))");
                    }
                    // wait <n> ms.
                    Thread.usleep(interval * 1000);
                    interval *= 2;
                    if (interval > 10000) interval = 10000;
                }
            }
            try
            {
                socket.send(data_pack(serdata));
            }
            catch (Error e)
            {
                connected = false;
                throw new RPCError.NETWORK_ERROR(e.message);
            }
        }

        public ISerializable rpc_receive() throws RPCError
        {
            if (connected)
            {
                uchar[] recv_encoded_data = data_unpack_from_stream(socket);

                if (recv_encoded_data.length == 0)
                {
                    connected = false;
                    throw new RPCError.NETWORK_ERROR(
                                "Connection closed before reply");
                }

                ISerializable ret;
                try {
                    ret = ISerializable.deserialize(recv_encoded_data);
                }
                catch (SerializerError e) {
                    throw new RPCError.SERIALIZER_ERROR(
                                "Error deserializing response");
                }
                return ret;
            }
            else throw new RPCError.NETWORK_ERROR(
                                "Connection closed before reply");
        }

        public new void connect()
        {
            try
            {
                ClientStreamSocket x = new ClientStreamSocket(my_addr);
                socket = x.socket_connect(dest_addr, dest_port);
                connected = true;
            }
            catch (Error e)
            {
                zcd.log_warn(@"TCPClient: socket connect error: $(e.message)");
            }
        }

        public void close() throws RPCError
        {
            if (connected)
            {
                if (socket != null)
                {
                    try
                    {
                        socket.close();
                    }
                    catch (Error e)
                    {
                        throw new RPCError.NETWORK_ERROR(e.message);
                    }
                    socket = null;
                }
                connected = false;
            }
        }

        ~TCPClient()
        {
            if (connected) close();
        }
    }
}

