/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004, 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <algorithm>
#include "common.h"
#include "rectangle.h"
#include "vrhomboid.h"


namespace {

void error( const char * s ) throw() __attribute__ ((noreturn));
void error( const char * s ) throw()
  { Ocrad::internal_error( s ); }

} // end namespace


Vrhomboid::Vrhomboid( int l, int lc, int r, int rc, int h ) throw()
  {
  if( r < l || h <= 0 )
    {
    std::fprintf( stderr, "l = %d, lc = %d, r = %d, rc = %d, h = %d\n",
                  l, lc, r, rc, h );
    error( "bad parameter building a Vrhomboid" );
    }
  _left = l; _lvcenter = lc; _right = r; _rvcenter = rc; _height = h;
  }


void Vrhomboid::left( int l ) throw()
  {
  if( l > _right ) error( "left, bad parameter resizing a Vrhomboid" );
  _left = l;
  }


void Vrhomboid::right( int r ) throw()
  {
  if( r < _left ) error( "right, bad parameter resizing a Vrhomboid" );
  _right = r;
  }


void Vrhomboid::height( int h ) throw()
  {
  if( h <= 0 ) error( "height, bad parameter resizing a Vrhomboid" );
  _height = h;
  }


void Vrhomboid::extend_left( int l ) throw()
  {
  if( l > _right )
    error( "extend_left, bad parameter resizing a Vrhomboid" );
  _lvcenter = vcenter( l ); _left = l;
  }


void Vrhomboid::extend_right( int r ) throw()
  {
  if( r < _left )
    error( "extend_right, bad parameter resizing a Vrhomboid" );
  _rvcenter = vcenter( r ); _right = r;
  }


int Vrhomboid::vcenter( int col ) const throw()
  {
  int dx = _right - _left, dy = _rvcenter - _lvcenter;
  int vc = _lvcenter;
  if( dx && dy ) vc += ( dy * ( col - _left ) ) / dx;
  return vc;
  }


bool Vrhomboid::includes( const Rectangle & r ) const throw()
  {
  if( r.left() < _left || r.right() > _right ) return false;
  int tl = top( r.left() ), bl = tl +_height - 1;
  int tr = top( r.right() ), br = tr +_height - 1;
  int t = std::max( tl, tr ), b = std::min( bl, br );
  return ( t <= r.top() && b >= r.bottom() );
  }


bool Vrhomboid::includes( int row, int col ) const throw()
  {
  if( col < _left || col > _right ) return false;
  int t = top( col ), b = t +_height - 1;
  return ( t <= row && b >= row );
  }
